#!/bin/bash

# This script will install and configure the Mayo ChIP-Seq analysis tool within the current working directory or a specified install directory.
# See the usage text for additional details.

##### Unix Command dependencies
# 1. wget
# 2. tar
# 3. unzip 

usage() {
	cat <<EOF
	
Usage: install.sh [--dir <directory>]
Install chip-seq in specified directory (default is current working directory).

Mandatory arguments to long options are mandatory for short options too.

  --dir                          Desired installation directory
  -h, --help                     Display this usage text
  -v, --verbose                  Display all command processing
  --install-bedtools             Install BEDTools 
  --install-bwa                  Install BWA 
  --install-ceas                 Install CEAS 
  --install-cython               Install cython 
  --install-fastqc               Install FastQC 
  --install-igvtools             Install IGVTools 
  --install-java                 Install Java  
  --install-macs                 Install MACS2 
  --install-meme                 Install MEME 
  --install-picard               Install Picard 
  --install-python               Install python 
  --install-r                    Install R 
  --install-references           Install reference files
  --install-samtools             Install SAMTools 
  --install-sicer                Install SICER 
  --install-wigtobigwig          Install wigToBigWig 
  --skip-bedtools                Skip BEDTools installation
  --skip-bwa                     Skip BWA installation
  --skip-ceas                    Skip CEAS installation
  --skip-cython                  Skip cython installation
  --skip-fastqc                  Skip FastQC installation
  --skip-igvtools                Skip IGVTools installation
  --skip-java                    Skip Java installation 
  --skip-macs                    Skip MACS2 installation
  --skip-meme                    Skip MEME installation
  --skip-picard                  Skip Picard installation
  --skip-python                  Skip python installation
  --skip-r                       Skip R installation
  --skip-references              Do not download/install reference files
  --skip-samtools                Skip SAMTools installation
  --skip-sicer                   Skip SICER installation
  --skip-wigtobigwig             Skip wigToBigWig installation

Report problems to moore.raymond@mayo.edu or kalmbach.michael@mayo.edu
EOF
}

INSTALL_DIR=`pwd`
VERBOSE=false

# Collect warnings to display to the user at the end. 
WARNINGS=""

while [ ! -z $1 ] ; do
	case "$1" in
		--install* ) 
		    if [[ -z $SKIP_BEDTOOLS ]]
			then 
				SKIP_BEDTOOLS=true
			fi 
			
			if [[ -z $SKIP_BWA ]]
			then 
				SKIP_BWA=true 
			fi
			
			if [[ -z $SKIP_CEAS ]]
			then 
				SKIP_CEAS=true 
			fi
			
			if [[ -z $SKIP_CYTHON ]]
			then 
				SKIP_CYTHON=true 
			fi
			
			
			if [[ -z $SKIP_FASTQC ]]
			then 
				SKIP_FASTQC=true 
			fi
			
			if [[ -z $SKIP_IGVTOOLS ]]
			then 
				SKIP_IGVTOOLS=true 
			fi
			
			if [[ -z $SKIP_JAVA ]]
			then 
				SKIP_JAVA=true 
			fi
			
			if [[ -z $SKIP_MACS ]]
			then 
				SKIP_MACS=true 
			fi
			
			if [[ -z $SKIP_MEME ]]
			then 
				SKIP_MEME=true 
			fi
			
			if [[ -z $SKIP_PICARD ]]
			then 
				SKIP_PICARD=true 
			fi
			
			if [[ -z $SKIP_PYTHON ]]
			then 
				SKIP_PYTHON=true 
			fi
			
			if [[ -z $SKIP_R ]]
			then 
				SKIP_R=true 
			fi
			
			if [[ -z $SKIP_REFERENCES ]]
			then 
				SKIP_REFERENCES=true 
			fi
			
			if [[ -z $SKIP_SAMTOOLS ]]
			then 
				SKIP_SAMTOOLS=true 
			fi
			
			if [[ -z $SKIP_SICER ]]
			then 
				SKIP_SICER=true 
			fi
			
			if [[ -z $SKIP_WIGTOBIGWIG ]]
			then 
				SKIP_WIGTOBIGWIG=true 
			fi
	esac
	
	case "$1" in
		--dir ) INSTALL_DIR=$2; shift 2;;
		-h | --help ) usage; exit ;;
		--use-python ) PYTHON=$2; shift 2;;
		-v | --verbose ) VERBOSE=false; shift ;;
		--install-bedtools ) SKIP_BEDTOOLS=false; shift ;;
		--install-bwa ) SKIP_BWA=false; shift ;;
        --install-ceas ) SKIP_CEAS=false; shift;;
		--install-cython ) SKIP_CYTHON=false; shift;;
		--install-fastqc ) SKIP_FASTQC=false; shift;;
		--install-igvtools ) SKIP_IGVTOOLS=false; shift;;
		--install-java ) SKIP_JAVA=false; shift;; 
		--install-macs ) SKIP_MACS=false; shift;;
		--install-meme ) SKIP_MEME=false; shift;;
		--install-picard ) SKIP_PICARD=false; shift;;
		--install-python ) SKIP_PYTHON=false; shift;;
		--install-r ) SKIP_R=false; shift;;
		--install-references ) SKIP_REFERENCES=false; shift;;
		--install-samtools ) SKIP_SAMTOOLS=false; shift;;
		--install-sicer ) SKIP_SICER=false; shift;;
  		--install-wigtobigwig ) SKIP_WIGTOBIGWIG=false; shift;;
		--skip-bedtools ) SKIP_BEDTOOLS=true; shift ;;
		--skip-bwa ) SKIP_BWA=true; shift ;;
        --skip-ceas ) SKIP_CEAS=true; shift;;
		--skip-cython ) SKIP_CYTHON=true; shift;;
		--skip-fastqc ) SKIP_FASTQC=true; shift;;
		--skip-igvtools ) SKIP_IGVTOOLS=true; shift;;
		--skip-java ) SKIP_JAVA=true; shift;; 
		--skip-macs ) SKIP_MACS=true; shift;;
		--skip-meme ) SKIP_MEME=true; shift;;
		--skip-picard ) SKIP_PICARD=true; shift;;
		--skip-python ) SKIP_PYTHON=true; shift;;
		--skip-r ) SKIP_R=true; shift;;
		--skip-references ) SKIP_REFERENCES=true; shift;;
		--skip-samtools ) SKIP_SAMTOOLS=true; shift;;
		--skip-sicer ) SKIP_SICER=true; shift;;
  		--skip-wigtobigwig ) SKIP_WIGTOBIGWIG=true; shift;;
		* ) echo "Unrecognized option: $1"; usage; exit; ;;
	esac
done

# If any of the install options are null, initialize them now.

if [[ -z $PYTHON ]] 
then
	PYTHON=python
fi

if [[ -z $SKIP_BEDTOOLS ]]
then 
	SKIP_BEDTOOLS=false
fi 
	
if [[ -z $SKIP_BWA ]]
then 
	SKIP_BWA=false 
fi
	
if [[ -z $SKIP_CEAS ]]
then
	SKIP_CEAS=false 
fi
	
if [[ -z $SKIP_CYTHON ]]
then 
	SKIP_CYTHON=false 
fi
	
if [[ -z $SKIP_FASTQC ]]
then 
	SKIP_FASTQC=false 
fi
	
if [[ -z $SKIP_IGVTOOLS ]]
then 
	SKIP_IGVTOOLS=false 
fi
	
if [[ -z $SKIP_JAVA ]]
then 
	SKIP_JAVA=false 
fi
	
if [[ -z $SKIP_MACS ]]
then 
	SKIP_MACS=false 
fi
	
if [[ -z $SKIP_MEME ]]
then 
	SKIP_MEME=false 
fi
	
if [[ -z $SKIP_PICARD ]]
then 
	SKIP_PICARD=false 
fi
	
if [[ -z $SKIP_PYTHON ]]
then 
	SKIP_PYTHON=false 
fi
	
if [[ -z $SKIP_R ]]
then 
	SKIP_R=false 
fi
	
if [[ -z $SKIP_REFERENCES ]]
then 
	SKIP_REFERENCES=false 
fi
	
if [[ -z $SKIP_SAMTOOLS ]]
then 
	SKIP_SAMTOOLS=false 
fi
	
if [[ -z $SKIP_SICER ]]
then 
	SKIP_SICER=false 
fi
	
if [[ -z $SKIP_WIGTOBIGWIG ]]
then 
	SKIP_WIGTOBIGWIG=false 
fi

# Direct URL downloads: listed here, in case of updates.

JAVA_URL='http://download.oracle.com/otn-pub/java/jdk/7u45-b18/jdk-7u45-linux-x64.tar.gz'
SAMTOOLS_URL='http://downloads.sourceforge.net/project/samtools/samtools/0.1.19/samtools-0.1.19.tar.bz2'
MEME_URL='http://ebi.edu.au/ftp/software/MEME/4.8.1/meme_4.8.1.tar.gz'
CEAS_URL='http://liulab.dfci.harvard.edu/CEAS/src/CEAS-Package-1.0.2.tar.gz'
PICARD_URL='http://downloads.sourceforge.net/project/picard/picard-tools/1.97/picard-tools-1.97.zip' 
BEDTOOLS_URL='https://bedtools.googlecode.com/files/BEDTools.v2.17.0.tar.gz'
PYTHON_URL='http://www.python.org/ftp/python/2.7.3/Python-2.7.3.tgz'
CYTHON_URL='http://cython.org/release/Cython-0.15.1.tar.gz'
R_URL='http://streaming.stat.iastate.edu/CRAN/bin/linux/ubuntu/precise/r-base_2.14.0.orig.tar.gz'
FASTQC_URL='http://www.bioinformatics.babraham.ac.uk/projects/fastqc/fastqc_v0.10.1.zip'
BWA_URL='http://downloads.sourceforge.net/project/bio-bwa/bwa-0.5.9rc1.tar.bz2'
WIGTOBIGWIG_URL='http://hgdownload.cse.ucsc.edu/admin/exe/linux.x86_64/wigToBigWig'
IGVTOOLS_URL='http://www.broadinstitute.org/igv/projects/downloads/igvtools_2.3.23.zip'
MACS_URL='http://github.com/taoliu/MACS/tarball/master'
SICER_URL='http://home.gwu.edu/~wpeng/SICER_V1.1.tgz'
WORKFLOW_IMG_URL='http://bioinformaticstools.mayo.edu/tools/chipseq/chipseq-pic.jpg'
WORKFLOW_DOC_URL='http://bioinformaticstools.mayo.edu/tools/chipseq/HiCHIP_workflow_summary.doc'
COLOR_LIST_URL='http://bioinformaticstools.mayo.edu/tools/chipseq/track_color_list.txt'

# Set up directory structure
### this script can be executed from anywhere...this brings you back to the same spot at the end.
CURRENT_WORKING_DIRECTORY=`pwd`
TOOL_INSTALL_DIR="$INSTALL_DIR/tools"
CONFIG_DIR="$INSTALL_DIR/config"
DOCS_DIR="$INSTALL_DIR/docs"
mkdir -p $INSTALL_DIR
mkdir -p $TOOL_INSTALL_DIR
mkdir -p $CONFIG_DIR




# If we are not installing in the current working directory, copy all of the ChIP-Seq files to the destination directory.
if [[ $INSTALL_DIR != $CURRENT_WORKING_DIRECTORY ]]
then
	cp -r $CURRENT_WORKING_DIRECTORY/* $INSTALL_DIR
fi

####### Check for required packages/applications that cannot be automatically installed ####


IS_PYTHON27_DEV_INSTALLED=`dpkg --get-selections | grep -v deinstall | grep -E "^python2.7-dev" | perl -ne 'if(/python2.7-dev/) { print "true"; } else { print "false"; }'`

# Check for python-dev package and recommend update
if [[ $IS_PYTHON27_DEV_INSTALLED = 'true' ]]
then
	echo "Package python2.7-dev is installed."
else
	WARNINGS="$WARNINGS\nWARNING: python2.7-dev does not seem to be installed. Please install this manually before running ChIP-Seq."
	echo -e "\tWARNING: python2.7-dev does not seem to be installed. Please install this manually before running ChIP-Seq."
fi

IS_PYTHON_DEV_INSTALLED=`dpkg --get-selections | grep -v deinstall | grep -E "^python-dev" | perl -ne 'if(/python-dev/) { print "true"; } else { print "false"; }'`

# Check for python-dev package and update
if [[ $IS_PYTHON_DEV_INSTALLED = 'true' ]]
then
	echo "Package python-dev is installed."
else
	WARNINGS="$WARNINGS\nWARNING: python-dev does not seem to be installed. Please install this manually before running ChIP-Seq."
	echo -e "\tWARNING: python-dev does not seem to be installed. Please install this manually before running ChIP-Seq."
fi

# Check for NumPy package
python -c "import numpy"

# Install numpy
if [[ $? = '0' ]]
then
	echo "NumPy Appears to be installed."
else
	WARNINGS="$WARNINGS\nWARNING: numpy does not seem to be installed. Please install this manually before running ChIP-Seq."
	echo -e "\tWARNING: numpy does not seem to be installed. Please install this manually before running ChIP-Seq."
fi

# Check Perl version & recommend update
echo "Checking installed Perl version."

PERL_VERSION=`perl -v | perl -ne 'print "$1" if /\(v([0-9]+\.[0-9]+)\.[0-9]+\)/'`
IS_SUPPORTED_VERSION=`echo $PERL_VERSION | perl -ne 'if (/5\.10/) { print "true"; } else { print "false"; }'`
PERL_INSTALLATION_PATH=`which perl`

if [[ $IS_SUPPORTED_VERSION = 'true' ]]
then
	echo -e "\tA suitable Perl version already appears to be installed at $PERL_INSTALLATION_PATH."
elif [[ -z $PERL_INSTALLATION_PATH ]]
then
	WARNINGS="$WARNINGS\nWARNING: Perl does not appear to be installed and available in your PATH. Please correct this before running ChIP-Seq."
	echo -e "\tWARNING: Perl does not appear to be installed and available in your PATH. Please correct this before running ChIP-Seq."
else
	WARNINGS="$WARNINGS\nWARNING: ChIP-Seq was tested using Perl 5.10. Your version is $PERL_VERSION. You may encounter failures with other Perl versions."
	echo -e "\tWARNING: ChIP-Seq was tested using Perl 5.10. Your version is $PERL_VERSION. You may encounter failures with other Perl versions."
fi

############################### End check for required packages ############################


# Check Java version. If it's a supported version, skip the installation. Note: If Java is not installed, the IS_SUPPORTED_VERSION should be false.
IS_SUPPORTED_VERSION=`java -version 2>&1 | perl -ne 'print "$1" if /java version \"(.+)_.+\"/' | perl -ne 'if (/1\.7\.0/) { print "true"; } else { print "false"; }' `
JAVA_INSTALLATION_PATH=`which java`

if [[ $SKIP_JAVA = true || $IS_SUPPORTED_VERSION = 'true' ]]
then
	echo "Skipping Java installation"

	if [[ $IS_SUPPORTED_VERSION = 'true' ]]
	then
		echo -e "\tA suitable Java version already appears to be installed at $JAVA_INSTALLATION_PATH."
	else
		WARNINGS="$WARNINGS\nWARNING: ChIP-Seq requires Java 1.7.0. You may encounter failures with other java versions."
		echo "WARNING: ChIP-Seq requires Java 1.7.0. You may encounter failures with other java versions."
	fi 
	
else
	# A compatible Java version is not currently installed
	JAVA_INSTALLATION_PATH="$TOOL_INSTALL_DIR/java/"
	echo "Installing Java into $JAVA_INSTALLATION_PATH"
	
	# The --no-cookies and --header options are required to use wget to download Java from Oracle.
	wget -q --no-cookies --header "Cookie: gpw_e24=http%3A%2F%2Fwww.oracle.com" --directory-prefix=$TOOL_INSTALL_DIR/java/ $JAVA_URL
	
	if [[ $VERBOSE = true ]]; then 
		tar -C $JAVA_INSTALLATION_PATH -zxvf $TOOL_INSTALL_DIR/java/*.tar.gz;
	else 
		tar -C $JAVA_INSTALLATION_PATH -zxf $TOOL_INSTALL_DIR/java/*.tar.gz; 
	fi
	
	# Update Java installation path to point to the java binary
	JAVA_INSTALLATION_PATH=$JAVA_INSTALLATION_PATH/$(find $JAVA_INSTALLATION_PATH -maxdepth 1 -type d -name 'jdk1.7*' | perl -p -e 's/.*\/([^\/]*+)$/$1/')/bin/java
	
	# Check installation
	IS_SUPPORTED_VERSION=`$JAVA_INSTALLATION_PATH -version 2>&1 | perl -ne 'print "$1" if /java version \"(.+)_.+\"/' | perl -ne 'if (/1\.7\.0/) { print "true"; } else { print "false"; }' `
	
	if [[ -z $IS_SUPPORTED_VERSION || $IS_SUPPORTED_VERSION = 'false' ]]
	then
		WARNINGS="$WARNINGS\nWARNING: Java 1.7.0 install appears to have failed. Please install Java 1.7.0 manually and update your tool_info.txt with its location."
	fi 
fi

PYTHON_VERSION=`python -V 2>&1 | perl -ne 'print "$1" if /^.*([0-9]+\.[0-9]+\.[0-9]+)/'`
IS_SUPPORTED_VERSION=`echo $PYTHON_VERSION | perl -ne 'if (/2\.7\.3/) { print "true"; } else { print "false"; }'`
PYTHON_INSTALLATION_PATH=`which python`

if [[ $SKIP_PYTHON = true || $IS_SUPPORTED_VERSION = 'true' ]]
then
	echo "Skipping Python installation"

	if [[ $IS_SUPPORTED_VERSION = 'true' ]]
	then
		echo -e "\tA suitable Python version already appears to be installed at $PYTHON_INSTALLATION_PATH."	
	elif [[ -z $PYTHON_INSTALLATION_PATH ]]
	then
		WARNINGS="$WARNINGS\nWARNING: Python does not appear to be installed and available in your PATH. Please correct this before running ChIP-Seq."
		echo -e "\tWARNING: Python does not appear to be installed and available in your PATH. Please correct this before running ChIP-Seq."
	else
		WARNINGS="$WARNINGS\nWARNING: ChIP-Seq was tested using Python 2.7.3. Your version is $PYTHON_VERSION. You may encounter failures with other Python versions."
		echo -e "\tWARNING: ChIP-Seq was tested using Python 2.7.3. Your version is $PYTHON_VERSION. You may encounter failures with other Python versions."	
	fi
else
	PYTHON_INSTALLATION_PATH=$TOOL_INSTALL_DIR/python/
	echo "Installing Python into $PYTHON_INSTALLATION_PATH"
	
	wget -q --directory-prefix=$PYTHON_INSTALLATION_PATH $PYTHON_URL
	
	if [[ $VERBOSE = true ]]; then 
		tar -C $PYTHON_INSTALLATION_PATH -zxvf $PYTHON_INSTALLATION_PATH/*.tgz;
	else 
		tar -C $PYTHON_INSTALLATION_PATH -zxf $PYTHON_INSTALLATION_PATH/*.tgz; 
	fi
	
	# Update Python installation path to include version number
    PYTHON_INSTALLATION_PATH=$PYTHON_INSTALLATION_PATH/Python-2.7.3
	
	cd $PYTHON_INSTALLATION_PATH
	./configure --with-suffix=no -q
	make
fi

# Install cython
if [[ $SKIP_CYTHON = true ]]
then
	echo "Skipping cython installation"
else
	CYTHON_INSTALLATION_PATH=$TOOL_INSTALL_DIR/cython/
	echo "Installing cython into $CYTHON_INSTALLATION_PATH"
	
	wget -q --directory-prefix=$CYTHON_INSTALLATION_PATH $CYTHON_URL
	
	if [[ $VERBOSE = true ]]; then 
		tar -C $CYTHON_INSTALLATION_PATH -zxvf $CYTHON_INSTALLATION_PATH/*.tar.gz;
	else 
		tar -C $CYTHON_INSTALLATION_PATH -zxf $CYTHON_INSTALLATION_PATH/*.tar.gz; 
	fi
	
	# Update Cython path to be the directory created by the tar extraction command
	CYTHON_INSTALLATION_PATH=`find $CYTHON_INSTALLATION_PATH -type d -mindepth 1 -maxdepth 1`
	
	cd $CYTHON_INSTALLATION_PATH
	
	python setup.py install -n --prefix $CYTHON_INSTALLATION_PATH --install-purelib=lib --install-platlib=lib.platform --install-scripts=scripts --install-data=data
fi

# Install R version 2.14.0
if [[ $SKIP_R = true ]]
then
	echo "Skipping R installation"
else
	R_INSTALLATION_PATH=$TOOL_INSTALL_DIR/R/
	echo "Installing R into $R_INSTALLATION_PATH"
	
	wget -q --directory-prefix=$R_INSTALLATION_PATH $R_URL
	
	if [[ $VERBOSE = true ]]; then 
		VERBOSE_FLAG="-v"
	else
		QUIET_FLAG="-q"
	fi
	
	tar -C $R_INSTALLATION_PATH $VERBOSE_FLAG -zxf $R_INSTALLATION_PATH/*.tar.gz;
	
	# Update R path to be the directory created by the tar extraction command
	R_INSTALLATION_PATH=`find $R_INSTALLATION_PATH -type d -mindepth 1 -maxdepth 1`
	
	cd $R_INSTALLATION_PATH
	
    ./configure --with-x=no $QUIET_FLAG

    if [[ $? != '0' ]]
    then
    	WARNINGS="$WARNINGS\nWARNING: R was not successfully configured. Please correct this before running ChIP-Seq."
		echo -e "\tWARNING: R was not successfully configured. Please correct this before running ChIP-Seq."
    else
    	make
    	if [[ $? != '0' ]]
    	then
    		WARNINGS="$WARNINGS\nWARNING: R was not successfully built. Please correct this before running ChIP-Seq."
			echo -e "\tWARNING: R was not successfully built. Please correct this before running ChIP-Seq."
		else
			make check
			if [[ $? != '0' ]]
    		then
    			WARNINGS="$WARNINGS\nWARNING: R did not pass its built-in checks. Please correct this before running ChIP-Seq."
				echo -e "\tWARNING: R did not pass its built-in checks.. Please correct this before running ChIP-Seq."
			fi
    	fi
    fi
    
fi

# Install FastQC version 0.10
if [[ $SKIP_FASTQC = true ]]
then
	echo "Skipping FastQC installation"
else
	FASTQC_INSTALLATION_PATH=$TOOL_INSTALL_DIR/fastqc/
	echo "Installing FastQC into $FASTQC_INSTALLATION_PATH"
	
	wget -q --directory-prefix=$FASTQC_INSTALLATION_PATH $FASTQC_URL
	
	if [[ $VERBOSE = true ]]
	then 
		unzip $FASTQC_INSTALLATION_PATH/*.zip -d $FASTQC_INSTALLATION_PATH
	else
		unzip -q $FASTQC_INSTALLATION_PATH/*.zip -d $FASTQC_INSTALLATION_PATH
	fi
	
	FASTQC_INSTALLATION_PATH=$FASTQC_INSTALLATION_PATH/FastQC/
	
	# Correct the GetOpts import for fastqc so it will allow the perl script to pass values to the fastqc java executable.
	perl -p -i -e 's/^use Getopt::Long;/# MAYO MODIFICATION: add the pass_through config option so Java arguments\n# are passed to the FastQC jar\nuse Getopt::Long qw(:config pass_through);\n#use Getopt::Long;h/' $FASTQC_INSTALLATION_PATH/fastqc
	
	chmod 755 $FASTQC_INSTALLATION_PATH/fastqc
fi

# Install BWA version 0.5.9
if [[ $SKIP_BWA = true ]]
then
	echo "Skipping BWA installation"
	
else
	BWA_INSTALLATION_PATH=$TOOL_INSTALL_DIR/bwa/
	echo "Installing BWA into $BWA_INSTALLATION_PATH"
	
	wget --directory-prefix=$BWA_INSTALLATION_PATH $BWA_URL
	
	if [[ $VERBOSE = true ]]; then 
		tar -C $BWA_INSTALLATION_PATH -jxvf $BWA_INSTALLATION_PATH/*.tar.bz2;
	else 
		tar -C $BWA_INSTALLATION_PATH -jxf $BWA_INSTALLATION_PATH/*.tar.bz2; 
	fi
	
	BWA_INSTALLATION_PATH=`find $BWA_INSTALLATION_PATH -type d -mindepth 1 -maxdepth 1`
	cd $BWA_INSTALLATION_PATH
	make && make all
	
fi

# Install MACS version 2.0.10
if [[ $SKIP_MACS = true ]]
then
	echo "Skipping Macs2 installation"
else
	MACS_INSTALLATION_PATH=$TOOL_INSTALL_DIR/macs2/
	echo "Installing Macs2 into $MACS_INSTALLATION_PATH"
	
	wget --directory-prefix=$MACS_INSTALLATION_PATH $MACS_URL
	
	if [[ $VERBOSE = true ]]; then 
		tar -C $MACS_INSTALLATION_PATH -xvf $MACS_INSTALLATION_PATH/master;
	else 
		tar -C $MACS_INSTALLATION_PATH -xf $MACS_INSTALLATION_PATH/master; 
	fi
	
	cd `find $MACS_INSTALLATION_PATH -mindepth 1 -maxdepth 1 -type d `
	
	$PYTHON setup.py install --prefix $MACS_INSTALLATION_PATH
	
	if [[ $? != 0 ]] 
	then
		WARNINGS="$WARNINGS\nWARNING: MACS installation did not complete successfully. Please follow the install directions at https://github.com/taoliu/MACS/blob/master/INSTALL.rst"
		echo -e "\tWARNING: MACS installation did not complete successfully. Please follow the install directions at https://github.com/taoliu/MACS/blob/master/INSTALL.rst"
	else
		export PYTHONPATH=$MACS_INSTALLATION_PATH/lib/python2.7/site-packages:$PYTHONPATH
		export PATH=$MACS_INSTALLATION_PATH/bin:$PATH
	
		WARNINGS="$WARNINGS\nWARNING: You will need to add the following lines to your .bashrc if you wish to continue using the chip-seq analysis pipeline after installation.\n\texport PYTHONPATH=$MACS_INSTALLATION_PATH/lib/python2.7/site-packages:\$PYTHONPATH\n\texport PATH=$MACS_INSTALLATION_PATH/bin:\$PATH"
		echo -e "\tWARNING: You will need to add the following lines to your .bashrc if you wish to continue using the chip-seq analysis pipeline after installation.\n\n\t\texport PYTHONPATH=$MACS_INSTALLATION_PATH/lib/python2.7/site-packages:\$PYTHONPATH\n\t\texport PATH=$MACS_INSTALLATION_PATH/bin:\$PATH"
	fi
fi

# Install SICER version 1.1
if [[ $SKIP_SICER = true ]]
then
	echo "Skipping SICER installation"
else
	SICER_INSTALLATION_PATH=$TOOL_INSTALL_DIR/sicer/
	echo "Installing SICER into $SICER_INSTALLATION_PATH"
	
	wget --directory-prefix=$SICER_INSTALLATION_PATH $SICER_URL
	
	if [[ $VERBOSE = true ]]; then 
		tar -C $SICER_INSTALLATION_PATH -zxvf $SICER_INSTALLATION_PATH/*.tgz;
	else 
		tar -C $SICER_INSTALLATION_PATH -zxf $SICER_INSTALLATION_PATH/*.tgz; 
	fi
	
	SICER_INSTALLATION_PATH=`find $SICER_INSTALLATION_PATH -type d -mindepth 1 -maxdepth 1`
	
	perl -p -i -e "s#/home/data/SICER1.1#$SICER_INSTALLATION_PATH#" $SICER_INSTALLATION_PATH/SICER/*.sh
	
	SICER_INSTALLATION_PATH=$SICER_INSTALLATION_PATH/SICER/
	
	chmod +x $SICER_INSTALLATION_PATH/*.sh
fi

# Install wigToBigWig
if [[ $SKIP_WIGTOBIGWIG = true ]]
then
	echo "Skipping wigToBigWig installation"
else
	WIGTOBIGWIG_INSTALLATION_PATH=$TOOL_INSTALL_DIR/wigtobigwig/
	echo "Installing wigToBigWig into $WIGTOBIGWIG_INSTALLATION_PATH"
	
	wget -q --directory-prefix=$WIGTOBIGWIG_INSTALLATION_PATH $WIGTOBIGWIG_URL
	chmod +x $WIGTOBIGWIG_INSTALLATION_PATH/wigToBigWig
fi

# Install IGVTools version 2.3.16
if [[ $SKIP_IGVTOOLS = true ]]
then
	echo "Skipping IGVTools installation"
else
	IGVTOOLS_INSTALLATION_PATH=$TOOL_INSTALL_DIR/igvtools/
	echo "Installing IGVTools into $IGVTOOLS_INSTALLATION_PATH"
	
	wget -q --directory-prefix=$IGVTOOLS_INSTALLATION_PATH $IGVTOOLS_URL
	if [[ $VERBOSE = true ]]
	then 
		unzip $IGVTOOLS_INSTALLATION_PATH/*.zip -d $IGVTOOLS_INSTALLATION_PATH
	else
		unzip -q $IGVTOOLS_INSTALLATION_PATH/*.zip -d $IGVTOOLS_INSTALLATION_PATH
	fi
	
	IGVTOOLS_INSTALLATION_PATH=$IGVTOOLS_INSTALLATION_PATH/IGVTools/
fi

# Install Samtools version 0.1.19
if [[ $SKIP_SAMTOOLS = true ]]
then
	echo "Skipping SamTools installation"
else
	if [[ $VERBOSE = true ]]; then echo -e "\tInstalling\e[1;32m SamTools\e[0m into $TOOL_INSTALL_DIR/samtools/"; fi
	mkdir -p $TOOL_INSTALL_DIR/samtools/
	cd $TOOL_INSTALL_DIR/samtools/
	wget -q --directory-prefix=$TOOL_INSTALL_DIR/samtools/ $SAMTOOLS_URL
	TAR=$(ls $TOOL_INSTALL_DIR/samtools/samtools*tar.bz2)
	if [[ $VERBOSE = true ]]; then tar -jxvf $TAR;
	else tar -jxf $TAR; fi
	TOOLDIR=$(find . -maxdepth 1 -type d -name 'samtools*' | cut -f2 -d'/')
	cd $TOOLDIR
	make
	
	INSTALL_CHECK=$(ls ${TOOL_INSTALL_DIR}/samtools/${TOOLDIR}/ | grep samtools)
	if [[ -z $INSTALL_CHECK ]]
	then
		echo -e "\t\e[1;31mERROR: Failed to download SamTools\e[0m\n"; 
	else
		echo -e "\tSuccessfully downloaded\e[1;34m SamTools\e[0m\n";
	fi
	SAMTOOLS_PATH="$TOOL_INSTALL_DIR/samtools/$TOOLDIR"
	cd $CURRENT_WORKING_DIRECTORY
fi

set -x

# Install MEME version 4.8.1
if [[ $SKIP_MEME = true ]]
then
	echo "Skipping MEME installation"
else
	if [[ $VERBOSE = true ]]; then echo -e "\tInstalling\e[1;32m MEME\e[0m into $TOOL_INSTALL_DIR/meme/"; fi
	mkdir -p $TOOL_INSTALL_DIR/meme/
	
	TOOL_SRC_DIR=$INSTALL_DIR/src/meme
	mkdir -p $TOOL_SRC_DIR
	wget -q --directory-prefix=$TOOL_SRC_DIR $MEME_URL
	cd $TOOL_SRC_DIR

	TAR=$(ls $TOOL_SRC_DIR/meme*tar.gz)
	if [[ $VERBOSE = true ]]; 
	then 
		tar -C $TOOL_SRC_DIR -zxvf $TAR;
	else 
		tar -C $TOOL_SRC_DIR -zxf $TAR; 
	fi
	
	TOOLDIR=$(find $TOOL_SRC_DIR -maxdepth 1 -mindepth 1 -type d -name 'meme*')
	cd $TOOLDIR
	./configure --prefix=$TOOL_INSTALL_DIR/meme/ --with-url=http://meme.nbcr.net/meme 
	
	if [[ $? != '0' ]]
    then
    	WARNINGS="$WARNINGS\nWARNING: MEME was not successfully configured. Please correct this before running ChIP-Seq."
		echo -e "\tWARNING: MEME was not successfully configured. Please correct this before running ChIP-Seq."
    else
    	make
    	if [[ $? != '0' ]]
    	then
    		WARNINGS="$WARNINGS\nWARNING: MEME was not successfully built. Please correct this before running ChIP-Seq."
			echo -e "\tWARNING: MEME was not successfully built. Please correct this before running ChIP-Seq."
		else
			make test
			if [[ $? != '0' ]]
    		then
    			WARNINGS="$WARNINGS\nWARNING: MEME did not pass its built-in checks. Please correct this before running ChIP-Seq."
				echo -e "\tWARNING: MEME did not pass its built-in checks. Please correct this before running ChIP-Seq."
			else
				make install
				if [[ $? != '0' ]]
    			then
    				WARNINGS="$WARNINGS\nWARNING: MEME did not install correctly. Please correct this before running ChIP-Seq."
					echo -e "\tWARNING: MEME did not install correctly. Please correct this before running ChIP-Seq."
				fi
			fi
    	fi
    fi
	
	INSTALL_CHECK=$(ls ${TOOL_INSTALL_DIR}/meme/bin | grep meme)
	if [[ -z $INSTALL_CHECK ]]
	then
		echo -e "\t\e[1;31mERROR: Failed to download MEME\e[0m\n"; 
	else
		echo -e "\tSuccessfully downloaded\e[1;34m MEME\e[0m\n";
		MEME_PATH="${TOOL_INSTALL_DIR}/meme/bin/meme"
	fi
	
	cd $CURRENT_WORKING_DIRECTORY
fi

set +x

# Install CEAS version 1.0.2
if [[ $SKIP_CEAS = true ]]
then
	echo "Skipping CEAS installation"
else
	CEAS_PATH="$TOOL_INSTALL_DIR/ceas/"
	if [[ $VERBOSE = true ]]; then echo -e "\tInstalling\e[1;32m CEAS\e[0m into $CEAS_PATH"; fi
	mkdir -p $CEAS_PATH
	cd $CEAS_PATH
	wget -q --directory-prefix=$CEAS_PATH $CEAS_URL
	TAR=$(ls $CEAS_PATH/CEAS*tar.gz)
	if [[ $VERBOSE = true ]]; then tar -zxvf $TAR;
	else tar -zxf $TAR; fi
	
	TOOLDIR=$(find . -maxdepth 1 -type d -name 'CEAS*' | cut -f2 -d'/')
	CEAS_PATH=$CEAS_PATH/$TOOLDIR
	
	cd $TOOLDIR
	python setup.py install --prefix="$CEAS_PATH"
	
	INSTALL_CHECK=$( ls $CEAS_PATH/bin | grep ceas)
	if [[ -z $INSTALL_CHECK ]]
	then
		echo -e "\t\e[1;31mERROR: Failed to download CEAS\e[0m\n"; 
	else
		echo -e "\tSuccessfully downloaded\e[1;34m CEAS\e[0m\n";
	fi
	
	export PYTHONPATH=$CEAS_PATH/lib/python2.7/site-packages:$PYTHONPATH
	export PATH=$CEAS_PATH/bin:$PATH
	
	WARNINGS="$WARNINGS\nWARNING: You will need to add the following lines to your .bashrc if you wish to continue using the chip-seq analysis pipeline after installation.\n\texport PYTHONPATH=$CEAS_PATH/lib/python2.7/site-packages:\$PYTHONPATH\n\texport PATH=$CEAS_PATH/bin:\$PATH"
	echo -e "\tWARNING: You will need to add the following lines to your .bashrc if you wish to continue using the chip-seq analysis pipeline after installation.\n\n\t\texport PYTHONPATH=$CEAS_PATH/lib/python2.7/site-packages:\$PYTHONPATH\n\t\texport PATH=$CEAS_PATH/bin:\$PATH"
	
	chmod +x $CEAS_PATH/bin/*	
	
	cd $CURRENT_WORKING_DIRECTORY
fi

# Install Picard version 1.97
if [[ $SKIP_PICARD = true ]]
then
	echo "Skipping Picard installation"
else
	if [[ $VERBOSE = true ]]; then echo -e "\tInstalling\e[1;32m Picard\e[0m into $TOOL_INSTALL_DIR/picard/"; fi
	mkdir -p $TOOL_INSTALL_DIR/picard/
	cd $TOOL_INSTALL_DIR/picard/
	wget -q --directory-prefix=$TOOL_INSTALL_DIR/picard/ $PICARD_URL
	ZIP=$(ls $TOOL_INSTALL_DIR/picard/picard*zip)
	if [[ $VERBOSE = true ]]; then unzip $ZIP;
	else unzip -qq $ZIP; fi
	
	TOOLDIR=$(find . -maxdepth 1 -type d -name 'picard*' | cut -f2 -d'/')
	INSTALL_CHECK=$(ls ${TOOLDIR} | grep MarkDuplicates.jar)
	if [[ -z $INSTALL_CHECK ]]
	then
		echo -e "\t\e[1;31mERROR: Failed to download Picard\e[0m\n"; 
	else
		echo -e "\tSuccessfully downloaded\e[1;34m Picard\e[0m\n";
	fi
	PICARD_PATH="$(pwd)/$TOOLDIR"
	cd $CURRENT_WORKING_DIRECTORY
fi

# Install BEDTools version 2.17.0
if [[ $SKIP_BEDTOOLS = true ]]
then
	echo "Skipping BEDTools installation"
else
	if [[ $VERBOSE = true ]]; then echo -e "\tInstalling\e[1;32m BedTools\e[0m into $TOOL_INSTALL_DIR/bedtools/"; fi
	mkdir -p $TOOL_INSTALL_DIR/bedtools/
	cd $TOOL_INSTALL_DIR/bedtools/
	wget -q --directory-prefix=$TOOL_INSTALL_DIR/bedtools/ $BEDTOOLS_URL
	TAR=$(ls $TOOL_INSTALL_DIR/bedtools/BED*tar.gz)
	if [[ $VERBOSE = true ]]; then tar -zxvf $TAR;
	else tar -zxf $TAR; fi
	
	TOOLDIR=$(find . -maxdepth 1 -type d -name 'bed*' | cut -f2 -d'/')
	if [ -d "$TOOLDIR" ]
	then 
		cd $TOOLDIR
		if [[ $VERBOSE = true ]];then make clean; make all; 
		else make clean > install.log 2>&1; make all > install.log 2>&1; fi
		
		INSTALL_CHECK=$(ls bin | grep bedtools)
		if [[ -z $INSTALL_CHECK ]]
		then
			echo -e "\t\e[1;31mERROR: Failed to install Bedtools\e[0m\n"; 
		else
			echo -e "\tSuccessfully installed\e[1;34m Bedtools\e[0m\n";
		fi
	else
		echo -e "\t\e[1;31mERROR: Failed to install Bedtools\e[0m\n"; 
	fi
	BEDTOOLS_PATH="$(pwd)/bin"
	cd $CURRENT_WORKING_DIRECTORY
fi

# Pull down documentation and images
mkdir $DOCS_DIR
wget -q --directory-prefix=$DOCS_DIR $WORKFLOW_DOC_URL
wget -q --directory-prefix=$DOCS_DIR $WORKFLOW_IMG_URL
wget -q --directory-prefix=$DOCS_DIR $COLOR_LIST_URL

WORKFLOW_DOC=$DOCS_DIR/HiCHIP_workflow_summary.doc
WORKFLOW_IMG=$DOCS_DIR/chipseq-pic.jpg
COLOR_LIST=$DOCS_DIR/track_color_list.txt

# Download references
# Don't forget to check disk space before downloading
echo -e "\n\n"
if [[ $SKIP_REFERENCES == false ]] 
then
	read -p "Where are your references located? " REFS
fi

if [[ $REFS == "" ]]; then echo -e "  Empty string.\n  You will need to edit\e[1;36m tool_info.txt\e[0m manually\n"; fi


# Generate tool_info.txt
# TODO - Make sure every variable has been assigned to a non-Mayo-specific directory
cat << EOF > $INSTALL_DIR/config/tool_info.txt
############################################ 
####   CONFIGURATION FILE for ChIP-Seq  ####
############################################
# This is the default configuration file for ChipSeq v2.1 Workflow
# Created by the installer on `date`

################
## references ##
################
BWA_REF=$REFS/allchr.fa
CEAS_REF=$REFS/hg19.refGene
UCSC_REF_FLAT=$REFS/ucsc_refflat_hg19_2011-01-24.txt
GENOME_TABLE=$REFS/genome_table.human.hg19.txt
IGV_REFERENCE_GENOME=$REFS/hg19.genome
ANNOTATION_DIR=$REFS/annotation/
GENE_TSS=Hg19.great2.0.genes.chr17.txt
ANTI_GAP=hg19.non_gap_chr17.bed
TEST_TERM=hs.GO_annotation.txt

###########
## tools ##
###########
CHIPSEQ_DIR=$INSTALL_DIR
PYTHON=$PYTHON_INSTALLATION_PATH
SICER=$SICER_INSTALLATION_PATH
BWA_PATH=$BWA_INSTALLATION_PATH
MACS_PATH=$MACS_INSTALLATION_PATH/bin/
MACS_PYTHON=$MACS_INSTALLATION_PATH/lib/python2.7/site-packages
CEAS_PATH=$CEAS_PATH/bin/
CEAS_PYTHON=$CEAS_PATH/lib/python2.7/site-packages/
JAVA=$JAVA_INSTALLATION_PATH
SAMTOOLS=$SAMTOOLS_PATH
BEDTOOLS=$BEDTOOLS_PATH
PICARD=$PICARD_PATH
FASTQC=$FASTQC_INSTALLATION_PATH/fastqc
R_PATH=$R_INSTALLATION_PATH/bin/R
MEME_PATH=$MEME_PATH
WIG2BIGWIG=$WIGTOBIGWIG_INSTALLATION_PATH/wigToBigWig
IGVTOOLS=$IGVTOOLS_INSTALLATION_PATH/igvtools

###############
## Constants ##
###############
IGV_LINK=ftp://my_server/prefix
IGV_SETUP_PDF=$REFS/IGV_Setup.pdf
WORKFLOW_SUMMARY_DOC=$WORKFLOW_DOC
WORKFLOW_SUMMARY_IMG=$WORKFLOW_IMG
TOOL_VERSION=ChipSeq.2.0
ORGANISM=human
LOCATION=MySite
PLATFORM=Illumina
ANNO_TYPE=GO

#### [Samtools] track_color_list.txt, FRAGMENT_SIZE, and STEP_SIZE are used to generate raw bedgraph/wig files
TCLR_LIST=$COLOR_LIST
FRAGMENT_SIZE=200
STEP_SIZE=20

#### parameters for BWA mapping
## if single end reads (SE), please specify "MAP_SE_ARGS" and "MAP_BOTH_ARGS"
## if paired-end reads (PE), please specify "MAP_PE_ARGS" and "MAP_BOTH_ARGS"
MAP_SE_ARGS=-n 10
MAP_PE_ARGS=-n 10 -a 500 -o 10000 -N 10 -s
MAP_BOTH_ARGS=-o 1 -l 32 -t 4 -k 2 -m 80000000

#### process mapped reads
## "MAPPING_QUALITY" is the cutoff for filtering out low mapping quality reads, only for SE data
## If you do not want to filter out low mapping quality reads, specify as "0", [default: 0] 
## Specify "dedup" if you want to remove duplicates, otherwise specify "nodedup", [default: dedup] 
MAP_QUALITY=0

#### modifications:
## for "REMOVE_DUP", specify as "REMOVE_DUP=dedup" (remove duplicates) or as "REMOVE_DUP=nodedup" (do not remove replicates)
REMOVE_DUP=dedup

## SICER Constants
SICER_ARGS=hg19 1 200 300 0.75 600 1E-2

## MACS2 Constants
MACS2_ARGS=-f BAM -g hs --keep-dup all -q 0.01
#MACS2_ARGS=-f BAM -g hs -s 100 --keep-dup 1 -q 0.01

## IDR Constants
IDR_ARGS=0.3 F p.value hg19
IDR_CUTOFF=0.01

## SGE Queue Constants
QUEUE=7-days
OPEN_SOURCE=true
EOF


# Print out path variable changes and execution recommendations
# TODO Make sure to update this section 
echo -e "\n\n**********************************************"
echo -e "The ChIP-Seq installation has completed.\n"


if [[ ! -z $WARNINGS ]]
then
	echo "The system encountered the following warnings during installation:"
	echo -e "$WARNINGS"
	echo ""
	echo "Also note: You will need to update your $INSTALL_DIR/config/tool_info.txt file to include paths to any manually-installed tools."
fi