#!/bin/bash

set -x

if [ $# != 1 ]
then
	echo "Usage: delivery.sh <run info file>";
	exit
fi	

# Parse arguments from run_info.txt
WORK_DIR=$( grep -w '^WORK_DIR' $1 | cut -d '=' -f2)
SEQ_TYPE=$( grep -w '^SEQ_TYPE' $1 | cut -d '=' -f2)
IDR_ENABLED=$( grep -w '^IDR_ANALYSIS' $1 | cut -d '=' -f2)
PEAK_CALLER=$( grep -w '^PEAK_CALLER' $1 | cut -d '=' -f2)
TOOL_INFO=$( grep -w 'TOOL_INFO' $1 | cut -d '=' -f2)

# Parse arguments from tool_info.txt
CHIPSEQ_DIR=$( grep -w 'CHIPSEQ_DIR' $TOOL_INFO | cut -d '=' -f2)
IGV_SETUP_PDF=$( grep -w 'IGV_SETUP_PDF' $TOOL_INFO | cut -d '=' -f2)
WORKFLOW_SUMMARY_DOC=$( grep -w 'WORKFLOW_SUMMARY_DOC' $TOOL_INFO | cut -d '=' -f2)
WORKFLOW_SUMMARY_IMG=$( grep -w 'WORKFLOW_SUMMARY_IMG' $TOOL_INFO | cut -d '=' -f2)

BOOL_CEAS=`grep -w '^RUN_CEAS' $1 | cut -d '=' -f2 | tr [:upper:] [:lower:]`   # case insensitive
BOOL_MEME=`grep -w '^RUN_MEME' $1 | cut -d '=' -f2 | tr [:upper:] [:lower:]`   # case insensitive
BOOL_GOM=`grep -w '^RUN_GOM' $1 | cut -d '=' -f2 | tr [:upper:] [:lower:]` # case insensitive


# Constants used in this file
DELIVERY_DIR=$WORK_DIR/delivery

echo "Delivering files."

if [[ ! -d $DELIVERY_DIR ]] ### check if delivery dir exists, otherwise create it
then
	mkdir $DELIVERY_DIR
	mkdir $DELIVERY_DIR/fastqc
	mkdir $DELIVERY_DIR/mapout
	mkdir $DELIVERY_DIR/bedgraph
	
	if [ $PEAK_CALLER == "macs2idr" ] || [ $PEAK_CALLER == "macs2noidr" ]
	then
		mkdir $DELIVERY_DIR/macs2out
		if [ $PEAK_CALLER == "macs2idr" ]
		then
			mkdir $DELIVERY_DIR/idrout
		fi
	elif [ $PEAK_CALLER == "sicer" ]
	then
		mkdir $DELIVERY_DIR/sicerout
	fi
fi

# Copy constant files into the delivery directory
cp $IGV_SETUP_PDF $DELIVERY_DIR
cp $WORKFLOW_SUMMARY_DOC $DELIVERY_DIR
cp $1 $DELIVERY_DIR
cp $TOOL_INFO $DELIVERY_DIR

if [[ ! -z $WORKFLOW_SUMMARY_IMG ]] 
then
	cp $WORKFLOW_SUMMARY_IMG $DELIVERY_DIR
fi

# TODO - MTK: what happens if these files are missing? Should we do some error checking? Warn the user?
echo "Copying mapout files."

cp `find $WORK_DIR -name "*bdg.gz"` $DELIVERY_DIR/mapout
cp `find $WORK_DIR -name "*wig.gz"` $DELIVERY_DIR/mapout
cp `find $WORK_DIR -name "*[PS]E.sorted.bam"` $DELIVERY_DIR/mapout
cp `find $WORK_DIR -name "*.dedup.s1.bam"` $DELIVERY_DIR/mapout
cp `find $WORK_DIR -name "*tdf"` $DELIVERY_DIR/mapout

if [ $SEQ_TYPE == "PE" ]
then
	cp `find $WORK_DIR -name "*.bam.size.txt"` $DELIVERY_DIR/mapout
fi

#echo "Copying fastq files."
cp -R `find $WORK_DIR/fastqc -name "*_fastqc"` $DELIVERY_DIR/fastqc

# Copy MACS2 output to delivery directory
if [ $PEAK_CALLER == "macs2idr" ] || [ $PEAK_CALLER == "macs2noidr" ]
then
	if [ $PEAK_CALLER == "macs2idr" ]
	then
		PR="pr0_"
	fi
	echo "Copying files generated by macs2."
	cp `find $WORK_DIR -name "*$PR\macs2_peaks.encodePeak"` $DELIVERY_DIR/macs2out
	cp `find $WORK_DIR -name "*$PR\macs2_peaks.bed"` $DELIVERY_DIR/macs2out
	cp `find $WORK_DIR -name "*$PR\macs2_summits.bed"` $DELIVERY_DIR/macs2out
	cp `find $WORK_DIR -name "*$PR\macs2_peaks.xls"` $DELIVERY_DIR/macs2out
	cp `find $WORK_DIR -name "*.bdg"` $DELIVERY_DIR/bedgraph
	if [ $SEQ_TYPE == "SE" ]
	then
		cp `find $WORK_DIR -name "*$PR\macs2_model.r"` $DELIVERY_DIR/macs2out
	fi
fi

# Copy SICER output to delivery directory
if [ $PEAK_CALLER == "sicer" ]
then
	echo "Copying files generated by sicer."
	cp `find $WORK_DIR -name "*islands-summary*"` $DELIVERY_DIR/sicerout
	cp `find $WORK_DIR -name "*island.bed"` $DELIVERY_DIR/sicerout
fi

# Copy IDR output to delivery directory
if [ $PEAK_CALLER == "macs2idr" ]
then
	echo "Copying files generated by idr."
	cp `find $WORK_DIR/idrout -name "*overlapped-peaks*"` $DELIVERY_DIR/idrout
	cp `find $WORK_DIR/idrout -name "*uri.sav"` $DELIVERY_DIR/idrout
	cp `find $WORK_DIR/idrout -name "*.pdf"` $DELIVERY_DIR/idrout
	cp `find $WORK_DIR/idrout -name "*.encodePeak"` $DELIVERY_DIR/idrout
	cp `find $WORK_DIR/idrout -name "*summary.txt"` $DELIVERY_DIR/idrout
	cp `find $WORK_DIR/idrout -name "*combined.txt"` $DELIVERY_DIR/idrout
fi

# Copy peak_vs_gene.xls file to appropriate delivery location
if [ $PEAK_CALLER == "macs2idr" ]
then
	cp `find $WORK_DIR -name "*peak_vs_gene.xls"` $DELIVERY_DIR/idrout
elif [  $PEAK_CALLER == "macs2noidr" ]
then
	cp `find $WORK_DIR -name "*peak_vs_gene.xls"` $DELIVERY_DIR/macs2out
elif [  $PEAK_CALLER == "sicer" ]
then
	cp `find $WORK_DIR -name "*peak_vs_gene.xls"` $DELIVERY_DIR/sicerout
fi


# Copy annotation results where applicable. -> MEME
if [[ $BOOL_MEME == "yes" ]]
then
	mkdir -p $DELIVERY_DIR/annotation/meme
	MEME_FILE_LIST=`find $WORK_DIR/memeout  -mindepth 1 -type d`
    for dir in $MEME_FILE_LIST
    do 
		cp -R $dir $DELIVERY_DIR/annotation/meme
    done
fi

# Copy annotation results where applicable. -> CEAS
if [[ $BOOL_CEAS == "yes" ]]
then	
	mkdir -p $DELIVERY_DIR/annotation/ceas
	cp $WORK_DIR/ceasout/*pdf $DELIVERY_DIR/annotation/ceas
	cp $WORK_DIR/ceasout/*xls $DELIVERY_DIR/annotation/ceas
fi

# Copy annotation results where applicable. -> GOM
if [[ $BOOL_GOM == "yes" ]]
then
	mkdir -p $DELIVERY_DIR/annotation/go
	cp $WORK_DIR/geneOntology/*.tab_bt_out.txt $DELIVERY_DIR/annotation/go
fi

# Create the HTML documentation
perl $CHIPSEQ_DIR/createMainDocument.pl -r $1
perl $CHIPSEQ_DIR/igv.pl $1

echo "Results are now available in $DELIVERY_DIR"

