#!/bin/bash
# Usage: dashboard.sh -r <runinfo> -s <state> [-c]
#set -x

usage() {
	cat <<EOF
	
Usage: dashboard.sh -s <state> [-c]
Update NGS Portal with specified state for given run_info file.

Mandatory arguments to long options are mandatory for short options too.

  -c, --complete	(optional) If specified, indicates that the step has completed.
  -h, --help		Print this usage information.
  -r, --runinfo     Specify the file containing the options that need to be run.
  -s, --state		State (Beginning, Started, Alignment, ... ) 
  
Report problems using dashboard.sh using FogBugz (Mayo internal site: http://bsu-bugs)
EOF
}

TEMP=`getopt -o chr:s: --long complete,help,runinfo:,state: -n 'dashboard.sh' -- "$@"`

eval set -- "$TEMP"

IS_COMPLETE=
RUN_INFO=
STATE=

while true; do
	case "$1" in
		-h | --help ) usage; exit ;;
		-c | --complete ) IS_COMPLETE="-c"; shift ;;
		-r | --runinfo ) RUN_INFO=$2; shift 2 ;;
		-s | --state ) STATE=$2; shift 2 ;;
		-- ) break; ;;
		* ) usage; exit; ;;
	esac
done

# Check required parameters
if [[ $RUN_INFO = "" || ! -s $RUN_INFO ]]
then
	echo "run_info file is a required parameter, but was not specified."
	exit 1
fi

if [[ $STATE = "" ]]
then
	echo "The new state is a required parameter, but was not specified."
	exit 1
fi

# Parse variables from run/tool_info.
USEREMAIL=`grep -w '^USEREMAIL' $RUN_INFO | cut -d '=' -f2`
WORK_DIR=`grep -w '^WORK_DIR' $RUN_INFO | cut -d '=' -f2`
RUNID=`grep -w '^RUNID' $RUN_INFO | cut -d '=' -f2`
SEQ_TYPE=`grep -w '^SEQ_TYPE' $RUN_INFO | cut -d '=' -f2`
FILTER_TYPE=`grep -w '^FILTER_TYPE' $RUN_INFO | cut -d '=' -f2`
END1_SEQ=( $(awk 'BEGIN {FS="="} {if ($1 ~ /^END1_SEQ/) print $2}' $RUN_INFO) )
END2_SEQ=( $(awk 'BEGIN {FS="="} {if ($1 ~ /^END2_SEQ/) print $2}' $RUN_INFO) )
tool_info=`grep -w '^TOOL_INFO' $RUN_INFO | cut -d '=' -f2`
MEMORY_INFO=`grep -w '^MEMORY_INFO' $RUN_INFO | cut -d '=' -f2`
PORTAL_PROJECT_NAME=`grep -w '^PORTAL_PROJECT_NAME' $RUN_INFO | cut -d '=' -f2`

# Parse tool_info file
JAVA=`grep -w '^JAVA' $tool_info | cut -d '=' -f2`
NGS_PORTAL_PATH=`grep -w '^NGS_PORTAL_PATH' $tool_info | cut -d '=' -f2`

# Parse memory_info file
PORTAL_JVM=`grep -w '^PORTAL_JVM' $MEMORY_INFO | cut -d '=' -f2`

SEC_ANLYS=$NGS_PORTAL_PATH"/"AddSecondaryAnalysis.jar
PROPS=$NGS_PORTAL_PATH"/"AddSecondaryAnalysis.properties

# Build up the list of LANE, LAB INDEX, and FLOWCELL_ID combinations

UNIQUE_ID_COUNT=0

ALL_SEQS=("${END1_SEQ[@]}" "${END2_SEQ[@]}")

for sequence in ${ALL_SEQS[@]}
do
	# Temporary assignment
    TMP_LANE=`echo $sequence | perl -p -e 's/.*_L([0-9])_.*/$1/'`
    TMP_LABINDEX=`echo $sequence | perl -p -i -e 's/.*_I([GATC]+).*/$1/'`
    TMP_FLOWCELL=`echo $sequence | perl -p -i -e 's/.*\.FC([A-Z0-9]+)_.*/$1/'`
    
    ALL_UNIQUE=true
    
    # The following loop attempts to avoid duplicates
	for (( i=0; i<UNIQUE_ID_COUNT && ALL_UNIQUE==true; i++ ))
    do
	    
    	if [[ ${LANE[$i]} = $TMP_LANE  && ${LABINDEX[$i]} = $TMP_LABINDEX && ${FLOWCELL_ID[$i]} = $TMP_FLOWCELL ]]
    	then
    		ALL_UNIQUE=false
    	fi
    	
    done
    
    if [[ $ALL_UNIQUE = true ]]
    then
    	LANE[$UNIQUE_ID_COUNT]=$TMP_LANE
    	LABINDEX[$UNIQUE_ID_COUNT]=$TMP_LABINDEX
    	FLOWCELL_ID[$(( UNIQUE_ID_COUNT++ ))]=$TMP_FLOWCELL
    fi
done

# if any of the list of lanes, lab indexes, or flow cells are empty, abort the update.
if [[ ${#LANE[@]} = 0 || ${#LABINDEX[@]} = 0 || ${#FLOWCELL_ID[@]} = 0 ]]
then
	echo "Unable to find at least one of the: lanes, lab indexes, or flowcells, please check input parameters."
	echo "Number of lanes = ${#LANE[@]}"
	echo "Number of lab indexes = ${#LABINDEX[@]}"
	echo "Number of flow cells = ${#FLOWCELL_ID[@]}"
    exit 1
fi

for (( i=0; i<UNIQUE_ID_COUNT; i++ ))
do
$JAVA $PORTAL_JVM -jar $SEC_ANLYS -p $PROPS -S http://rcftomprod1 -r $RUNID -f ${FLOWCELL_ID[$i]} -l ${LANE[$i]} -i ${LABINDEX[$i]} --analysisName $PORTAL_PROJECT_NAME -s $STATE -a ChipSeq $COMPLETE
done
