#!/bin/bash
# Usage: /projects/bsi/bictools/scripts/ChIPSeq/tags/v2.0/chipseq.sh -r /home/m110443/projects/chipseq2_svn/CONFIG/run_info.txt [--keepintermediatefiles ]
# Usage note: Make sure mayobiotools -> python v 2.7.3 is on 
# Usage note: Make sure mayobiotools -> R-2.14.0 is on
# this is my version 
echo `date`
echo ""

usage() {
	cat <<EOF
	
Usage: chipseq.sh -r /full/path/to/run_info.txt [--keepintermediatefiles] [--skip-align]
Run ChIP-Seq workflow using specified run_info.txt file.

Mandatory arguments to long options are mandatory for short options too.

  --keepintermediatefiles     Keep intermediate files instead of cleaning up after
                                workflow execution.
  -r, --runinfo               Specify the file containing the options that need to be run.
  --skip-align                Skip the alignment step  

Report problems using chipseq.sh using FogBugz (Mayo internal site: http://bsu-bugs)
EOF
}

TEMP=`getopt -o hr: --long help,runinfo:,keepintermediatefiles,skip-align,skipalign,skipalignment -n 'chipseq.sh' -- "$@"`

eval set -- "$TEMP"

RUN_INFO=
CLEAN_UP_FILES=true
PERFORM_ALIGNMENT=true

while true; do
	case "$1" in
		-r | --runinfo ) RUN_INFO=$2; shift 2;;
		--keepintermediatefiles ) CLEAN_UP_FILES=false; shift ;;
		-h | --help ) usage; exit ;;
		--skip-align | --skipalign | --skipalignment ) PERFORM_ALIGNMENT=false; shift ;;
		-- ) break; ;;
		* ) usage; exit; ;;
	esac
done

set -x
	
# perform initial check to ensure run info file exists
if [ ! $RUN_INFO ]
then
	usage;
	echo -e "\n\nERROR : run_info file was not specified";
	exit 1;
fi

dir_info=`dirname $RUN_INFO`
if [ "$dir_info" = "." ]
then
	echo -e "ERROR : run_info=$RUN_INFO should be specified as a complete path\n";
	exit 1;
fi

if [ ! -s $RUN_INFO ]
then
	echo -e "ERROR : run_info=$RUN_INFO does not exist\n";
	exit 1;
fi

### Remove tailing white space...it screws up the naming schemes
sed -e 's/ *$//' $RUN_INFO > tmp.txt 
mv tmp.txt $RUN_INFO
dos2unix $RUN_INFO

# Gather variables from configuration files
TOOL_INFO=`grep -w '^TOOL_INFO' $RUN_INFO | cut -d '=' -f2`

if [ ! -s $TOOL_INFO ]
then
	echo -e "ERROR : tool_info=$TOOL_INFO does not exist\n";
	exit 1;
fi

SOURCE_DIR=`grep -w '^CHIPSEQ_DIR' $TOOL_INFO | cut -d '=' -f2`

##### Check/Validate All Parameters #####
VARCHECK=`perl $SOURCE_DIR/checkCONFIG.pl $RUN_INFO`
if [ "$VARCHECK" != "" ]
then
	echo $VARCHECK
	exit 1;
fi

USEREMAIL=`grep -w '^USEREMAIL' $RUN_INFO | cut -d '=' -f2`
PEAK_CALLER=`grep -w '^PEAK_CALLER' $RUN_INFO | cut -d '=' -f2`
WORK_DIR=`grep -w '^WORK_DIR' $RUN_INFO | cut -d '=' -f2`
RUNID=`grep -w '^RUNID' $RUN_INFO | cut -d '=' -f2`
SAMPLENAME=`grep -w '^SAMPLENAME' $RUN_INFO | cut -d '=' -f2`
## TODO: add check to ensure no spaces exist in samplename
SEQ_TYPE=`grep -w '^SEQ_TYPE' $RUN_INFO | cut -d '=' -f2`
END1_SEQ=( $(awk 'BEGIN {FS="="} {if ($1 ~ /^END1_SEQ/) print $2}' $RUN_INFO) )
FILTER_TYPE=`grep -w '^FILTER_TYPE' $RUN_INFO | cut -d '=' -f2`
## TODO: check if single, must be U1 or U2, check for synergy
IP_FILE=( $(awk 'BEGIN {FS="="} {if ($1 ~ /^IP_FILE/) print $2}' $RUN_INFO) )
INPUT_FILE=( $(awk 'BEGIN {FS="="} {if ($1 ~ /^INPUT_FILE/) print $2}' $RUN_INFO) )
MEMORY_INFO=`grep -w '^MEMORY_INFO' $RUN_INFO | cut -d '=' -f2`
PORTAL_PROJECT_NAME=`grep -w '^PORTAL_PROJECT_NAME' $RUN_INFO | cut -d '=' -f2`
BOOL_CEAS=`grep -w '^RUN_CEAS' $RUN_INFO | cut -d '=' -f2 | tr [:upper:] [:lower:]`   # case insensitive
BOOL_MEME=`grep -w '^RUN_MEME' $RUN_INFO | cut -d '=' -f2 | tr [:upper:] [:lower:]`   # case insensitive
BOOL_GOM=`grep -w '^RUN_GOM' $RUN_INFO | cut -d '=' -f2 | tr [:upper:] [:lower:]` # case insensitive


### IF PE CHECK U02, U12, U22
### ELSE check SE U0 or U1
if [ "$SEQ_TYPE" == "PE" ]
then
	if [[ "$FILTER_TYPE" == "U02" || "$FILTER_TYPE" == "U12" || "$FILTER_TYPE" == "U22" ]]
	then
		echo "Filter is good: $FILTER_TYPE"
	else
		echo "BAD Filter Param: $FILTER_TYPE"
		exit 1;
	fi
elif [ "$SEQ_TYPE" == "SE" ]
then
	if [[ "$FILTER_TYPE" == "U0" || "$FILTER_TYPE" == "U1" ]]
	then
		echo "Filter is good: $FILTER_TYPE"
	else
		echo "BAD Filter Param: $FILTER_TYPE"
		exit 1;
	fi
else
	echo "BAD Seq Param: $SEQ_TYPE"
	exit 1;
fi


if [[ ! -e $TOOL_INFO ]]
then
	echo "The tool_info file specified at location $TOOL_INFO does not exist. Please verify that your run_info file contains the correct tool_info file location."
	exit 1
fi

# Parse tool_info.txt file

JAVA=`grep -w '^JAVA' $TOOL_INFO | cut -d '=' -f2`
TOOL_VERSION=`grep -w '^TOOL_VERSION' $TOOL_INFO | cut -d '=' -f2`
NGS_PORTAL_PATH=`grep -w '^NGS_PORTAL_PATH' $TOOL_INFO | cut -d '=' -f2`
QUEUE=`grep -w '^QUEUE' $TOOL_INFO | cut -d '=' -f2`
OPEN_SOURCE=`grep -w '^OPEN_SOURCE' $TOOL_INFO | cut -d '=' -f2`

if [[ ! -e $MEMORY_INFO ]]
then
	echo "The memory_info file specified at location $MEMORY_INFO does not exist. Please verify that your run_info file contains the correct memory_info file location."
	exit 1
fi

# Parse memory_info.txt file
ALIGN_VMEM=`grep -w '^ALIGN_VMEM' $MEMORY_INFO | cut -d '=' -f2`
CLEANUP_VMEM=`grep -w '^CLEANUP_VMEM' $MEMORY_INFO | cut -d '=' -f2`
DASHBOARD_VMEM=`grep -w '^DASHBOARD_VMEM' $MEMORY_INFO | cut -d '=' -f2`
DEFAULT_VMEM=`grep -w '^DEFAULT_VMEM' $MEMORY_INFO | cut -d '=' -f2`
DELIVERY_VMEM=`grep -w '^DELIVERY_VMEM' $MEMORY_INFO | cut -d '=' -f2`
PORTAL_JVM=`grep -w '^PORTAL_JVM' $MEMORY_INFO | cut -d '=' -f2`

# Referenced scripts
ALIGN=$SOURCE_DIR"/"align.sh
MAPQC=$SOURCE_DIR"/"mapqc.sh
MAPVIS=$SOURCE_DIR"/"mapvis.sh
PKCALL=$SOURCE_DIR"/"pkcall.sh
NOIDR=$SOURCE_DIR"/"noidr.sh
IDRPREP=$SOURCE_DIR"/"idrprep.sh
REPCALL=$SOURCE_DIR"/"repcall.sh
IDR=$SOURCE_DIR"/"idr.sh
NEARGENES=$SOURCE_DIR"/"neargenes.sh
ANNOT_CEAS=$SOURCE_DIR"/"ceas_annotation.sh
ANNOT_MEME=$SOURCE_DIR"/"meme_annotation.sh
ANNOT_GOM=$SOURCE_DIR"/"gom_annotation.sh
DELIVERY=$SOURCE_DIR"/"delivery.sh
CLEANUP=$SOURCE_DIR"/"cleanup.sh
DASHBOARD=$SOURCE_DIR"/"dashboard.sh


SEC_ANLYS=$NGS_PORTAL_PATH"/"AddSecondaryAnalysis.jar
PROPS=$NGS_PORTAL_PATH"/"AddSecondaryAnalysis.properties

# Verify configuration file
# TODO: implement additional checks

if [ -e $WORK_DIR/delivery ]
then
	echo -e "ERROR : delivery directory=$WORK_DIR/delivery exists. Stopping to preserve data.\n"
#	exit 1
fi

if [[ ! $USEREMAIL ]]
then
	echo -e "ERROR : User email is required for submitting to SGE. Please check your run-info file.\n"
	exit 1
fi

NUM_IP=${#IP_FILE[@]}
NUM_INPUT=${#INPUT_FILE[@]}
if [ "$NUM_IP" -ne "$NUM_INPUT" ]
then
	echo -e "ERROR: Unequal number of IP Files as to INPUT Files!\nMust have 1-to-1 comparison explictily configured."
	exit 1
fi

set -x

UNIQ_IP=($(echo "${IP_FILE[@]}" | tr ' ' '\n' | sort | uniq | tr '\n' ' '))
NUM_UNIQ_IP=${#UNIQ_IP[@]}
if [ "$NUM_UNIQ_IP" -lt "$NUM_IP" ]
then
	INPUT_LABLES=`grep -w '^INPUT_LABLES' $RUN_INFO | cut -d '=' -f2`
	if [ ! "$INPUT_LABLES" ]
	then
		echo -e "ERROR: This configuration requires INPUT_LABLES in run_info file."
		exit 1
	fi
fi

# Set up final variables

### NOTE: For the -t 1-n , n is the number of samples i.e. separate entries/columns in the END1_SEQ or IP_FILE row (so n parallel array jobs)
t_array_jobs=${#END1_SEQ[@]}
t_array_IP=${#IP_FILE[@]}

# Note: this is only the labindex/lane of the first sequence. This is enough to get the group ID
LANE=`echo ${END1_SEQ[0]} | perl -p -e 's/.*_L([0-9])_.*/$1/'`
LABINDEX=`echo ${END1_SEQ[0]} | perl -p -i -e 's/.*_I([GATC]+).*/$1/'`

# the -terse option allows us to grab the job id and use it to build dependency graphs between qsub operations 
QSUBARGS="-terse -wd $WORK_DIR/logs -m ae -M $USEREMAIL -l h_stack=10M -q $QUEUE"

# Begin run
# If we are internal to Mayo, update the dashboard. Otherwise, skip this step.
if [[ $OPEN_SOURCE -eq "false" ]]
then
	$DASHBOARD -r $RUN_INFO -s Beginning -c 

	$DASHBOARD -r $RUN_INFO -s Alignment 
fi

# Make certain that output directory exists
mkdir -p $WORK_DIR/logs

if [[ $PERFORM_ALIGNMENT = true ]]
then
	ALIGN_JOB_ID=`qsub $QSUBARGS -N $TOOL_VERSION.align.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -t 1-$t_array_jobs -l h_vmem=$ALIGN_VMEM $ALIGN $RUN_INFO`
	# Strip array task info off job id
	#ALIGN_JOB_ID=`echo $ALIGN_JOB_ID | perl -pe "s/\..*//"`
	ALIGN_JOB_ID=$(echo $ALIGN_JOB_ID | cut -d ' ' -f3 | cut -d '.' -f1)
else
	# TODO: Move verification to separate file to make this cleaner
	# verify that aligned & sorted bams exist
	ERROR=false

    if [ -e $WORK_DIR/fastqc ]
    then
        # Check for directories for each of the inputs. If any of these are missing, exit with an error.
		for ((i=0;i<${#INPUT_FILE[@]};i=i+1))
		do
			DIRECTORY=$WORK_DIR/fastqc/$( basename ${INPUT_FILE[i]} .fastq.gz )_fastqc
			
			if [ ! -e $DIRECTORY ]
			then
				echo "ERROR: Unable to find $DIRECTORY, but ${INPUT_FILE[i]} is specified in run_info.txt."
				exit 1
			fi
		done
		
		for ((i=0;i<${#IP_FILE[@]};i=i+1))
		do
			DIRECTORY=$WORK_DIR/fastqc/$( basename ${IP_FILE[i]} .fastq.gz )_fastqc
			
			if [ ! -e $DIRECTORY ]
			then
				echo "ERROR: Unable to find $DIRECTORY, but ${IP_FILE[i]} is specified in run_info.txt."
				exit 1
			fi
		done
    else
    	ERROR=true	
	fi
	
	if [[ $ERROR = true ]]
	then
		echo "ERROR: Could not find $WORK_DIR/fastqc files. Please ensure fastqc files are available in $WORK_DIR/fastqc."
		exit 1;
	fi
	
	if [ -e $WORK_DIR/mapout ]
    then
        # Check for directories for each of the inputs. If any of these are missing, exit with an error.
		for ((i=0;i<${#INPUT_FILE[@]};i=i+1))
		do
			BASE_FILE_NAME=$WORK_DIR/mapout/$( basename ${INPUT_FILE[i]} .fastq.gz ).$SEQ_TYPE
			
			FILE_LIST_SIZE=0
			
			FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.sorted.bam
			FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.$FILTER_TYPE.dedup.s1.bam
			FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.$FILTER_TYPE.dedup_metrics.txt
			
			if [[ $SEQ_TYPE = "PE" ]]
			then
				FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.bam
				FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.bam.$FILTER_TYPE.bam
				FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.$FILTER_TYPE.dedup.bam
				FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.$FILTER_TYPE.sorted.bam
			fi
		   
			for((j=0;j<7;j=j+1))
			do
				FILE=${FILE_LIST[j]}
				if [ ! -f $FILE ]
				then
					echo "ERROR: Unable to find $FILE, but ${INPUT_FILE[i]} is specified in run_info.txt."
					exit 1
				fi
			done
		done
		
		for ((i=0;i<${#IP_FILE[@]};i=i+1))
		do
			BASE_FILE_NAME=$WORK_DIR/mapout/$( basename ${IP_FILE[i]} .fastq.gz ).$SEQ_TYPE
			
			FILE_LIST_SIZE=0
			
			FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.sorted.bam
			FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.$FILTER_TYPE.dedup.s1.bam
			FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.$FILTER_TYPE.dedup_metrics.txt
			
			if [[ $SEQ_TYPE = "PE" ]]
			then
				FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.bam
				FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.bam.$FILTER_TYPE.bam
				FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.$FILTER_TYPE.dedup.bam
				FILE_LIST[$(( FILE_LIST_SIZE++ ))]=$BASE_FILE_NAME.$FILTER_TYPE.sorted.bam
			fi
		   
			for((j=0;j<${#FILE_LIST[@]};j=j+1))
			do
				FILE=${FILE_LIST[j]}
				if [ ! -f $FILE ]
				then
					echo "ERROR: Unable to find $FILE, but ${IP_FILE[i]} is specified in run_info.txt."
					exit 1
				fi
			done
		done
    else
    	ERROR=true	
	fi
	
	if [[ $ERROR = true ]]
	then
		echo "ERROR: Could not find $WORK_DIR/mapout files. Please ensure all BAM/SAM/SAI files are available in $WORK_DIR/fastqc."
		exit 1;
	fi
	
	ALIGN_JOB_ID=-1
fi

# If we are internal to Mayo, update the dashboard. Otherwise, skip this step.
if [[ $OPEN_SOURCE -eq "false" ]]
then
	qsub $QSUBARGS -N $TOOL_VERSION.dashboard.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $ALIGN_JOB_ID -l h_vmem=$DASHBOARD_VMEM -b y $DASHBOARD -r $RUN_INFO -s Alignment -c 

	qsub $QSUBARGS -N $TOOL_VERSION.dashboard.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $ALIGN_JOB_ID -l h_vmem=$DASHBOARD_VMEM -b y $DASHBOARD -r $RUN_INFO -s PeakCalling  
fi

### 1.2. Generate mapping summary and QC files
MAPQC_JOB_ID=`qsub $QSUBARGS -N $TOOL_VERSION.mapqc.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $ALIGN_JOB_ID -t 1-$t_array_jobs -l h_vmem=$DEFAULT_VMEM $MAPQC $RUN_INFO`

# Strip array task info off job id
MAPQC_JOB_ID=`echo $MAPQC_JOB_ID | perl -pe "s/\..*//"`

### 1.3. Generate visualization files
qsub $QSUBARGS -N $TOOL_VERSION.mapvis.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $ALIGN_JOB_ID -t 1-$t_array_jobs -l h_vmem=$DEFAULT_VMEM $MAPVIS $RUN_INFO

### RM. I'm always having to run this every time anyway...I might as well add it until we have a better solution.
RETRY_JOB_ID=`qsub $QSUBARGS -N $TOOL_VERSION.wig2tdf.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $MAPQC_JOB_ID -l h_vmem=$DEFAULT_VMEM $SOURCE_DIR/wig2tdf.sh $WORK_DIR/mapout`

### 1.4. Run Peak Caller
PKCALL_JOB_ID=`qsub -N $TOOL_VERSION.pkcall.$SAMPLENAME.PN$PORTAL_PROJECT_NAME $QSUBARGS -hold_jid $RETRY_JOB_ID -t 1-$t_array_IP -l h_vmem=$DEFAULT_VMEM $PKCALL $RUN_INFO`

# Strip array task info off job id
PKCALL_JOB_ID=`echo $PKCALL_JOB_ID | perl -pe "s/\..*//"`

# If we are internal to Mayo, update the dashboard. Otherwise, skip this step.
if [[ $OPEN_SOURCE -eq "false" ]]
then
	qsub $QSUBARGS -N $TOOL_VERSION.dashboard.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $PKCALL_JOB_ID -l h_vmem=$DASHBOARD_VMEM -b y $DASHBOARD -r $RUN_INFO -s PeakCalling -c 

	qsub $QSUBARGS -N $TOOL_VERSION.dashboard.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $PKCALL_JOB_ID -l h_vmem=$DASHBOARD_VMEM -b y $DASHBOARD -r $RUN_INFO -s Statistics  
fi

#### 2. Collect statistics
if [[ $PEAK_CALLER = "sicer" ]]
then
	### 2.1 noidr.sh
	IDR_TMP_ID=`qsub $QSUBARGS -N $TOOL_VERSION.noidr.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $PKCALL_JOB_ID -t 1-$t_array_IP -l h_vmem=$DEFAULT_VMEM $NOIDR $RUN_INFO`
	IDR_DONE_JOB_ID=`echo $IDR_TMP_ID | cut -d ' ' -f3 | cut -d '.' -f1`



elif [[ $PEAK_CALLER = "macs2idr" ]]
then
	### 2.2 idrprep.sh 
	IDRPREP_JOB_ID=`qsub $QSUBARGS -N $TOOL_VERSION.idrprep.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $PKCALL_JOB_ID -l h_vmem=$DEFAULT_VMEM $IDRPREP $RUN_INFO`
	
	### 2.3 repcall.sh 
	### split bam file from each IP replicate into 2 same-sized pseudo reps, call peaks from each IP rep and also its two pseudo reps
	REPCALL_JOB_ID=`qsub $QSUBARGS -N $TOOL_VERSION.repcall.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $IDRPREP_JOB_ID -t 1-$t_array_IP -l h_vmem=$DEFAULT_VMEM $REPCALL $RUN_INFO`
	# Strip array task info off job id
	REPCALL_JOB_ID=`echo $REPCALL_JOB_ID | perl -pe "s/\..*//"`

	### 2.4 idr.sh
	IDR_TMP_ID=`qsub $QSUBARGS -N $TOOL_VERSION.idr.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $REPCALL_JOB_ID -l h_vmem=$DEFAULT_VMEM $IDR $RUN_INFO`
	 IDR_DONE_JOB_ID=`echo $IDR_TMP_ID | cut -d ' ' -f3 | cut -d '.' -f1`


else
	# In other cases, set an invalid job ID so we continue right away.
	IDR_DONE_JOB_ID=-1
fi

# If we are internal to Mayo, update the dashboard. Otherwise, skip this step.
if [[ $OPEN_SOURCE -eq "false" ]]
then
	# Regardless of the path used (sicer, macs2idr, or macs2noidr), wait for the tasks to complete.
	qsub $QSUBARGS -N $TOOL_VERSION.dashboard.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $IDR_DONE_JOB_ID,$PKCALL_JOB_ID -l h_vmem=$DASHBOARD_VMEM -b y $DASHBOARD -r $RUN_INFO -s Statistics -c 
fi

POST_ANNOTATION="$IDR_DONE_JOB_ID,$PKCALL_JOB_ID"

# If we are internal to Mayo, update the dashboard. Otherwise, skip this step.
if [[ ( $OPEN_SOURCE -eq "false" ) && ( $BOOL_CEAS == "yes" || $BOOL_MEME == "yes" || $BOOL_GOM == "yes" ) ]]
then
	qsub $QSUBARGS -N $TOOL_VERSION.dashboard.PN$PORTAL_PROJECT_NAME -hold_jid $IDR_DONE_JOB_ID,$PKCALL_JOB_ID -l h_vmem=$DASHBOARD_VMEM -b y $JAVA -Xmx1G -Xms512M  $DASHBOARD -r $RUN_INFO -s Annotation
fi

# Reset t_array_IP value to 2 if we are using IDR analysis.
if [[ $PEAK_CALLER == "macs2idr" ]] 
then
	t_array_IP=2
	echo "Warning: only the first two samples listed in runinfo.txt:IP_FILES will be used for IDR analysis and annotation."
fi
	

#### 3. Run Annotation Suite -> CEAS
if [[ $BOOL_CEAS == "yes" ]]
then
	echo -e "You have chosen to run CEAS Annotation."
	### 3.1 Run CEAS Annotation script, able to pick out naming scheme internally.
	ANNOTCEAS=`qsub $QSUBARGS -hold_jid $IDR_DONE_JOB_ID,$PKCALL_JOB_ID -N $TOOL_VERSION.annot_ceas.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -t 1-$t_array_IP -l h_vmem=$DEFAULT_VMEM $ANNOT_CEAS $RUN_INFO`
	ANNOTCEAS_ID=$(echo $ANNOTCEAS | cut -d ' ' -f3 | cut -d '.' -f1)
	POST_ANNOTATION=$POST_ANNOTATION","$ANNOTCEAS_ID
fi

#### 3. Run Annotation Suite -> MEME  
if [[ $BOOL_MEME == "yes" ]]
 then
	echo -e "You have chosen to run MEME Annotation."
	### 3.1 Run CEAS Annotation script, able to pick out naming scheme internally.
	ANNOTMEME=`qsub $QSUBARGS -hold_jid $IDR_DONE_JOB_ID,$PKCALL_JOB_ID -N $TOOL_VERSION.annot_meme.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -t 1-$t_array_IP -l h_vmem=$DEFAULT_VMEM $ANNOT_MEME $RUN_INFO`
	ANNOTMEME_ID=$(echo $ANNOTMEME | cut -d ' ' -f3 | cut -d '.' -f1)
	POST_ANNOTATION=$POST_ANNOTATION","$ANNOTMEME_ID
fi
 
#### 3. Run Annotation Suite -> Gene Ontology Module  
if [[ $BOOL_GOM == "yes" ]]
 then
	echo -e "You have chosen to run GOM Annotation."
	### 3.1 Run CEAS Annotation script, able to pick out naming scheme internally.
	ANNOTGOM=`qsub $QSUBARGS -hold_jid $IDR_DONE_JOB_ID,$PKCALL_JOB_ID -N $TOOL_VERSION.annot_gom.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -t 1-$t_array_IP -l h_vmem=$DEFAULT_VMEM $ANNOT_GOM $RUN_INFO`
	ANNOTGOM_ID=$(echo $ANNOTGOM | cut -d ' ' -f3 | cut -d '.' -f1)
	POST_ANNOTATION=$POST_ANNOTATION","$ANNOTGOM_ID
fi
 
# If we are internal to Mayo, update the dashboard. Otherwise, skip this step.
if [[ ( $OPEN_SOURCE -eq "false" ) && ( $BOOL_CEAS == "yes" || $BOOL_MEME == "yes" || $BOOL_GOM == "yes" ) ]]
then
	qsub $QSUBARGS -N $TOOL_VERSION.dashboard.PN$PORTAL_PROJECT_NAME -hold_jid $POST_ANNOTATION -l h_vmem=$DASHBOARD_VMEM -b y $JAVA -Xmx1G -Xms512M  $DASHBOARD -r $RUN_INFO -s Annotation -c
fi
 
### 4.1 delivery.sh
#DELIVERY_JOB_ID=`qsub $QSUBARGS -N $TOOL_VERSION.delivery.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $POST_ANNOTATION -l h_vmem=$DELIVERY_VMEM $DELIVERY $RUN_INFO`
DELIVERY_JOB_ID=`qsub $QSUBARGS -N $TOOL_VERSION.delivery.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $POST_ANNOTATION -l h_vmem=$DELIVERY_VMEM $DELIVERY $RUN_INFO`

if [[ $CLEAN_UP_FILES == true ]]
then
	### 5.1 cleanup.sh
	CLEANUP_JOB_ID=`qsub $QSUBARGS -N $TOOL_VERSION.cleanup.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $DELIVERY_JOB_ID -l h_vmem=$CLEANUP_VMEM $CLEANUP $RUN_INFO`
else
	# In other cases, set an invalid job ID so we continue right away.
	CLEANUP_JOB_ID=-1;
fi

# If we are internal to Mayo, update the dashboard. Otherwise, skip this step.
if [[ $OPEN_SOURCE -eq "false" ]]
then
	qsub $QSUBARGS -N $TOOL_VERSION.dashboard.$SAMPLENAME.PN$PORTAL_PROJECT_NAME -hold_jid $CLEANUP_JOB_ID,$DELIVERY_JOB_ID -l h_vmem=$DASHBOARD_VMEM -b y $DASHBOARD -r $RUN_INFO -s Complete -c 
fi

set +x
echo "The workflow is complete. When complete, the results will be available in $WORK_DIR"

echo `date`
