#!/bin/bash

############################################################################
## Description:
## Run FastQC on input fastqs
##
## Author: Jared Evans
## Date: 5/22/14
##
## Parameters:
## <input dir> - Directory where input fastqs are located
## <output_dir> - Directory where FastQC files should be written to
## <sample names(s1:s2)> - Colon seperated list of sample names
## <sample fastqs(fq1:fq2)> - Colon seperated list of fastq files for each sample
## <tool_info> - CAP-miRSeq tool_info.txt config file
## <sample_number> - Optional number indicating which sample to use
##
############################################################################


# determine whether this script is being called through Sun Grid Engine
if [ -z $SGE_TASK_ID ]; then 
	use_sge=0
else 
	use_sge=1
fi

set -e
set -u 

if [ $# != 5 -a "$use_sge" = "1" ]; then 
	echo "usage: <input dir> <output dir> <sample names(sn1:sn2)> <input fastq names (s1:s2)> <tool_info>";
elif [ $# != 6 -a "$use_sge" = "0" ] ; then 
	echo "usage: <input dir> <output dir> <sample names(sn1:sn2)> <input fastq names (s1:s2)> <tool_info> <sample_number>";
else 
	set -x
	echo `date`
	input_dir=$1
	output_dir=$2
	samples=$3
	fastqs=$4
	tool_info=$5


	# SGE passes this as part of an array job, but when run standalone the value needs to be passed on the command line
	if [ "$use_sge" = "1" ]; then
		sample_number=$SGE_TASK_ID
	else
		sample_number=$6
	fi

	fastqc_path=$( cat $tool_info | grep -w '^FASTQC_PATH' | cut -d '=' -f2)
	script_path=$( cat $tool_info | grep -w '^SCRIPT_PATH' | cut -d '=' -f2)

	sample=$( echo $samples | tr ":" "\n" | head -$sample_number | tail -1 )
	fastq=$( echo $fastqs | tr ":" "\n" | head -$sample_number | tail -1 )
	
	mkdir -p $output_dir/$sample
	
	#run fastqc
	$fastqc_path/fastqc -o $output_dir/$sample $input_dir/$fastq
	
	echo `date`
fi	

