/*
 * Decompiled with CFR 0.152.
 */
package edu.mayo.genomicutils.vcf.calc;

import com.google.gson.JsonElement;
import com.google.gson.JsonPrimitive;
import edu.mayo.genomicutils.Optimized;
import edu.mayo.genomicutils.vcf.calc.VariantCalculations;
import edu.mayo.genomicutils.vcf.header.HeaderFieldDefinition;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VariantCalculator {
    private static final Logger LOGGER = LoggerFactory.getLogger(VariantCalculator.class);
    private static final int VCF_COLUMN_FORMAT = 8;
    private Double configurableDoubleMin;
    private Double configurableDoubleMax;
    private Set<String> undefinedFormatFields = new HashSet<String>();
    private String[] sampleNames;
    private Map<String, HeaderFieldDefinition> formatMetaData = new HashMap<String, HeaderFieldDefinition>();

    public VariantCalculator(String[] sampleNames, Collection<HeaderFieldDefinition> formatDefs, Double configurableDoubleMin, Double configurableDoubleMax) {
        this.sampleNames = sampleNames;
        this.configurableDoubleMin = configurableDoubleMin;
        this.configurableDoubleMax = configurableDoubleMax;
        for (HeaderFieldDefinition formatDef : formatDefs) {
            this.formatMetaData.put(formatDef.id, formatDef);
        }
    }

    public VariantCalculations calculate(String[] vcfRow) throws ParseException {
        String formatColumn = vcfRow[8];
        String[] formatFields = Optimized.split(formatColumn, ':');
        int GTPosition = this.findGT(formatFields);
        int ADPosition = this.findAD(formatFields);
        for (String formatField : formatFields) {
            HeaderFieldDefinition metadata = this.formatMetaData.get(formatField);
            if (metadata != null) continue;
            metadata = this.getDefaultFormatMetadata(formatField);
            this.formatMetaData.put(formatField, metadata);
        }
        VariantCalculations calculations = new VariantCalculations(formatFields.length, this.configurableDoubleMin, this.configurableDoubleMax);
        for (int i = 9; i < vcfRow.length; ++i) {
            String sampleName = this.sampleNames[i - 8 - 1];
            String sampleValue = vcfRow[i];
            this.processSample(sampleName, sampleValue, formatFields, this.formatMetaData, GTPosition, ADPosition, calculations);
        }
        return calculations;
    }

    private void processSample(String sampleName, String sampleValue, String[] formatFields, Map<String, HeaderFieldDefinition> formatMetaData, int GTPosition, int ADPosition, VariantCalculations calculations) throws ParseException {
        String[] fieldValues = Optimized.split(sampleValue, ':');
        if (fieldValues.length > formatFields.length) {
            String message = "WARNING: VCF2VariantPipe.parseSample: the number of tokens in the format field (" + formatFields.length + ") and the number of tokens in the sample (" + fieldValues.length + ") do not agree. \nFORMAT:" + Arrays.toString(formatFields) + "\nSAMPLE: " + sampleName + "\n";
            LOGGER.warn(message);
            throw new ParseException(message, 0);
        }
        if (GTPosition > -1) {
            String gtValue = fieldValues[GTPosition];
            gtValue = gtValue.replace('|', '/');
            String[] tokens = Optimized.split(gtValue, '/');
            Zygocity zygocity = VariantCalculator.calculateZygosity(tokens);
            if (VariantCalculator.sampleHasVariant(tokens)) {
                ++calculations.GenotypePostitiveCount;
                calculations.GenotypePositiveSamples.add((JsonElement)new JsonPrimitive(sampleName));
                if (zygocity == Zygocity.Heterozygous) {
                    calculations.hetrozygus.add((JsonElement)new JsonPrimitive(sampleName));
                } else if (zygocity == Zygocity.Homozygous) {
                    calculations.homozygus.add((JsonElement)new JsonPrimitive(sampleName));
                }
            } else if (zygocity == Zygocity.WildType) {
                JsonPrimitive zy = new JsonPrimitive(sampleName);
                calculations.wildtype.add((JsonElement)zy);
            }
        }
        for (int i = 0; i < fieldValues.length; ++i) {
            String fieldName = formatFields[i];
            String fieldValue = fieldValues[i];
            HeaderFieldDefinition metadata = formatMetaData.get(fieldName);
            if (metadata.type.equals((Object)HeaderFieldDefinition.ValueType.Integer)) {
                if (metadata.isArray()) {
                    for (String arrayValue : Optimized.split(fieldValue, ',')) {
                        if (this.isMissingValue(arrayValue)) continue;
                        try {
                            int intValue = this.parseInt(arrayValue);
                            calculations.updateIntegerMinMax(i, intValue);
                        }
                        catch (NumberFormatException intValue) {
                            // empty catch block
                        }
                    }
                } else if (!this.isMissingValue(fieldValue)) {
                    try {
                        int intValue = this.parseInt(fieldValue);
                        calculations.updateIntegerMinMax(i, intValue);
                    }
                    catch (NumberFormatException intValue) {}
                }
            } else if (metadata.type.equals((Object)HeaderFieldDefinition.ValueType.Float)) {
                if (metadata.isArray()) {
                    for (String arrayValue : Optimized.split(fieldValue, ',')) {
                        if (this.isMissingValue(arrayValue)) continue;
                        try {
                            double doubleValue = this.parseDouble(arrayValue);
                            calculations.updateDoubleMinMax(i, doubleValue);
                        }
                        catch (NumberFormatException numberFormatException) {
                            // empty catch block
                        }
                    }
                } else if (!this.isMissingValue(fieldValue)) {
                    try {
                        double doubleValue = this.parseDouble(fieldValue);
                        calculations.updateDoubleMinMax(i, doubleValue);
                    }
                    catch (NumberFormatException doubleValue) {
                        // empty catch block
                    }
                }
            }
            if (ADPosition <= -1 || ADPosition != i) continue;
            String[] arrayValues = Optimized.split(fieldValue, ',');
            ArrayList<Double> doubles = new ArrayList<Double>();
            try {
                for (String value : arrayValues) {
                    if (this.isMissingValue(value)) continue;
                    doubles.add(this.parseDouble(value));
                }
                this.addMinMaxAD(calculations.totalAD, doubles);
                continue;
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
    }

    private HeaderFieldDefinition getDefaultFormatMetadata(String formatField) {
        HeaderFieldDefinition metadata = new HeaderFieldDefinition();
        metadata.id = formatField;
        metadata.fieldType = "FORMAT";
        if (formatField.equals("GT")) {
            metadata.desc = "Genotype";
            metadata.number = 1;
            metadata.type = HeaderFieldDefinition.ValueType.String;
        } else if (formatField.equals("DP")) {
            metadata.desc = "Read Depth";
            metadata.number = 1;
            metadata.type = HeaderFieldDefinition.ValueType.Integer;
        } else if (formatField.equals("FT")) {
            metadata.desc = "sample genotype filter indicating if this genotype was called";
            metadata.number = 1;
            metadata.type = HeaderFieldDefinition.ValueType.String;
        } else if (formatField.equals("GL")) {
            metadata.desc = "genotype likelihoods";
            metadata.number = null;
            metadata.type = HeaderFieldDefinition.ValueType.Float;
        } else if (formatField.equals("GLE")) {
            metadata.desc = "genotype likelihoods of heterogeneous ploidy, used in presence of uncertain copy number";
            metadata.number = null;
            metadata.type = HeaderFieldDefinition.ValueType.String;
        } else if (formatField.equals("PL")) {
            metadata.desc = "the phred-scaled genotype likelihoods rounded to the closest integer";
            metadata.number = null;
            metadata.type = HeaderFieldDefinition.ValueType.Integer;
        } else if (formatField.equals("GP")) {
            metadata.desc = "the phred-scaled genotype posterior probabilities;intended to store imputed genotype probabilities";
            metadata.number = null;
            metadata.type = HeaderFieldDefinition.ValueType.Float;
        } else if (formatField.equals("GQ")) {
            metadata.desc = "conditional genotype quality, encoded as a phred quality";
            metadata.number = 1;
            metadata.type = HeaderFieldDefinition.ValueType.Integer;
        } else if (formatField.equals("HQ")) {
            metadata.desc = "haplotype qualities, two comma separated phred qualities";
            metadata.number = null;
            metadata.type = HeaderFieldDefinition.ValueType.Integer;
        } else if (formatField.equals("PS")) {
            metadata.desc = "phase set. A phase set is defined as a set of phased genotypes to which this genotype belongs.";
            metadata.number = 1;
            metadata.type = HeaderFieldDefinition.ValueType.Integer;
        } else if (formatField.equals("PQ")) {
            metadata.desc = "phasing quality, the phred-scaled probability that alleles are ordered incorrectly in a heterozygote (against all other members in the phase set).";
            metadata.number = 1;
            metadata.type = HeaderFieldDefinition.ValueType.Integer;
        } else if (formatField.equals("EC")) {
            metadata.desc = "comma separated list of expected alternate allele counts for each alternate allele in the same order as listed in the ALT field (typically used in association analyses)";
            metadata.number = null;
            metadata.type = HeaderFieldDefinition.ValueType.Integer;
        } else if (formatField.equals("MQ")) {
            metadata.desc = "RMS mapping quality, similar to the version in the INFO field.";
            metadata.number = 1;
            metadata.type = HeaderFieldDefinition.ValueType.Integer;
        } else {
            metadata.desc = "not defined";
            metadata.number = null;
            metadata.type = HeaderFieldDefinition.ValueType.String;
            if (!this.undefinedFormatFields.contains(formatField)) {
                LOGGER.warn(String.format("WARNING: A ##FORMAT header line was not found for %s.  By default, this field will use ##FORMAT=<ID=%s,Number=.,Type=String,Description=\"unknown\">", formatField, formatField));
                this.undefinedFormatFields.add(formatField);
            }
        }
        return metadata;
    }

    private void addMinMaxAD(VariantCalculations.Totals totalAD, List<Double> values) {
        for (int i = 1; i < values.size(); ++i) {
            double d = values.get(i);
            if (d > totalAD.max) {
                totalAD.max = d;
            }
            if (!(d < totalAD.min)) continue;
            totalAD.min = d;
        }
    }

    private int findGT(String[] t) {
        return this.findT(t, "GT");
    }

    private int findAD(String[] t) {
        return this.findT(t, "AD");
    }

    private int findT(String[] t, String tok) {
        for (int i = 0; i < t.length; ++i) {
            if (!t[i].equalsIgnoreCase(tok)) continue;
            return i;
        }
        return -1;
    }

    protected static boolean sampleHasVariant(String[] tokens) {
        for (String token : tokens) {
            if (!token.equals("1") && !token.equals("2")) continue;
            return true;
        }
        return false;
    }

    private boolean isMissingValue(String value) {
        String trimVal = value.trim();
        return trimVal.equals(".");
    }

    private double parseDouble(String s) {
        if (s.equalsIgnoreCase("Infinity") || s.equalsIgnoreCase("nan") || s.equalsIgnoreCase("-nan")) {
            return this.configurableDoubleMax;
        }
        try {
            return Optimized.parseDouble(s);
        }
        catch (NumberFormatException nfe) {
            return Double.parseDouble(s);
        }
    }

    private int parseInt(String s) {
        if (s.equalsIgnoreCase("Infinity") || s.equalsIgnoreCase("nan") || s.equalsIgnoreCase("-nan")) {
            return Integer.MAX_VALUE;
        }
        return Optimized.parseInt(s);
    }

    public static Zygocity calculateZygosity(String[] tokens) {
        int zerocount = 0;
        int count = 0;
        for (String token : tokens) {
            String trimToken = token.trim();
            if (trimToken.length() == 0) continue;
            char c = trimToken.charAt(0);
            if (c == '0') {
                ++zerocount;
                continue;
            }
            if (!Character.isDigit(c)) continue;
            ++count;
        }
        if (count == 1) {
            return Zygocity.Heterozygous;
        }
        if (count > 1) {
            return Zygocity.Homozygous;
        }
        if (zerocount > 1) {
            return Zygocity.WildType;
        }
        return Zygocity.NoCall;
    }

    public static enum Zygocity {
        WildType,
        Heterozygous,
        Homozygous,
        NoCall;

    }
}

