/*
 * Decompiled with CFR 0.152.
 */
package edu.mayo.bior.catalog;

import edu.mayo.bior.catalog.CatalogFormatException;
import edu.mayo.bior.catalog.DataSourceKey;
import edu.mayo.bior.catalog.HumanBuildAssembly;
import edu.mayo.bior.catalog.verification.CatalogVerifier;
import edu.mayo.bior.catalog.verification.MessageLogger;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CatalogDataSource {
    private static final String SHORT_UNIQUE_NAME = DataSourceKey.ShortUniqueName.name();
    private static final String DESCRIPTION = DataSourceKey.Description.name();
    private static final String SOURCE = DataSourceKey.Source.name();
    private static final String DATASET = DataSourceKey.Dataset.name();
    private static final String VERSION = DataSourceKey.Version.name();
    private static final String BUILD = DataSourceKey.Build.name();
    private static final String DATA_SOURCE_RELEASE_DATE = DataSourceKey.DataSourceReleaseDate.name();
    private static final String FORMAT = DataSourceKey.Format.name();
    private static final List<String> REQ_PROP_NAMES_1_1_0 = new ArrayList<String>();
    private static final List<String> REQ_PROP_NAMES_1_1_1 = new ArrayList<String>();
    private static final List<String> REQ_PROP_NAMES_1_1_2 = new ArrayList<String>();
    private static final String FORMAT_1_1_0 = "1.1.0";
    private static final String FORMAT_1_1_1 = "1.1.1";
    private static final String FORMAT_1_1_2 = "1.1.2";
    private static final List<String> SUPPORTED_FORMATS;
    private Logger sLogger = LoggerFactory.getLogger(CatalogDataSource.class);
    private HumanBuildAssembly mHumanBuildValue = null;
    private File mDatasourceFile;
    private Map<String, String> mProps = new LinkedHashMap<String, String>();
    private static final String RELEASE_DATE_FORMAT = "yyyy-MM-dd";
    private static final SimpleDateFormat releaseDateFormatParser;

    public CatalogDataSource(File datasourceFile) throws CatalogFormatException {
        this.mDatasourceFile = datasourceFile;
        this.readProperties(datasourceFile);
        this.checkFormat(datasourceFile);
        this.checkRequiredProperties(datasourceFile);
    }

    private void readProperties(File datasourceFile) throws CatalogFormatException {
        if (datasourceFile == null) {
            throw new CatalogFormatException("Supplied null file to CatalogDataSource");
        }
        if (!datasourceFile.exists()) {
            throw new CatalogFormatException(String.format("'%s' does not exist", datasourceFile.getPath()));
        }
        if (!datasourceFile.isFile()) {
            throw new CatalogFormatException(String.format("'%s' is not a file", datasourceFile.getPath()));
        }
        if (!datasourceFile.canRead()) {
            throw new CatalogFormatException(String.format("'%s' is not readable", datasourceFile.getPath()));
        }
        BufferedReader rdr = null;
        try {
            rdr = new BufferedReader(new FileReader(datasourceFile.getPath()));
            Properties properties = new Properties();
            properties.load(rdr);
            ArrayList<String> keysSeenMultipleTimes = new ArrayList<String>();
            for (String key : properties.stringPropertyNames()) {
                String value = properties.getProperty(key);
                if (value.contains("\"")) {
                    this.sLogger.error("Error: value for key [" + key + "] within the datasource.properties file must not contain a double-quote.  These will be replaced with single-quotes.  File: " + datasourceFile.getPath());
                    value = value.replaceAll("\"", "'");
                }
                for (String requiredPropName : REQ_PROP_NAMES_1_1_2) {
                    if (!requiredPropName.equalsIgnoreCase(key)) continue;
                    if (this.mProps.containsKey(requiredPropName)) {
                        keysSeenMultipleTimes.add(requiredPropName);
                        continue;
                    }
                    this.mProps.put(requiredPropName, value);
                }
                if (!key.equals(BUILD) || value == null) continue;
                this.mHumanBuildValue = HumanBuildAssembly.assemblyFromString(value);
            }
            if (!keysSeenMultipleTimes.isEmpty()) {
                String keyMsg = "had multiple instances of key";
                if (keysSeenMultipleTimes.size() != 1) {
                    keyMsg = keyMsg + "s";
                }
                Collections.sort(keysSeenMultipleTimes);
                keyMsg = keyMsg + String.format(" '%s'", StringUtils.join(keysSeenMultipleTimes, (String)","));
                throw new CatalogFormatException(String.format("'%s' %s", datasourceFile.getPath(), keyMsg));
            }
        }
        catch (IOException e) {
            try {
                String msg = String.format("Problem reading data source properties file '%s'", datasourceFile.getPath());
                this.sLogger.error(msg);
                this.sLogger.error("Caused by: " + e.getMessage());
                throw new CatalogFormatException(msg, e);
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(rdr);
                throw throwable;
            }
        }
        IOUtils.closeQuietly((Reader)rdr);
    }

    private void checkFormat(File datasourceFile) throws CatalogFormatException {
        String format = this.getFormat();
        if (format == null) {
            String msg = String.format("Must supply %s property in '%s'", FORMAT, datasourceFile.getPath());
            throw new CatalogFormatException(msg);
        }
        if (!SUPPORTED_FORMATS.contains(format)) {
            String msg = String.format("%s value '%s' from '%s' not in supported set '%s'", FORMAT, format, datasourceFile.getPath(), StringUtils.join(SUPPORTED_FORMATS, (String)", "));
            throw new CatalogFormatException(msg);
        }
    }

    private void checkRequiredProperties(File datasourceFile) throws CatalogFormatException {
        List<String> requiredProperties;
        if (this.is110Format()) {
            requiredProperties = REQ_PROP_NAMES_1_1_0;
        } else if (this.is111Format()) {
            requiredProperties = REQ_PROP_NAMES_1_1_1;
        } else if (this.is112Format()) {
            requiredProperties = REQ_PROP_NAMES_1_1_2;
        } else {
            throw new RuntimeException("Programming error. Should have legal Format when checkRequiredProps is called.");
        }
        ArrayList<String> propertiesNotSupplied = new ArrayList<String>();
        for (String key : requiredProperties) {
            if (null != this.mProps.get(key)) continue;
            propertiesNotSupplied.add(key);
        }
        if (!propertiesNotSupplied.isEmpty()) {
            Collections.sort(propertiesNotSupplied);
            String propertyDescription = "property";
            if (propertiesNotSupplied.size() > 1) {
                propertyDescription = "properties";
            }
            String msg = String.format("Required datasource %s '%s' not supplied in '%s'", propertyDescription, StringUtils.join(propertiesNotSupplied, (String)","), datasourceFile.getPath());
            throw new CatalogFormatException(msg);
        }
    }

    public File getFile() {
        return this.mDatasourceFile;
    }

    public String getShortUniqueName() {
        return this.getPropValue(SHORT_UNIQUE_NAME);
    }

    public String getDescription() {
        return this.getPropValue(DESCRIPTION);
    }

    public String getSource() {
        return this.getPropValue(SOURCE);
    }

    public String getDataset() {
        return this.getPropValue(DATASET);
    }

    public String getVersion() {
        return this.getPropValue(VERSION);
    }

    public String getBuild() {
        return this.getPropValue(BUILD);
    }

    public String getDataSourceReleaseDate() {
        return this.getPropValue(DATA_SOURCE_RELEASE_DATE);
    }

    public Date getDataSourceReleaseDateAsDateObject() throws CatalogFormatException {
        try {
            return releaseDateFormatParser.parse(this.getDataSourceReleaseDate());
        }
        catch (ParseException pe) {
            String mesg = String.format("%s value '%s' is incorrectly formatted (%s) in file %s", DataSourceKey.DataSourceReleaseDate.name(), this.getDataSourceReleaseDate(), RELEASE_DATE_FORMAT, this.getFile().getAbsolutePath());
            throw new CatalogFormatException(mesg, pe);
        }
    }

    public String getFormat() {
        return this.getPropValue(FORMAT);
    }

    private String getPropValue(String key) {
        String value = this.mProps.get(key);
        if (value == null) {
            return null;
        }
        return value.trim();
    }

    public boolean is110Format() {
        return FORMAT_1_1_0.equals(this.getFormat());
    }

    public boolean is111Format() {
        return FORMAT_1_1_1.equals(this.getFormat());
    }

    public boolean is112Format() {
        return FORMAT_1_1_2.equals(this.getFormat());
    }

    public List<String> getPropertiesWithBlankValue() {
        ArrayList<String> propertiesWithBlankValue = new ArrayList<String>();
        for (Map.Entry<String, String> entrySet : this.mProps.entrySet()) {
            if (!StringUtils.isBlank((String)entrySet.getValue())) continue;
            propertiesWithBlankValue.add(entrySet.getKey());
        }
        Collections.sort(propertiesWithBlankValue);
        return propertiesWithBlankValue;
    }

    public void verify(CatalogVerifier.VAL_TYPE valType, MessageLogger logger) {
        List<String> propsWithBlankValue = this.getPropertiesWithBlankValue();
        ArrayList<String> keysToComplainAbout = new ArrayList<String>();
        if (propsWithBlankValue.contains(VERSION)) {
            keysToComplainAbout.add(VERSION);
        }
        if (valType == CatalogVerifier.VAL_TYPE.STRICT) {
            if (propsWithBlankValue.contains(BUILD)) {
                keysToComplainAbout.add(BUILD);
            }
            if (propsWithBlankValue.contains(DESCRIPTION)) {
                keysToComplainAbout.add(DESCRIPTION);
            }
        }
        if (!keysToComplainAbout.isEmpty()) {
            String description = "Property";
            if (keysToComplainAbout.size() > 1) {
                description = "Properties";
            }
            String msg = String.format("%s '%s' in '%s' should have a value", description, StringUtils.join(keysToComplainAbout, (String)","), this.mDatasourceFile.getPath());
            logger.logWarning(msg);
        }
        if (this.getHumanBuildAssembly() == null) {
            String msg = String.format("Human Genome Assembly was not determined from property '%s' in '%s'", BUILD, this.mDatasourceFile.getPath());
            logger.logWarning(msg);
        } else {
            logger.logInfo("Catalog is based on Human Genome Build '" + this.getHumanBuildAssembly().name() + "'. Will check chromosomes, positions, and reference alleles accordingly.");
        }
        String releaseDate = this.getDataSourceReleaseDate();
        try {
            if (releaseDate != null && !"".equals(releaseDate)) {
                releaseDateFormatParser.parse(this.getDataSourceReleaseDate());
            }
        }
        catch (ParseException e) {
            String msg = String.format("%s value '%s' is either incorrectly formatted (%s) or a date that does not exist. Details: %s", new Object[]{DataSourceKey.DataSourceReleaseDate, releaseDate, RELEASE_DATE_FORMAT, e.getMessage()});
            logger.logError(msg, 114);
        }
    }

    public HumanBuildAssembly getHumanBuildAssembly() {
        return this.mHumanBuildValue;
    }

    public boolean isHumanCatalog() {
        return this.mHumanBuildValue.equals((Object)HumanBuildAssembly.GRCh37) || this.mHumanBuildValue.equals((Object)HumanBuildAssembly.GRCh38);
    }

    static {
        REQ_PROP_NAMES_1_1_0.addAll(Arrays.asList(SHORT_UNIQUE_NAME, DESCRIPTION, SOURCE, VERSION, BUILD, FORMAT));
        REQ_PROP_NAMES_1_1_1.addAll(REQ_PROP_NAMES_1_1_0);
        REQ_PROP_NAMES_1_1_1.add(DATASET);
        REQ_PROP_NAMES_1_1_2.addAll(REQ_PROP_NAMES_1_1_1);
        REQ_PROP_NAMES_1_1_2.add(DATA_SOURCE_RELEASE_DATE);
        SUPPORTED_FORMATS = Arrays.asList(FORMAT_1_1_0, FORMAT_1_1_1, FORMAT_1_1_2);
        releaseDateFormatParser = new SimpleDateFormat(RELEASE_DATE_FORMAT);
        releaseDateFormatParser.setLenient(false);
    }
}

