#!/bin/bash

###====================================================================
### Build jars and deployable zip files for external deployment
###====================================================================

targetDir=$1
biorToolkitProjDir=$2
biorCatalogProjDir=$3
pipesProjDir=$4

function usage() {
  echo "Usage:  buildExternalDeployment.sh  <ExternalDistDir>  <biorToolkitProjDir>  <biorCatalogProjDir>  <pipesProjDir>"
  echo "Example: buildExternalDeployment.sh   $HOME/ExternalDistribution  $HOME/bior/bior-toolkit  $HOME/bior/bior_catalog  $HOME/pipes"
  echo "NOTE: Use full paths for all directories"
}


if [ -z "$targetDir" ] || [ ! -d "$targetDir" ]  ; then
  echo "Target directory for external distribution not specified (or does not exist)"
  usage
  exit 1
fi

if [ -z "$biorToolkitProjDir" ] || [ ! -d "$biorToolkitProjDir" ] ; then
  echo "bior-toolkit project directory not specified (or does not exist)"
  usage
  exit 1
fi

if [ -z "$biorCatalogProjDir" ] || [ ! -d "$biorCatalogProjDir" ]  ; then
  echo "bior_catalog project directory not specified (or does not exist)"
  echo "Will skip..."
  usage
fi

if [ -z "$pipesProjDir" ] || [ ! -d "$pipesProjDir" ]  ; then
  echo "pipes project directory not specified (or does not exist)"
  echo "Will skip..."
  usage
fi


# Get name and version (ex: "bior-toolkit_2.2.2-SNAPSHOT") from pom.xml in current dir
function getNameVersion() {
  dir=$1
  # echo "dir: [$dir]"

  # Find the project name in pom.xml
  # Ex: Given "<name>bior-toolkit</name>" in pom.xml, pull out "bior-toolkit" (remove the tags and any spaces/tabs)
  name=$(grep "<name>" $dir/pom.xml | head -1 | tr -d '\t' | tr -d ' ' | sed 's/<name>//g' | sed 's/<\/name>//g')
  # echo "name: [$name]"
  
  # Find "<version>2.2.2-SNAPSHOT</version>" in pom.xml and pull out "2.2.2-SNAPSHOT" (remove the tags and any spaces/tabs)
  version=$(grep "<version>" $dir/pom.xml | head -1 | tr -d '\t' | tr -d ' ' | sed 's/<version>//g' | sed 's/<\/version>//g')
  # echo "version: [$version]"
   
  # Construct name-version  (ex: "bior-toolkit_2.2.2-SNAPSHOT")
  # Use the $nameVersion variable after calling this function
  nameVersion="${name}-${version}"
  # echo "nameVersion: [$nameVersion]"
}




### =======================================================
### Build the external zip for the bior-toolkit project
### =======================================================

### NOTE: Make sure the google doc has been downloaded as PDF and checked into SVN before running this!!! 

getNameVersion $biorToolkitProjDir

cd $targetDir
unzip -q $biorToolkitProjDir/target/bior-toolkit-*-distribution.zip
oldDir=`echo bior-toolkit-*`
# Change from "bior-toolkit-2.2.2-SNAPSHOT" for instance to "bior_2.2.2-beta"
# newDir=`echo bior-toolkit-* | sed 's/bior-toolkit-/bior_/g' | sed 's/SNAPSHOT/beta/g'`
newDir=`echo bior-toolkit-* |  sed 's/SNAPSHOT/beta/g'`
mv $oldDir $newDir
cd $newDir

# Change the sage "dev" to "prod"
# How to do this in one command line rather than interactively
sed -i 's/SAGE_SERVER=.*$/SAGE_SERVER=DISABLE/g' conf/cli/Global.properties 

mv conf/bior.properties.external conf/bior.properties
cp $biorToolkitProjDir/setupEnv.sh.external ./setupEnv.sh


# Copy all extra files to this directory  (do NOT copy the .svn/ subdirectories)
# use the -v flag with rsync to see what is copied
echo "Copy all doc, examples, license, scripts to temp dir"
rsync -a  $biorToolkitProjDir/doc      . --exclude .svn  --exclude .DS_Store
rsync -a  $biorToolkitProjDir/examples . --exclude .svn
rsync -a  $biorToolkitProjDir/license  . --exclude .svn
rsync -a  $biorToolkitProjDir/scripts  . --exclude .svn
cp $biorToolkitProjDir/RELEASE_NOTES.md .
cp $biorToolkitProjDir/README.txt .
cp $biorToolkitProjDir/License.txt .


## Turn off the tar'ing of ._xxxxxx files
export COPYFILE_DISABLE=true
cd ..
echo "Tar the bior-toolkit deployment directory..."
tar -zcf ${newDir}.tar.gz  $newDir

echo "Remove the temp directory (MAKE SURE IT IS 10 chars or LONGER!!!)..."
len=${#newDir}
if [ $len -ge 10 ]; then
  echo "Yes it is >= 10 chars  (len=[$len])"
  echo "Removing dir: $newDir"
  rm -rf "$newDir"
fi


### =======================================================
### Tar-Gzip all the source code directories
### =======================================================
biorToolkitProjDir=$2
biorCatalogProjDir=$3
pipesProjDir=$4

echo "Tar the bior-toolkit source dir..."
dir=$(basename $biorToolkitProjDir)
cd $biorToolkitProjDir/..
# Use the --transform flag to rename the directory from our LPEA one to a more palatable name
#   --transform s/LPEA_GSU_bior_toolkit/bior-toolkit/
tar  --exclude="$dir/Dist" --exclude="$dir/target" --exclude="$dir/.svn" --exclude="$dir/.git*" --exclude="$dir/.settings" \
   --transform "s/^$dir/bior-toolkit/" \
   -zcf  $targetDir/${nameVersion}.source.tar.gz \
   $dir

if [ -z "$biorCatalogProjDir" ] ; then
  echo "Skipping source code build for: catalog building project"
else
  echo "Tar the bior_catalog source dir..."
  getNameVersion $biorCatalogProjDir
  dir=$(basename $biorCatalogProjDir)
  cd $biorCatalogProjDir/..
  tar  --exclude="$dir/DataFiles" --exclude="$dir/target"  --exclude="$dir/.svn" --exclude="$dir/.git*" --exclude="$dir/.settings" \
       -zcf  $targetDir/${nameVersion}.source.tar.gz  \
       bior_catalog
fi

if [ -z "$pipesProjDir" ] ; then
  echo "Skipping source code build for: catalog building project"
else  
  echo "Tar the pipes source dir..."
  getNameVersion $pipesProjDir
  dir=$(basename $pipesProjDir)
  cd $pipesProjDir/..
  tar  --exclude="$dir/target"  --exclude="$dir/.svn" --exclude="$dir/.git*" --exclude="$dir/.settings" \
       -zcf  $targetDir/${nameVersion}.source.tar.gz \
       pipes
fi

echo ""
echo "Output files available here:"
ls -la $targetDir/*.tar.gz
echo "Done."
