/*
 * Decompiled with CFR 0.152.
 */
package edu.mayo.bior.catalog.verification;

import com.google.gson.JsonObject;
import edu.mayo.bior.catalog.GoldenAttribute;
import edu.mayo.bior.catalog.verification.CatalogEntryGoldenJson;
import edu.mayo.bior.catalog.verification.HumanReferenceInfo;
import edu.mayo.bior.catalog.verification.MessageLogger;
import edu.mayo.bior.catalog.verification.SeqLookup;
import edu.mayo.bior.catalog.verification.VerifyUtils;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

public class CatalogVariantVerifier {
    private static final Logger sLogger = Logger.getLogger(CatalogVariantVerifier.class);
    public static final String UNKNOWN_CHR = "UNKNOWN";
    private SeqLookup mSequenceLookup = null;
    private MessageLogger mLogger;
    private long mNumberRefAllelesMatch = 0L;
    private long mNumberRefAllelesNotMatch = 0L;
    private static final String GOLDEN_REF_ALLELE = GoldenAttribute._refAllele.name();
    private static final String GOLDEN_ALT_ALLELES = GoldenAttribute._altAlleles.name();
    private static final String GOLDEN_STRAND = GoldenAttribute._strand.name();
    private static final Pattern DNA_NUCLEOTIDES = Pattern.compile("^[ACGTN]{1,}+$");

    public CatalogVariantVerifier(HumanReferenceInfo humanReferenceInfo, String chromosomeRefSeqToLoad, MessageLogger logger) {
        this.mLogger = logger;
        if (UNKNOWN_CHR.equals(chromosomeRefSeqToLoad)) {
            this.logInfo("Will not verify reference allele values for chromosome UNKNOWN");
            return;
        }
        try {
            if (humanReferenceInfo != null && humanReferenceInfo.getChrSizeMap() != null && humanReferenceInfo.getRefseqFileNm() != null) {
                this.mSequenceLookup = new SeqLookup(humanReferenceInfo.getRefseqFileNm());
            }
        }
        catch (IOException e) {
            String msg = "Unable to successfully initialize reference sequence lookup utility for chromosome '" + chromosomeRefSeqToLoad + "'. Will not be verifying reference allele values.  Exception: " + e.getMessage();
            this.logWarning(msg);
        }
    }

    public void verify(JsonObject catalogRowJson, CatalogEntryGoldenJson goldenJsonVariant) throws NumberFormatException, IOException {
        if (!this.isVariant(goldenJsonVariant)) {
            return;
        }
        String refAllele = goldenJsonVariant.getRefAllele();
        this.verifyGoldenJsonRefAllele(refAllele, catalogRowJson);
        this.verifyGoldenJsonAltAllele(goldenJsonVariant.getAltAlleles(), catalogRowJson, refAllele);
        if (StringUtils.isBlank((String)refAllele)) {
            this.logError(GOLDEN_REF_ALLELE + " value is empty for variant for chr: " + goldenJsonVariant.getChr() + " position: " + goldenJsonVariant.getMinBP(), 600);
        } else if (goldenJsonVariant.getMinBP() != null && goldenJsonVariant.getMaxBP() != null) {
            long jsonGoldenVariantLength = goldenJsonVariant.getMaxBP() - goldenJsonVariant.getMinBP() + 1L;
            if ((long)refAllele.length() != jsonGoldenVariantLength) {
                String msg = String.format("%s length [%d] is not equal to calculated length [%d]. Json: '%s'", GOLDEN_REF_ALLELE, refAllele.length(), jsonGoldenVariantLength, catalogRowJson.toString());
                this.logError(msg, 601);
            }
            if (this.mSequenceLookup != null) {
                this.verifyRefAlleleMatchesRefAssembly(goldenJsonVariant, refAllele);
            }
        }
    }

    private void verifyRefAlleleMatchesRefAssembly(CatalogEntryGoldenJson goldenJsonVariant, String refAllele) {
        String refFromRefAssembly = null;
        try {
            refFromRefAssembly = this.mSequenceLookup.getRefSeq(goldenJsonVariant.getChr(), goldenJsonVariant.getMinBP(), goldenJsonVariant.getMaxBP());
        }
        catch (Exception e) {
            this.logError(e.getMessage(), 500);
        }
        if (!VerifyUtils.isEmpty(refFromRefAssembly)) {
            if (!this.isRefAlleleMatch(refAllele, refFromRefAssembly, goldenJsonVariant.getChr(), goldenJsonVariant.getMinBP(), goldenJsonVariant.getMaxBP())) {
                ++this.mNumberRefAllelesNotMatch;
                int code = 501;
                if (VerifyUtils.isChrM(goldenJsonVariant.getChr())) {
                    code = 502;
                }
                this.logError(GOLDEN_REF_ALLELE + " [" + refAllele + "] does not match reference sequence value [" + refFromRefAssembly + "] for:  chr=" + goldenJsonVariant.getChr() + " position=" + goldenJsonVariant.getMinBP(), code);
            } else {
                ++this.mNumberRefAllelesMatch;
            }
        } else {
            this.logError("Allele lookup against reference sequence returned empty result for chr=" + goldenJsonVariant.getChr() + " minBP=" + goldenJsonVariant.getMinBP() + " maxBP=" + goldenJsonVariant.getMaxBP(), 503);
            ++this.mNumberRefAllelesNotMatch;
        }
    }

    private boolean isRefAlleleMatch(String refAllele, String refFromRefAssembly, String chr, long min, long max) {
        if (refAllele.toUpperCase().contains("N") || refFromRefAssembly.toUpperCase().contains("N")) {
            this.logWarning("An 'N' was detected in one of the reference sequences, which may signify a misalignment of the ref allele to the reference assembly.  (Ref='" + refAllele + "',  refAssemblySequence='" + refFromRefAssembly + "')  " + chr + ":" + min + "-" + max);
        }
        if (refAllele.equalsIgnoreCase(refFromRefAssembly)) {
            return true;
        }
        if (refAllele.length() != refFromRefAssembly.length()) {
            return false;
        }
        boolean areAllMatches = true;
        for (int i = 0; i < refAllele.length(); ++i) {
            boolean isChrMatch;
            boolean isAnN = refAllele.charAt(i) == 'N' || refFromRefAssembly.charAt(i) == 'N';
            boolean bl = isChrMatch = refAllele.charAt(i) == refFromRefAssembly.charAt(i);
            if (isAnN || isChrMatch) continue;
            areAllMatches = false;
        }
        return areAllMatches;
    }

    public long getNumberRefAllelesMatchReferenceSequence() {
        return this.mNumberRefAllelesMatch;
    }

    public long getNumberRefAllelesNotMatchReferenceSequence() {
        return this.mNumberRefAllelesNotMatch;
    }

    private boolean isVariant(CatalogEntryGoldenJson goldenJson) {
        return goldenJson.isVariant();
    }

    private void verifyGoldenJsonRefAllele(String goldenRefAllele, JsonObject catalogRowJson) {
        if (!CatalogVariantVerifier.validDNANucleotides(goldenRefAllele)) {
            this.logError(GOLDEN_REF_ALLELE + " [" + goldenRefAllele + "] contains an invalid nucleotide value. Json: " + catalogRowJson.toString(), 602);
        }
        String ref = null;
        if (catalogRowJson.get("REF") != null) {
            ref = catalogRowJson.get("REF").getAsString();
        } else if (catalogRowJson.get("ref") != null) {
            ref = catalogRowJson.get("ref").getAsString();
        }
        if (ref != null && !ref.equals(goldenRefAllele)) {
            this.logWarning(GOLDEN_REF_ALLELE + " [" + goldenRefAllele + "] is not equal to the REF value [" + ref + "]");
        }
    }

    private void verifyGoldenJsonAltAllele(List<String> altAlleleList, JsonObject catalogRowJson, String goldenRefAllele) {
        if (altAlleleList == null) {
            return;
        }
        HashSet<String> altAlleleSet = new HashSet<String>();
        for (String eachAltAllele : altAlleleList) {
            if (eachAltAllele.length() == 0) {
                this.logError("One of the " + GOLDEN_ALT_ALLELES + " values [" + eachAltAllele + "] is empty value. json: " + catalogRowJson.toString(), 603);
                continue;
            }
            if (!CatalogVariantVerifier.validDNANucleotides(eachAltAllele)) {
                this.logError("One of the " + GOLDEN_ALT_ALLELES + " [" + eachAltAllele + "] contains an invalid nucleotide value. Json: " + catalogRowJson.toString(), 604);
            }
            if (goldenRefAllele != null && goldenRefAllele.equals(eachAltAllele)) {
                this.logError("One of the " + GOLDEN_ALT_ALLELES + " [" + eachAltAllele + "] is equal to the " + GOLDEN_REF_ALLELE + ". Json: " + catalogRowJson.toString(), 605);
            }
            if (altAlleleSet.add(eachAltAllele)) continue;
            this.logWarning("One of the " + GOLDEN_ALT_ALLELES + " values [" + eachAltAllele + "] is repeated multiple times. Json: " + catalogRowJson.toString());
        }
    }

    static boolean validDNANucleotides(String dnaSeq) {
        return DNA_NUCLEOTIDES.matcher(dnaSeq).find();
    }

    private void logInfo(String msg) {
        this.mLogger.logInfo(msg);
    }

    private void logWarning(String msg) {
        this.mLogger.logWarning(msg);
    }

    private void logError(String msg, int code) {
        this.mLogger.logError(msg, code);
    }
}

