/*
 * Decompiled with CFR 0.152.
 */
package edu.mayo.bior.catalog.verification;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;
import com.google.gson.stream.MalformedJsonException;
import edu.mayo.bior.catalog.verification.MessageLogger;
import edu.mayo.pipes.history.ColumnMetaData;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;

public class CatalogJsonVerifier {
    private Map<String, ColumnMetaData> mColumnInfo;
    private MessageLogger mLogger;
    private JsonParser mParser = new JsonParser();
    private List<Error> mJsonErrors = new ArrayList<Error>();
    private List<String> mJsonWarnings = new ArrayList<String>();
    private List<String> mCatalogColsNotFoundInSomeJsonRows = new ArrayList<String>();
    private Set<String> mJsonKeysNotFoundInColumnsTsv = new HashSet<String>();
    private Map<String, Long> mNumberOfTimesKeySeenInCatalog = new HashMap<String, Long>();
    private Map<String, Long> mMaxValuesForKey = new HashMap<String, Long>();

    public CatalogJsonVerifier(Map<String, ColumnMetaData> columnInfo, MessageLogger logger) {
        this.mColumnInfo = columnInfo;
        this.mLogger = logger;
        if (!this.haveColumnInfo()) {
            this.logError("Unable to check json has valid keys because we have no column info from columns.tsv.", 401);
        }
    }

    protected void _test_setInitial_numberOfTimesKeySeenInCatalog(String key, long num) {
        this.mNumberOfTimesKeySeenInCatalog.put(key, num);
    }

    protected void _test_setInitial_maxValuesForKey(String key, long num) {
        this.mMaxValuesForKey.put(key, num);
    }

    public void verify(String json) throws JsonParseException {
        JsonObject catalogRowJsonObj;
        try {
            catalogRowJsonObj = this.mParser.parse(json).getAsJsonObject();
        }
        catch (Exception e) {
            this.logError(String.format("Trouble parsing '%s'. Msg: '%s'", json, e.getMessage()), 400);
            return;
        }
        this.mJsonErrors.clear();
        this.mJsonWarnings.clear();
        boolean keysAndQuotingValid = this.traverseJsonForInvalidJson(json);
        this.addErrorToListIfQuotingNotValid(keysAndQuotingValid);
        if (this.haveColumnInfo()) {
            List<String> columnsNotFoundInJson = this.getColumnTsvFieldsNotFoundInJson(catalogRowJsonObj);
            this.addColumnsNotFoundToKeyList(columnsNotFoundInJson);
            Set<String> jsonKeysNotFound = this.getJsonKeysNotInColumnsTsv(catalogRowJsonObj);
            this.addJsonKeysNotFoundInColumnsTsv_toKeyListAndErrorList(jsonKeysNotFound);
        }
        this.logAnyJsonWarnings(this.mJsonWarnings, json);
        this.logAnyJsonErrors(this.mJsonErrors, json);
    }

    private void addErrorToListIfQuotingNotValid(boolean keysAndQuotingValid) {
        if (!keysAndQuotingValid) {
            String msg = "If there aren't other obvious problems, look for unquoted keys/alphanumeric values or single quoted keys/values";
            this.mJsonErrors.add(new Error(msg, 404));
        }
    }

    private void addColumnsNotFoundToKeyList(List<String> columnsNotFoundInJson) {
        for (String jsonKey : columnsNotFoundInJson) {
            if (this.mCatalogColsNotFoundInSomeJsonRows.contains(jsonKey)) continue;
            this.mCatalogColsNotFoundInSomeJsonRows.add(jsonKey);
        }
    }

    private void addJsonKeysNotFoundInColumnsTsv_toKeyListAndErrorList(Set<String> jsonKeysNotFound) {
        this.mJsonKeysNotFoundInColumnsTsv.addAll(jsonKeysNotFound);
        if (!jsonKeysNotFound.isEmpty()) {
            ArrayList<String> sortedJsonKeys = new ArrayList<String>(jsonKeysNotFound);
            Collections.sort(sortedJsonKeys);
            String msg = String.format("Keys in json not found in columns.tsv: '%s'", StringUtils.join(sortedJsonKeys, (String)","));
            this.mJsonErrors.add(new Error(msg, 405));
        }
    }

    private void logAnyJsonWarnings(List<String> jsonWarnings, String json) {
        if (!jsonWarnings.isEmpty()) {
            this.logInfo(String.format("Found the following warnings parsing this json '%s'", json));
            for (String warning : jsonWarnings) {
                this.logWarning(warning);
            }
        }
    }

    private void logAnyJsonErrors(List<Error> jsonErrors, String json) {
        if (!jsonErrors.isEmpty()) {
            int MAX = 1000;
            if (json.length() > 1000) {
                this.logInfo(String.format("Found the following issues parsing this json (1st %d chrs): '%s' ....", MAX, json.substring(0, MAX)));
            } else {
                this.logInfo(String.format("Found the following issues parsing this json '%s'", json));
            }
            for (Error error : jsonErrors) {
                this.logError(error.getMsg(), error.getCode());
            }
        }
    }

    private boolean haveColumnInfo() {
        return this.mColumnInfo != null && this.mColumnInfo.size() > 0;
    }

    public Set<String> keysNotFoundInColumnInfo() {
        return this.mJsonKeysNotFoundInColumnsTsv;
    }

    public long getNumberOfTimesKeySeenInCatalog(String key) {
        Long numTimesSeen = this.mNumberOfTimesKeySeenInCatalog.get(key);
        if (numTimesSeen == null) {
            return 0L;
        }
        return numTimesSeen;
    }

    public long getMaxValuesForKey(String key) {
        Long maxValues = this.mMaxValuesForKey.get(key);
        if (maxValues == null) {
            return 0L;
        }
        return maxValues;
    }

    private List<String> getColumnTsvFieldsNotFoundInJson(JsonObject catalogRowJsonObj) {
        ArrayList<String> columnsNotFoundInJson = new ArrayList<String>();
        for (String colName : this.mColumnInfo.keySet()) {
            Object jsonObjFound = this.getJsonValueForCatalogKey(colName, catalogRowJsonObj, this.mColumnInfo.get(colName));
            if (jsonObjFound == null) {
                columnsNotFoundInJson.add(colName);
                continue;
            }
            this.incrementNumberOfTimesKeySeenInCatalog(this.mColumnInfo.get(colName).getColumnName());
        }
        return columnsNotFoundInJson;
    }

    private Set<String> getJsonKeysNotInColumnsTsv(JsonObject catalogRowJsonObj) {
        return this.getJsonKeysNotInColumnsTsv(catalogRowJsonObj, null);
    }

    private Set<String> getJsonKeysNotInColumnsTsv(JsonObject catalogRowJsonObj, String parentJsonKeyName) {
        HashSet<String> jsonKeysNotInColumnInfo = new HashSet<String>();
        for (Map.Entry thisEntry : catalogRowJsonObj.entrySet()) {
            JsonElement thisJsonElement;
            String thisJsonKey = (String)thisEntry.getKey();
            Object currentJsonObj = this.getJavaObjForJSON(thisJsonKey, thisJsonElement = (JsonElement)thisEntry.getValue(), jsonKeysNotInColumnInfo);
            if (currentJsonObj == null) {
                this.addToWarningsIfNonJsonObject(thisJsonElement, thisJsonKey);
                continue;
            }
            String key = parentJsonKeyName != null ? parentJsonKeyName + "." + thisJsonKey : thisJsonKey;
            this.addWarningsIfNotFoundOrTypeMismatch(key, currentJsonObj, jsonKeysNotInColumnInfo);
        }
        return jsonKeysNotInColumnInfo;
    }

    private void addWarningsIfNotFoundOrTypeMismatch(String jsonKey, Object currentJsonObj, Set<String> jsonKeysNotInColumnInfo) {
        ColumnMetaData colMeta = this.mColumnInfo.get(jsonKey);
        if (colMeta == null) {
            jsonKeysNotInColumnInfo.add(jsonKey);
            this.mJsonWarnings.add(String.format("Catalog key '%s' not found in columns.tsv. Cannot check data type.", jsonKey));
        } else if (!this.dataTypesMatch(currentJsonObj, colMeta)) {
            String msg = "Data type does not match between json [" + currentJsonObj.getClass().getSimpleName() + "] and columns.tsv [" + colMeta.getType() + "] for key: " + colMeta.getColumnName();
            this.mJsonErrors.add(new Error(msg, 406));
        }
    }

    private void addToWarningsIfNonJsonObject(JsonElement thisJsonElement, String thisJsonKey) {
        if (!thisJsonElement.isJsonObject()) {
            String msg = String.format("Java Object could not be created for Json key '%s' and value '%s'.", thisJsonKey, thisJsonElement.toString());
            this.mJsonWarnings.add(msg);
        }
    }

    private Object getJavaObjForJSON(String jsonKey, JsonElement jsonElement, Set<String> jsonKeysNotInColumnInfoMap) {
        Object currentJsonObj = null;
        if (jsonElement.isJsonNull()) {
            return null;
        }
        if (jsonElement.isJsonObject()) {
            Set<String> jsonKeysNotFoundInCols = this.getJsonKeysNotInColumnsTsv(jsonElement.getAsJsonObject(), jsonKey);
            jsonKeysNotInColumnInfoMap.addAll(jsonKeysNotFoundInCols);
            currentJsonObj = null;
        } else if (jsonElement.isJsonArray()) {
            currentJsonObj = jsonElement.getAsJsonArray();
        } else if (jsonElement.isJsonPrimitive()) {
            String strVal;
            JsonPrimitive jsonPrimVal = jsonElement.getAsJsonPrimitive();
            if (jsonPrimVal.isBoolean()) {
                Boolean boolVal = jsonPrimVal.getAsBoolean();
                currentJsonObj = boolVal;
            } else if (jsonPrimVal.isNumber()) {
                currentJsonObj = this.jsonNumberToJavaNumber(jsonPrimVal);
            } else if (jsonPrimVal.isString() && (strVal = jsonPrimVal.getAsString()) != null) {
                currentJsonObj = strVal;
            }
        } else {
            this.mJsonErrors.add(new Error("traverseJsonByJsonKeyset(): haven't found a json datatype for json key: " + jsonKey, 407));
            currentJsonObj = null;
        }
        return currentJsonObj;
    }

    private boolean traverseJsonForInvalidJson(String json) {
        JsonReader jsonRdrFromTabix;
        try {
            jsonRdrFromTabix = new JsonReader((Reader)new InputStreamReader(new ByteArrayInputStream(json.getBytes("UTF-8"))));
        }
        catch (UnsupportedEncodingException e) {
            this.logError("Couldn't get JsonReader to verify json " + json, 403);
            return true;
        }
        boolean isAllJsonValid = this.traverseJsonForInvalidJson(jsonRdrFromTabix);
        IOUtils.closeQuietly((Closeable)jsonRdrFromTabix);
        return isAllJsonValid;
    }

    private boolean traverseJsonForInvalidJson(JsonReader jsonRdrFromTabix) {
        boolean isAllJsonValid = true;
        String currentPropName = null;
        try {
            if (jsonRdrFromTabix.isLenient()) {
                String msg = "Json Reader is set to lenient in traverseJsonToCheckForInvalidJson(). Verification needs the reader set to strict checking for this method to perform checks adequately.";
                jsonRdrFromTabix.close();
                throw new RuntimeException(msg);
            }
            jsonRdrFromTabix.beginObject();
            while (jsonRdrFromTabix.hasNext()) {
                currentPropName = jsonRdrFromTabix.nextName();
                if (currentPropName.contains(".")) {
                    String msg = String.format("Not allowed to have '.' in json key '%s'", currentPropName);
                    this.mJsonErrors.add(new Error(msg, 408));
                    isAllJsonValid = false;
                }
                if (jsonRdrFromTabix.peek() == JsonToken.NAME) continue;
                Object value = this.readNextJsonTokenValue(jsonRdrFromTabix, currentPropName);
                if (value == null) {
                    this.mJsonErrors.add(new Error("Did not return a json value for json property name: " + currentPropName, 409));
                    isAllJsonValid = false;
                    continue;
                }
                if (!this.isValueArrayAndContainsPipe(value)) continue;
                this.mJsonErrors.add(new Error("Detected a pipe '|' character in a JSON Array value.  This is dangerous since pipe characters are used for the delimiter when drilling out array values.  This will cause an error when drilling out this key.  " + currentPropName + ":" + value, 420));
            }
            jsonRdrFromTabix.endObject();
        }
        catch (MalformedJsonException mal) {
            this.logJsonParsingErrorMessage(currentPropName, (Exception)((Object)mal));
            isAllJsonValid = false;
        }
        catch (IOException io) {
            this.logJsonParsingErrorMessage(currentPropName, io);
            isAllJsonValid = false;
        }
        return isAllJsonValid;
    }

    private boolean isValueArrayAndContainsPipe(Object value) {
        if (!this.isNonEmptyStringArrayList(value)) {
            return false;
        }
        ArrayList valueList = (ArrayList)value;
        for (String valStr : valueList) {
            if (!valStr.contains("|")) continue;
            return true;
        }
        return false;
    }

    private boolean isNonEmptyStringArrayList(Object value) {
        return value != null && value instanceof ArrayList && ((ArrayList)value).size() > 0 && ((ArrayList)value).get(0) instanceof String;
    }

    private void logJsonParsingErrorMessage(String propertyName, Exception e) {
        String msg = propertyName != null ? String.format("Problem parsing json property '%s'. Msg: '%s'", propertyName, e.getMessage()) : String.format("Problem parsing json. Msg: '%s'", e.getMessage());
        this.mJsonErrors.add(new Error(msg, 410));
    }

    private Object getJsonValueForCatalogKey(String columnName, JsonObject jsonObj, ColumnMetaData catalogCol) {
        String[] columnNameElems;
        if (!columnName.contains(".")) {
            columnNameElems = new String[]{columnName};
        } else {
            columnNameElems = columnName.split("\\.");
            if (columnNameElems.length < 1) {
                String msg = String.format("Bad column name format. Less than one element: '%s'", columnName);
                this.mJsonErrors.add(new Error(msg, 411));
                return null;
            }
        }
        Object currentJsonObj = jsonObj;
        Iterator<String> columnNameElemItr = Arrays.asList(columnNameElems).iterator();
        while (columnNameElemItr.hasNext()) {
            String colElem = columnNameElemItr.next();
            JsonElement jsonElem = null;
            if (currentJsonObj instanceof JsonObject) {
                JsonObject currentJson = currentJsonObj;
                jsonElem = currentJson.get(colElem);
            }
            if (jsonElem == null) {
                return null;
            }
            if (jsonElem.isJsonObject()) {
                JsonObject embeddedJsonObj = jsonElem.getAsJsonObject();
                if (embeddedJsonObj == null) {
                    this.mJsonWarnings.add("Json object retrieval was null for column key '" + columnName + "' and column value '" + colElem + "'.");
                    return null;
                }
                if (!columnNameElemItr.hasNext()) {
                    return embeddedJsonObj;
                }
                String nextColNameElement = columnNameElemItr.next();
                currentJsonObj = this.getJsonValueForCatalogKey(nextColNameElement, embeddedJsonObj, catalogCol);
            } else if (jsonElem.isJsonArray()) {
                JsonArray jsonArr = jsonElem.getAsJsonArray();
                currentJsonObj = jsonArr;
                this.setMaxValuesForKeyIfGreater(catalogCol.getColumnName(), jsonArr.size());
            } else if (jsonElem.isJsonPrimitive()) {
                JsonPrimitive jsonPrim = jsonElem.getAsJsonPrimitive();
                if (jsonPrim.isJsonNull()) {
                    currentJsonObj = jsonPrim;
                } else if (jsonPrim.isBoolean()) {
                    currentJsonObj = jsonPrim.getAsBoolean();
                } else if (jsonPrim.isNumber()) {
                    String numStr = jsonPrim.getAsString();
                    currentJsonObj = !this.isValidInteger(numStr, catalogCol.getType()) ? (Number)jsonPrim.getAsDouble() : (Number)jsonPrim.getAsInt();
                } else if (jsonPrim.isString()) {
                    currentJsonObj = jsonPrim.getAsString();
                    String strVal = (String)currentJsonObj;
                    if (!this.isQuotedString(strVal)) {
                        this.mJsonErrors.add(new Error("Invalid json formatting: json string value '" + strVal + "' is not quoted.", 412));
                    }
                } else {
                    if (jsonPrim.isJsonObject()) {
                        throw new RuntimeException("ERROR: getJsonForCatalogColumn(): column element name: " + colElem + " WARNING: Embedded JsonObject is Json primitive but says it is a json object too. Not expecting this.");
                    }
                    if (jsonPrim.isJsonArray()) {
                        throw new RuntimeException("ERROR: getJsonForCatalogColumn(): column element name: " + colElem + " WARNING: Embedded JsonObject is Json primitive but says it is a json array too. Not expecting this.");
                    }
                }
            }
            if (currentJsonObj != null && columnNameElemItr.hasNext()) continue;
            return currentJsonObj;
        }
        return null;
    }

    private Object readNextJsonTokenValue(JsonReader rdr, String currentPropName) throws IOException {
        Serializable jsonValue = null;
        JsonToken jsonToken = rdr.peek();
        if (jsonToken == JsonToken.NULL) {
            String msg = String.format("JSON null for key '%s'. This causes issues with pre 4.2 commands. Please leave '%s' out when the value is null.", currentPropName, currentPropName);
            this.mJsonErrors.add(new Error(msg, 413));
            rdr.skipValue();
        } else if (jsonToken == JsonToken.BEGIN_ARRAY) {
            ArrayList<Object> arrayElements = new ArrayList<Object>();
            int idx = 0;
            rdr.beginArray();
            while (rdr.hasNext()) {
                arrayElements.add(this.readNextJsonTokenValue(rdr, currentPropName + "_" + idx));
                ++idx;
            }
            rdr.endArray();
            jsonValue = arrayElements;
        } else if (jsonToken == JsonToken.BEGIN_OBJECT) {
            jsonValue = true;
            if (!this.traverseJsonForInvalidJson(rdr)) {
                jsonValue = null;
            }
        } else if (jsonToken == JsonToken.NUMBER) {
            String numStr = rdr.nextString();
            try {
                if (numStr.contains(".")) {
                    jsonValue = new Double(numStr);
                }
                if (this.isLongInt(numStr)) {
                    jsonValue = new Long(numStr);
                }
                jsonValue = new Double(numStr);
            }
            catch (NumberFormatException e) {
                throw new IOException(String.format("Couldn't turn number %s into a float or int", numStr));
            }
        } else if (jsonToken == JsonToken.STRING) {
            String s = rdr.nextString();
            jsonValue = s;
        } else if (jsonToken == JsonToken.BOOLEAN) {
            jsonValue = rdr.nextBoolean();
        }
        return jsonValue;
    }

    private boolean isLongInt(String numStr) {
        if (!numStr.matches("\\d+")) {
            return false;
        }
        if (numStr.length() < 18) {
            return true;
        }
        try {
            Long.parseLong(numStr);
        }
        catch (Exception e) {
            return false;
        }
        return true;
    }

    private void incrementNumberOfTimesKeySeenInCatalog(String key) {
        Long numTimesSeen = this.mNumberOfTimesKeySeenInCatalog.get(key);
        if (numTimesSeen == null) {
            this.mNumberOfTimesKeySeenInCatalog.put(key, 1L);
        } else {
            this.mNumberOfTimesKeySeenInCatalog.put(key, numTimesSeen + 1L);
        }
    }

    private boolean dataTypesMatch(Object jsonObj, ColumnMetaData columnDefinition) {
        ColumnMetaData.Type definedDataType = columnDefinition.getType();
        String definedCount = columnDefinition.getCount();
        if (jsonObj instanceof Boolean && definedDataType == ColumnMetaData.Type.Boolean) {
            return true;
        }
        if ((jsonObj instanceof Integer || jsonObj instanceof Long) && definedDataType == ColumnMetaData.Type.Integer) {
            return true;
        }
        if ((jsonObj instanceof Float || jsonObj instanceof Double || jsonObj instanceof BigDecimal || jsonObj instanceof Integer) && definedDataType.equals((Object)ColumnMetaData.Type.Float)) {
            return true;
        }
        if (jsonObj instanceof String && definedDataType == ColumnMetaData.Type.String) {
            return true;
        }
        return jsonObj instanceof JsonArray && (definedCount.equals(".") || definedCount.equals("A"));
    }

    private Object jsonNumberToJavaNumber(JsonPrimitive jsonNumber) {
        Number num = this.stringToJavaNumber(jsonNumber.getAsString());
        if (num instanceof Integer || num instanceof Long || num instanceof Float || num instanceof BigDecimal || num instanceof Double) {
            return num;
        }
        return null;
    }

    private void setMaxValuesForKeyIfGreater(String key, long numValues) {
        Long maxValues = this.mMaxValuesForKey.get(key);
        if (maxValues == null) {
            this.mMaxValuesForKey.put(key, numValues);
        } else if (numValues > maxValues) {
            this.mMaxValuesForKey.put(key, numValues);
        }
    }

    private boolean isValidInteger(String intStrValue, ColumnMetaData.Type type) {
        if (type.equals((Object)ColumnMetaData.Type.Integer)) {
            try {
                Integer intVal = new Integer(intStrValue);
                if (intVal % 1 != 0) {
                    return false;
                }
                if (intVal >= Integer.MIN_VALUE && intVal <= Integer.MAX_VALUE) {
                    return true;
                }
            }
            catch (NumberFormatException e) {
                return false;
            }
        }
        return false;
    }

    private boolean isQuotedString(String str) {
        if (str == null) {
            return false;
        }
        return !str.startsWith("\"") || !str.endsWith("\"");
    }

    private Number stringToJavaNumber(String value) {
        if (!value.contains(".")) {
            try {
                long valueAsLong = Long.parseLong(value);
                if (valueAsLong >= Integer.MIN_VALUE && valueAsLong <= Integer.MAX_VALUE) {
                    return Integer.parseInt(value);
                }
                return valueAsLong;
            }
            catch (NumberFormatException valueAsLong) {
                // empty catch block
            }
        }
        try {
            Double valueAsDouble = Double.parseDouble(value);
            if (valueAsDouble >= (double)1.4E-45f && valueAsDouble <= 3.4028234663852886E38) {
                return Float.valueOf(Float.parseFloat(value));
            }
            return valueAsDouble;
        }
        catch (NumberFormatException nf) {
            return new BigDecimal(value);
        }
    }

    private void logInfo(String msg) {
        this.mLogger.logInfo(msg);
    }

    private void logWarning(String msg) {
        this.mLogger.logWarning(msg);
    }

    private void logError(String msg, int code) {
        this.mLogger.logError(msg, code);
    }

    private class Error {
        private String msg;
        private int code;

        public Error(String msg, int code) {
            this.msg = msg;
            this.code = code;
        }

        public String getMsg() {
            return this.msg;
        }

        public int getCode() {
            return this.code;
        }
    }
}

