/*
 * Decompiled with CFR 0.152.
 */
package edu.mayo.bior.catalog.list;

import edu.mayo.bior.catalog.list.CatalogInfo;
import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Properties;

public class CatalogListGenerator {
    public static void usage() {
        System.out.println("\n_bior_catalog_list  <CATALOG_DIR>\n\nThis class crawls the given catalog parent directory (use $BIOR_CATALOG in most cases)\nand builds tab-delimited output to STDOUT that contains key information from each of the catalogs, such as:\n    - Status       (ex: Active  or  Deprecated)\n    - Path         (ex: /research/bsi/data/catalogs/bior/v1/dbSNP/142_GRCh37.p13/variants_nodups.v2/00-All.vcf.tsv.bgz)\n    - LastUpdated  (ex: 2015-11-16 09:39:54)\n    - ShortName    (ex: dbSNP_142_GRCh37p13)\n    - Indexes      (ex: GENE, ID, RSID)\n    - DataSource:  (ex: dbSNP)\n    - Version      (ex: 142)\n    - Build        (ex: GRCh37.p13)\n    - BuildBase    (ex: GRCh37)\n    - Dataset      (ex: Variants)\n\nThis output is used to build the $BIOR_CATALOG/CATALOG_LIST.txt file.\n");
    }

    public static void main(String[] args) {
        if (args.length != 1) {
            CatalogListGenerator.usage();
            return;
        }
        String catalogDirPathToCrawl = args[0];
        try {
            String output = new CatalogListGenerator().createCatalogList(catalogDirPathToCrawl);
            System.out.println(output);
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    public String createCatalogList(String catalogDirPathToCrawl) throws IOException {
        File catalogDir = new File(catalogDirPathToCrawl);
        if (!catalogDir.exists()) {
            System.err.println("Directory does not exist: " + catalogDirPathToCrawl);
            return "";
        }
        if (!catalogDir.isDirectory()) {
            System.err.println("Path is not a directory: " + catalogDirPathToCrawl);
            return "";
        }
        String canonicalPath = catalogDir.getCanonicalPath();
        System.err.println("Catalog path: " + catalogDirPathToCrawl);
        System.err.println("(canonical):  " + canonicalPath);
        List<File> allCatalogs = this.getCatalogs(new File(canonicalPath));
        List<File> catalogs = this.removeNonCatalogDirs(this.removeDuplicates(allCatalogs));
        this.sortCatalogs(catalogs);
        StringBuilder out = new StringBuilder();
        out.append("#Status\tPath\tLastUpdated\tShortName\tIndexes\tDataSource\tVersion\tBuild\tBuildBase\tDataset\n");
        for (File catalog : catalogs) {
            CatalogInfo catalogInfo = this.getCatalogInfo(catalog);
            out.append(catalogInfo + "\n");
        }
        return out.toString();
    }

    private CatalogInfo getCatalogInfo(File catalogFile) throws IOException {
        CatalogInfo catalogInfo = new CatalogInfo();
        catalogInfo.status = this.getStatus(catalogFile);
        catalogInfo.path = catalogFile.getCanonicalPath();
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        catalogInfo.lastUpdated = dateFormat.format(catalogFile.lastModified());
        Properties props = this.getCatalogDatasourceProps(catalogFile);
        catalogInfo.shortName = this.emptyIfNull(props.getProperty("ShortUniqueName"));
        catalogInfo.indexes = this.getIndexes(catalogFile);
        catalogInfo.dataSource = this.emptyIfNull(props.getProperty("Source"));
        catalogInfo.version = this.emptyIfNull(props.getProperty("Version"));
        catalogInfo.build = this.truncateAfterFirstWord(this.emptyIfNull(props.getProperty("Build")));
        catalogInfo.buildBase = this.getBuildBase(catalogInfo.build);
        catalogInfo.dataset = this.emptyIfNull(props.getProperty("Dataset"));
        return catalogInfo;
    }

    private String emptyIfNull(String s) {
        if (s == null) {
            return "";
        }
        return s;
    }

    private String truncateAfterFirstWord(String str) {
        int idxSpace = str.indexOf(" ");
        if (idxSpace == -1) {
            return str;
        }
        return str.substring(0, idxSpace).trim();
    }

    private Properties getCatalogDatasourceProps(File catalogFile) throws IOException {
        String datasourcePropsName = catalogFile.getName().replace(".tsv.bgz", ".datasource.properties").replace(".tsv.gz", ".datasource.properties");
        File datasourcePropsFile = new File(catalogFile.getParentFile(), datasourcePropsName);
        Properties props = new Properties();
        if (!datasourcePropsFile.exists()) {
            return props;
        }
        FileInputStream instream = new FileInputStream(datasourcePropsFile);
        props.load(instream);
        ((InputStream)instream).close();
        return props;
    }

    private String getBuildBase(String buildFull) {
        if (buildFull == null || buildFull.trim().length() == 0) {
            return "";
        }
        int idx = 0;
        for (int i = 0; i < buildFull.length() && Character.isLetterOrDigit(buildFull.charAt(i)); ++i) {
            ++idx;
        }
        return buildFull.substring(0, idx);
    }

    private List<String> getIndexes(File catalogFile) {
        ArrayList<String> indexes = new ArrayList<String>();
        File[] indexFiles = new File(catalogFile.getParentFile(), "index").listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File file, String name) {
                return name.endsWith(".idx.h2.db");
            }
        });
        if (indexFiles == null) {
            return indexes;
        }
        for (File idxFile : indexFiles) {
            String idx = idxFile.getName().replace(".idx.h2.db", "");
            String ctg = catalogFile.getName();
            int numCharsMatching = 0;
            while (idx.charAt(0) == ctg.charAt(0)) {
                idx = idx.substring(1);
                ctg = ctg.substring(1);
                ++numCharsMatching;
            }
            if (numCharsMatching < 3) continue;
            indexes.add(idx);
        }
        Collections.sort(indexes, String.CASE_INSENSITIVE_ORDER);
        return indexes;
    }

    private List<File> sortCatalogs(List<File> catalogs) {
        Comparator<File> fileCompare = new Comparator<File>(){

            @Override
            public int compare(File f1, File f2) {
                try {
                    return f1.getCanonicalPath().compareToIgnoreCase(f2.getCanonicalPath());
                }
                catch (IOException e) {
                    e.printStackTrace();
                    return 0;
                }
            }
        };
        Collections.sort(catalogs, fileCompare);
        return catalogs;
    }

    private List<File> getCatalogs(File startingDir) throws IOException {
        File[] files;
        ArrayList<File> catalogs = new ArrayList<File>();
        for (File f : files = startingDir.listFiles()) {
            if (f.isDirectory()) {
                catalogs.addAll(this.getCatalogs(f));
                continue;
            }
            if (!f.getName().endsWith(".tsv.bgz")) continue;
            catalogs.add(f.getCanonicalFile());
        }
        return catalogs;
    }

    private List<File> removeDuplicates(List<File> files) throws IOException {
        block0: for (int i = files.size() - 1; i >= 0; --i) {
            if (files.get(i).getAbsolutePath().contains("latest")) {
                files.remove(i);
                continue;
            }
            for (int j = 0; j < i; ++j) {
                if (!files.get(i).getCanonicalPath().equals(files.get(j).getCanonicalPath())) continue;
                files.remove(i);
                continue block0;
            }
        }
        return files;
    }

    private List<File> removeNonCatalogDirs(List<File> files) throws IOException {
        for (int i = files.size() - 1; i >= 0; --i) {
            String path = files.get(i).getCanonicalPath();
            if (!path.contains("_BuildCatalogs") && !path.contains("/build/") && !path.contains(".svn")) continue;
            files.remove(i);
        }
        return files;
    }

    private boolean isDeprecated(File catalog) {
        File deprecatedFile = new File(catalog.getParentFile(), catalog.getName().replace(".tsv.bgz", ".DEPRECATED.txt"));
        return deprecatedFile.exists();
    }

    private CatalogInfo.CatalogStatus getStatus(File catalog) {
        CatalogInfo.CatalogStatus status = CatalogInfo.CatalogStatus.Active;
        if (this.isDeprecated(catalog)) {
            status = CatalogInfo.CatalogStatus.Deprecated;
        }
        return status;
    }
}

