/*
 * Decompiled with CFR 0.152.
 */
package edu.mayo.pipes.bioinformatics;

import edu.mayo.pipes.bioinformatics.FormatHeader;
import edu.mayo.pipes.bioinformatics.HeaderFieldDefinition;
import edu.mayo.pipes.bioinformatics.HeaderFieldDefinitionHelpers;
import edu.mayo.pipes.bioinformatics.InfoHeader;
import edu.mayo.pipes.bioinformatics.KeyValueHeaderPair;
import edu.mayo.pipes.bioinformatics.MetaHeader;
import edu.mayo.pipes.bioinformatics.SampleHeader;
import edu.mayo.pipes.bioinformatics.VCFParseException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.log4j.Logger;

public class VCFHeaderParser {
    private static final Logger logger = Logger.getLogger(VCFHeaderParser.class);
    private final String fileFormatRegexStr = "##fileformat=VCFv(.+)";
    private static final int FILE_FORMAT_REGEX_GRP_VERSION = 1;
    private Pattern fileFormatRegexPattern = Pattern.compile("##fileformat=VCFv(.+)");
    private static final float FILE_FORMAT_UNKNOWN = -1.0f;
    private float fileFormat = -1.0f;
    private Map<String, InfoHeader> infoDefs = new LinkedHashMap<String, InfoHeader>();
    private Map<String, HeaderFieldDefinition> formatDefs = new LinkedHashMap<String, HeaderFieldDefinition>();
    private Map<String, MetaHeader> metaDefs = new LinkedHashMap<String, MetaHeader>();
    private Map<String, SampleHeader> sampleDefs = new LinkedHashMap<String, SampleHeader>();
    private List<String> infoFlags = new ArrayList<String>();
    private WarningCallback warningCallback = new WarningCallback(){

        @Override
        public void warning(String message) {
            logger.warn((Object)message);
        }
    };

    public VCFHeaderParser() {
    }

    public VCFHeaderParser(WarningCallback callback) {
        this.warningCallback = callback;
    }

    public List<String> getInfoFlags() {
        return this.infoFlags;
    }

    public void parse(List<String> headerLines) {
        this.reset();
        for (String line : headerLines) {
            String fieldType = this.getFieldType(line);
            try {
                if (fieldType.equals("fileformat")) {
                    this.parseFileFormat(line);
                    continue;
                }
                if (fieldType.equals("INFO")) {
                    this.parseInfoLine(line);
                    continue;
                }
                if (fieldType.equals("FORMAT")) {
                    this.parseFormatLine(line);
                    continue;
                }
                if (fieldType.equals("META")) {
                    this.parseMetaLine(line);
                    continue;
                }
                if (!fieldType.equals("SAMPLE")) continue;
                this.parseSampleLine(line);
            }
            catch (Exception e) {
                this.warningCallback.warning(String.format("WARNING: The following header line appears to have a problem:\n%s\n%s", line, e.getMessage()));
            }
        }
    }

    private void reset() {
        this.infoDefs.clear();
        this.formatDefs.clear();
        this.metaDefs.clear();
        this.sampleDefs.clear();
    }

    private void parseFileFormat(String line) throws VCFParseException {
        Matcher m = this.fileFormatRegexPattern.matcher(line);
        if (!m.find()) {
            throw new VCFParseException("Line is malformed.  Valid syntax is: ##fileformat=VCFv<some number>");
        }
        String versionStr = m.group(1);
        try {
            this.fileFormat = Float.parseFloat(versionStr);
        }
        catch (NumberFormatException nfe) {
            throw new VCFParseException(String.format("Invalid decimal value for the file format version: %s", versionStr));
        }
    }

    public Float getFileFormat() {
        if (this.fileFormat == -1.0f) {
            return null;
        }
        return Float.valueOf(this.fileFormat);
    }

    private void parseInfoLine(String line) throws VCFParseException, UnsupportedEncodingException {
        InfoHeader info = new InfoHeader();
        info.createFromString(line);
        if (info.type.equals((Object)HeaderFieldDefinitionHelpers.ValueType.Flag)) {
            this.infoFlags.add(info.id);
        }
        if (this.infoDefs.containsKey(info.id)) {
            throw new VCFParseException(String.format("Multiple ##INFO lines exist with the same ID: %s", info.id));
        }
        this.infoDefs.put(info.id, info);
    }

    private void parseFormatLine(String line) throws VCFParseException, UnsupportedEncodingException {
        FormatHeader format = new FormatHeader();
        format.createFromString(line);
        if (this.formatDefs.containsKey(format.id)) {
            throw new VCFParseException(String.format("Multiple ##FORMAT lines exist with the same ID: %s", format.id));
        }
        this.formatDefs.put(format.id, format);
    }

    private void parseMetaLine(String line) throws VCFParseException, UnsupportedEncodingException {
        MetaHeader meta = new MetaHeader();
        meta.createFromString(line);
        if (this.metaDefs.containsKey(meta.getId())) {
            throw new VCFParseException(String.format("Multiple ##META lines exist with the same ID: %s", meta.getId()));
        }
        this.metaDefs.put(meta.getId(), meta);
    }

    private void parseSampleLine(String line) throws VCFParseException, UnsupportedEncodingException {
        SampleHeader sampleDef = new SampleHeader();
        sampleDef.createFromString(line);
        sampleDef.setTypesFromMeta(this.metaDefs);
        this.verifyOnlyOneSampleId(sampleDef);
        this.verifySampleFieldsMatchMetaLines(sampleDef);
        this.verifyNoDuplicateSampleKeys(sampleDef);
        this.verifySampleTypeAndMultiplicityAgainstMeta(sampleDef);
        this.sampleDefs.put(sampleDef.getId(), sampleDef);
    }

    private void verifyNoDuplicateSampleKeys(SampleHeader sampleDef) throws VCFParseException {
        Integer count;
        LinkedHashMap<String, Integer> keyCountMap = new LinkedHashMap<String, Integer>();
        for (KeyValueHeaderPair keyValPair : sampleDef.getKeyValuePairs()) {
            count = (Integer)keyCountMap.get(keyValPair.key);
            if (count == null) {
                keyCountMap.put(keyValPair.key, 1);
                continue;
            }
            keyCountMap.put(keyValPair.key, count + 1);
        }
        for (String key : keyCountMap.keySet()) {
            count = (Integer)keyCountMap.get(key);
            if (count <= 1) continue;
            throw new VCFParseException(String.format("Duplicate ##SAMPLE key referenced: %s", key));
        }
    }

    private void verifySampleFieldsMatchMetaLines(SampleHeader sampleDef) throws VCFParseException {
        ArrayList<String> nonMetaMatchingKeys = new ArrayList<String>();
        boolean isAtLeastOneMatch = false;
        for (KeyValueHeaderPair keyValPair : sampleDef.getKeyValuePairs()) {
            if ("ID".equalsIgnoreCase(keyValPair.key) || "Description".equalsIgnoreCase(keyValPair.key)) continue;
            if (!this.metaDefs.containsKey(keyValPair.key)) {
                nonMetaMatchingKeys.add(keyValPair.key);
                continue;
            }
            isAtLeastOneMatch = true;
        }
        if (!isAtLeastOneMatch) {
            throw new VCFParseException(String.format("Sample fields don't match ##META line: %s", ((Object)nonMetaMatchingKeys).toString()));
        }
    }

    private void verifyOnlyOneSampleId(SampleHeader sampleDef) throws VCFParseException {
        List<KeyValueHeaderPair> keyValPairs = sampleDef.getKeyValuePairs();
        int numIdsFound = 0;
        for (KeyValueHeaderPair keyValPair : keyValPairs) {
            if (!"ID".equalsIgnoreCase(keyValPair.key)) continue;
            ++numIdsFound;
        }
        if (numIdsFound > 1) {
            throw new VCFParseException(String.format("Invalid value for ID=%s.  Value already exists.", sampleDef.getId()));
        }
    }

    private void verifySampleTypeAndMultiplicityAgainstMeta(SampleHeader sampleDef) throws VCFParseException {
        for (KeyValueHeaderPair keyValPair : sampleDef.getKeyValuePairs()) {
            MetaHeader metaFieldDef = this.getMetaFromId(keyValPair.key);
            if (metaFieldDef == null) continue;
            if (metaFieldDef.number != null && metaFieldDef.number.intValue() != keyValPair.values.size()) {
                throw new VCFParseException(String.format("Found %s value(s) for %s where there should be exactly %s.", keyValPair.values.size(), keyValPair.key, metaFieldDef.number));
            }
            boolean isFlagMeta = HeaderFieldDefinitionHelpers.ValueType.Flag.equals((Object)metaFieldDef.type);
            boolean isFlagKeyVal = HeaderFieldDefinitionHelpers.ValueType.Flag.equals((Object)keyValPair.valueType);
            if ((isFlagMeta || isFlagKeyVal) && keyValPair.values.size() != 0) {
                throw new VCFParseException(String.format("%s is of Type=flag and cannot have any values assigned.", metaFieldDef.id));
            }
            for (String val : keyValPair.values) {
                HeaderFieldDefinitionHelpers.getMetaValue(metaFieldDef.type, val);
            }
        }
    }

    private MetaHeader getMetaFromId(String key) {
        for (MetaHeader metaFieldDef : this.metaDefs.values()) {
            if (!metaFieldDef.getId().equalsIgnoreCase(key)) continue;
            return metaFieldDef;
        }
        return null;
    }

    public Collection<InfoHeader> getInfoDefinitions() {
        return this.infoDefs.values();
    }

    public InfoHeader getInfoDefinition(String id) {
        return this.infoDefs.get(id);
    }

    public Collection<HeaderFieldDefinition> getFormatDefinitions() {
        return this.formatDefs.values();
    }

    public HeaderFieldDefinition getFormatDefinition(String id) {
        return this.formatDefs.get(id);
    }

    public Collection<MetaHeader> getMetaDefinitions() {
        return this.metaDefs.values();
    }

    public MetaHeader getMetaDefinition(String id) {
        return this.metaDefs.get(id);
    }

    public Collection<SampleHeader> getSampleDefinitions() {
        return this.sampleDefs.values();
    }

    public SampleHeader getSampleDefinition(String id) {
        return this.sampleDefs.get(id);
    }

    private String getFieldType(String line) {
        if (!line.startsWith("##")) {
            return "";
        }
        if (line.contains("=")) {
            String[] split = line.split("=");
            return split[0].substring(2);
        }
        return "";
    }

    static interface WarningCallback {
        public void warning(String var1);
    }
}

