/*
 * Decompiled with CFR 0.152.
 */
package edu.mayo.pipes.bioinformatics;

import edu.mayo.pipes.bioinformatics.HeaderFieldDefinition;
import edu.mayo.pipes.bioinformatics.InfoHeaderFieldDefinition;
import edu.mayo.pipes.bioinformatics.MetaHeaderFieldDefinition;
import edu.mayo.pipes.bioinformatics.SampleDefinition;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.log4j.Logger;

public class VCFHeaderParser {
    private static final Logger logger = Logger.getLogger(VCFHeaderParser.class);
    private final String fileFormatRegexStr = "##fileformat=VCFv(.+)";
    private static final int FILE_FORMAT_REGEX_GRP_VERSION = 1;
    private Pattern fileFormatRegexPattern = Pattern.compile("##fileformat=VCFv(.+)");
    private static final float FILE_FORMAT_UNKNOWN = -1.0f;
    private float fileFormat = -1.0f;
    private final String infoRegexStrLegacy = "##INFO=<ID=(.+),Number=(.+),Type=(.+),Description=\"(.*)\">";
    private final String infoRegexStr = "##INFO=<ID=(.+),Number=(.+),Type=(.+),Description=\"(.*)\",Source=\"(.*)\",Version=\"(.*)\">";
    private static final int INFO_REGEX_GRP_ID = 1;
    private static final int INFO_REGEX_GRP_NUM = 2;
    private static final int INFO_REGEX_GRP_TYPE = 3;
    private static final int INFO_REGEX_DESCRIPTION = 4;
    private static final int INFO_REGEX_SOURCE = 5;
    private static final int INFO_REGEX_VERSION = 6;
    private Pattern infoRegexPatternLegacy = Pattern.compile("##INFO=<ID=(.+),Number=(.+),Type=(.+),Description=\"(.*)\">");
    private Pattern infoRegexPattern = Pattern.compile("##INFO=<ID=(.+),Number=(.+),Type=(.+),Description=\"(.*)\",Source=\"(.*)\",Version=\"(.*)\">");
    private final String formatRegexStr = "##FORMAT=<ID=(.+),Number=(.+),Type=(.+),Description=\"(.*)\">";
    private static final int FORMAT_REGEX_GRP_ID = 1;
    private static final int FORMAT_REGEX_GRP_NUM = 2;
    private static final int FORMAT_REGEX_GRP_TYPE = 3;
    private static final int FORMAT_REGEX_DESCRIPTION = 4;
    private Pattern formatRegexPattern = Pattern.compile("##FORMAT=<ID=(.+),Number=(.+),Type=(.+),Description=\"(.*)\">");
    private final String metaRegexStr = "##META=<ID=(.+),Number=(.+),Type=(.+),Values=\\[(.*)\\],Description=\"(.*)\">";
    private static final int META_REGEX_GRP_ID = 1;
    private static final int META_REGEX_GRP_NUM = 2;
    private static final int META_REGEX_GRP_TYPE = 3;
    private static final int META_REGEX_GRP_VALUES = 4;
    private static final int META_REGEX_DESCRIPTION = 5;
    private Pattern metaRegexPattern = Pattern.compile("##META=<ID=(.+),Number=(.+),Type=(.+),Values=\\[(.*)\\],Description=\"(.*)\">");
    private Map<String, InfoHeaderFieldDefinition> infoDefs = new HashMap<String, InfoHeaderFieldDefinition>();
    private Map<String, HeaderFieldDefinition> formatDefs = new HashMap<String, HeaderFieldDefinition>();
    private Map<String, MetaHeaderFieldDefinition> metaDefs = new HashMap<String, MetaHeaderFieldDefinition>();
    private final String sampleRegexStr = "##SAMPLE=<ID=([^;]+)(.+)?>";
    private static final int SAMPLE_REGEX_GRP_ID = 1;
    private static final int SAMPLE_REGEX_GRP_KEY_VALS = 2;
    private Pattern sampleRegexPattern = Pattern.compile("##SAMPLE=<ID=([^;]+)(.+)?>");
    private Map<String, SampleDefinition> sampleDefs = new HashMap<String, SampleDefinition>();
    private WarningCallback warningCallback = new WarningCallback(){

        @Override
        public void warning(String message) {
            logger.warn((Object)message);
        }
    };

    public VCFHeaderParser() {
    }

    public VCFHeaderParser(WarningCallback callback) {
        this.warningCallback = callback;
    }

    public void parse(List<String> headerLines) {
        this.reset();
        for (String line : headerLines) {
            String fieldType = this.getFieldType(line);
            try {
                if (fieldType.equals("fileformat")) {
                    this.parseFileFormat(line);
                    continue;
                }
                if (fieldType.equals("INFO")) {
                    this.parseInfoLine(line);
                    continue;
                }
                if (fieldType.equals("FORMAT")) {
                    this.parseFormatLine(line);
                    continue;
                }
                if (fieldType.equals("META")) {
                    this.parseMetaLine(line);
                    continue;
                }
                if (!fieldType.equals("SAMPLE")) continue;
                this.parseSampleLine(line);
            }
            catch (VCFParseException e) {
                this.warningCallback.warning(String.format("WARNING: The following header line appears to have a problem: %s\n%s", line, e.getMessage()));
            }
        }
    }

    private void reset() {
        this.infoDefs.clear();
        this.formatDefs.clear();
        this.metaDefs.clear();
        this.sampleDefs.clear();
    }

    private void parseFileFormat(String line) throws VCFParseException {
        Matcher m = this.fileFormatRegexPattern.matcher(line);
        if (!m.find()) {
            throw new VCFParseException("Line is malformed.  Valid syntax is: ##fileformat=VCFv<some number>");
        }
        String versionStr = m.group(1);
        try {
            this.fileFormat = Float.parseFloat(versionStr);
        }
        catch (NumberFormatException nfe) {
            throw new VCFParseException(String.format("Invalid decimal value for the file format version: %s", versionStr));
        }
    }

    public Float getFileFormat() {
        if (this.fileFormat == -1.0f) {
            return null;
        }
        return Float.valueOf(this.fileFormat);
    }

    private void parseInfoLine(String line) throws VCFParseException {
        boolean b3;
        Matcher mLegacy = this.infoRegexPatternLegacy.matcher(line);
        Matcher mCurrent = this.infoRegexPattern.matcher(line);
        boolean mLegacyMatch = mLegacy.find();
        boolean mCurrentMatch = mCurrent.find();
        if (!mLegacyMatch && !mCurrentMatch) {
            if ((double)this.fileFormat > 4.2) {
                throw new VCFParseException("Line is malformed.  Valid syntax is: ##INFO=<ID=ID,Number=number,Type=type,Description=\"description\",Source=\"source\",Version=\"version\">");
            }
            throw new VCFParseException("Line is malformed.  Valid syntax is: ##INFO=<ID=ID,Number=number,Type=type,Description=\"description\">");
        }
        Matcher m = mCurrentMatch ? mCurrent : mLegacy;
        InfoHeaderFieldDefinition def = new InfoHeaderFieldDefinition();
        def.fieldType = "INFO";
        def.id = m.group(1);
        def.type = this.parseType(m.group(3));
        def.number = this.parseNumber(m.group(2), def.type);
        def.desc = this.unescapeDoubleQuotedStrings(m.group(4));
        boolean b1 = this.fileFormat >= 4.2f;
        boolean b2 = mCurrentMatch;
        boolean bl = b3 = m.groupCount() == 6;
        if (this.fileFormat >= 4.2f && mCurrentMatch && m.groupCount() == 6) {
            def.source = this.unescapeDoubleQuotedStrings(m.group(5));
            def.version = this.unescapeDoubleQuotedStrings(m.group(6));
        }
        if (this.infoDefs.containsKey(def.id)) {
            throw new VCFParseException(String.format("Invalid value for ID=%s.  Value already exists.", def.id));
        }
        this.infoDefs.put(def.id, def);
    }

    private void parseFormatLine(String line) throws VCFParseException {
        Matcher m = this.formatRegexPattern.matcher(line);
        if (!m.find()) {
            throw new VCFParseException("Line is malformed.  Valid syntax is: ##FORMAT=<ID=ID,Number=number,Type=type,Description=\"description\">");
        }
        HeaderFieldDefinition def = new HeaderFieldDefinition();
        def.fieldType = "FORMAT";
        def.id = m.group(1);
        def.type = this.parseType(m.group(3));
        def.number = this.parseNumber(m.group(2), def.type);
        def.desc = this.unescapeDoubleQuotedStrings(m.group(4));
        if (this.formatDefs.containsKey(def.id)) {
            throw new VCFParseException(String.format("Invalid value for ID=%s.  Value already exists.", def.id));
        }
        this.formatDefs.put(def.id, def);
    }

    private void parseMetaLine(String line) throws VCFParseException {
        Matcher m = this.metaRegexPattern.matcher(line);
        if (!m.find()) {
            throw new VCFParseException("Line is malformed.  Valid syntax is: ##META=<ID=ID,Number=number,Type=type,Values=[v1,v2,v3],Description=\"description\">");
        }
        HeaderFieldDefinition.ValueType type = this.parseType(m.group(3));
        MetaHeaderFieldDefinition def = null;
        switch (type) {
            case Character: {
                def = new MetaHeaderFieldDefinition();
                break;
            }
            case Flag: {
                def = new MetaHeaderFieldDefinition();
                break;
            }
            case Float: {
                def = new MetaHeaderFieldDefinition();
                break;
            }
            case Integer: {
                def = new MetaHeaderFieldDefinition();
                break;
            }
            case String: {
                def = new MetaHeaderFieldDefinition();
            }
        }
        def.fieldType = "META";
        def.id = m.group(1);
        def.type = type;
        def.number = this.parseNumber(m.group(2), def.type);
        def.desc = this.unescapeDoubleQuotedStrings(m.group(5));
        String valuesStr = m.group(4);
        if (valuesStr.trim().length() > 0) {
            for (String value : valuesStr.split(",")) {
                def.values.add(this.getMetaValue(type, value));
            }
        }
        if (this.metaDefs.containsKey(def.id)) {
            throw new VCFParseException(String.format("Invalid value for ID=%s.  Value already exists.", def.id));
        }
        this.metaDefs.put(def.id, def);
    }

    private Object getMetaValue(HeaderFieldDefinition.ValueType type, String value) throws VCFParseException {
        switch (type) {
            case Character: {
                if (value.length() != 1) {
                    throw new VCFParseException(String.format("The value %s is not a character.", value));
                }
                return Character.valueOf(value.charAt(0));
            }
            case Float: {
                try {
                    return new Float(value);
                }
                catch (NumberFormatException nfe) {
                    throw new VCFParseException(String.format("The value %s is not a float.", value));
                }
            }
            case Integer: {
                try {
                    return new Integer(value);
                }
                catch (NumberFormatException nfe) {
                    throw new VCFParseException(String.format("The value %s is not an integer.", value));
                }
            }
            case String: {
                try {
                    return URLDecoder.decode(value, "UTF-8");
                }
                catch (UnsupportedEncodingException e) {
                    throw new VCFParseException(String.format("The value %s is not UTF-8 encoded.", value));
                }
            }
        }
        return null;
    }

    private HeaderFieldDefinition.ValueType parseType(String typeStr) throws VCFParseException {
        try {
            return HeaderFieldDefinition.ValueType.fromString(typeStr);
        }
        catch (Exception e) {
            throw new VCFParseException(String.format("Invalid value for Type=%s", typeStr));
        }
    }

    private Integer parseNumber(String numberStr, HeaderFieldDefinition.ValueType type) throws VCFParseException {
        if (numberStr.equalsIgnoreCase(".") || numberStr.equalsIgnoreCase("A") || numberStr.equalsIgnoreCase("G")) {
            return null;
        }
        try {
            Integer number = Integer.parseInt(numberStr);
            if (type.equals((Object)HeaderFieldDefinition.ValueType.Flag) && !number.equals(0)) {
                throw new VCFParseException(String.format("Invalid value for Number=%s.  Should be Number=0 if Type=Flag.", number));
            }
            return number;
        }
        catch (NumberFormatException nfe) {
            throw new VCFParseException(String.format("Invalid value for Number=%s", numberStr));
        }
    }

    private void parseSampleLine(String line) throws VCFParseException {
        Matcher m = this.sampleRegexPattern.matcher(line);
        if (!m.find()) {
            throw new VCFParseException("Line is malformed.  Valid syntax is: ##SAMPLE=<ID=sampleID;key[=data[,data]];...>");
        }
        String name = m.group(1);
        SampleDefinition sampleDef = new SampleDefinition(name);
        if (m.groupCount() == 2) {
            String keyVals = m.group(2);
            HashMap<String, String> nonBooleanMetas = new HashMap<String, String>();
            ArrayList<String> booleanMetas = new ArrayList<String>();
            for (String kv : keyVals.split(";")) {
                String value;
                String key;
                if (kv.trim().length() == 0) continue;
                if (kv.contains("=")) {
                    String[] pair = kv.split("=");
                    key = pair[0];
                    value = pair[1];
                } else {
                    key = kv;
                    value = null;
                }
                if (!this.metaDefs.containsKey(key)) {
                    throw new VCFParseException(String.format("%s does not reference a ##META", key));
                }
                if (nonBooleanMetas.containsKey(key) || booleanMetas.contains(key)) {
                    throw new VCFParseException(String.format("Duplicate ##META ID referenced: %s", key));
                }
                if (value == null) {
                    booleanMetas.add(key);
                    continue;
                }
                nonBooleanMetas.put(key, value);
            }
            for (MetaHeaderFieldDefinition fieldDef : this.metaDefs.values()) {
                if (fieldDef.type.equals((Object)HeaderFieldDefinition.ValueType.Flag)) {
                    if (nonBooleanMetas.containsKey(fieldDef.id)) {
                        throw new VCFParseException(String.format("%s is of Type=flag and cannot have any values assigned.", fieldDef.id));
                    }
                    if (booleanMetas.contains(fieldDef.id)) {
                        sampleDef.setBoolean(fieldDef.id, true);
                        continue;
                    }
                    sampleDef.setBoolean(fieldDef.id, false);
                    continue;
                }
                if (!nonBooleanMetas.containsKey(fieldDef.id)) continue;
                String valueStr = (String)nonBooleanMetas.get(fieldDef.id);
                ArrayList<String> values = new ArrayList<String>();
                if (fieldDef.number == null || fieldDef.number > 1) {
                    Collections.addAll(values, valueStr.split(","));
                } else {
                    values.add(valueStr);
                }
                if (fieldDef.number != null && fieldDef.number.intValue() != values.size()) {
                    throw new VCFParseException(String.format("Found %s value(s) for %s where there should be exactly %s.", values.size(), fieldDef.id, fieldDef.number));
                }
                for (String v : values) {
                    Object metaValue = this.getMetaValue(fieldDef.type, v);
                    switch (fieldDef.type) {
                        case Float: {
                            sampleDef.addFloat(fieldDef.id, (Float)metaValue);
                            break;
                        }
                        case Integer: {
                            sampleDef.addInteger(fieldDef.id, (Integer)metaValue);
                            break;
                        }
                        case String: {
                            sampleDef.addString(fieldDef.id, (String)metaValue);
                            break;
                        }
                        case Character: {
                            sampleDef.addCharacter(fieldDef.id, (Character)metaValue);
                        }
                    }
                }
            }
        }
        if (this.sampleDefs.containsKey(sampleDef.getName())) {
            throw new VCFParseException(String.format("Invalid value for ID=%s.  Value already exists.", sampleDef.getName()));
        }
        this.sampleDefs.put(sampleDef.getName(), sampleDef);
    }

    private String unescapeDoubleQuotedStrings(String dblQuotedStr) {
        return dblQuotedStr.replace("\\\"", "\"").replace("\\\\", "\\");
    }

    public Collection<InfoHeaderFieldDefinition> getInfoDefinitions() {
        return this.infoDefs.values();
    }

    public InfoHeaderFieldDefinition getInfoDefinition(String id) {
        return this.infoDefs.get(id);
    }

    public Collection<HeaderFieldDefinition> getFormatDefinitions() {
        return this.formatDefs.values();
    }

    public HeaderFieldDefinition getFormatDefinition(String id) {
        return this.formatDefs.get(id);
    }

    public Collection<MetaHeaderFieldDefinition> getMetaDefinitions() {
        return this.metaDefs.values();
    }

    public MetaHeaderFieldDefinition getMetaDefinition(String id) {
        return this.metaDefs.get(id);
    }

    public Collection<SampleDefinition> getSampleDefinitions() {
        return this.sampleDefs.values();
    }

    public SampleDefinition getSampleDefinition(String id) {
        return this.sampleDefs.get(id);
    }

    private String getFieldType(String line) {
        if (!line.startsWith("##")) {
            return "";
        }
        if (line.contains("=")) {
            String[] split = line.split("=");
            return split[0].substring(2);
        }
        return "";
    }

    private class VCFParseException
    extends Exception {
        public VCFParseException(String message) {
            super(message);
        }
    }

    static interface WarningCallback {
        public void warning(String var1);
    }
}

