#!/bin/bash

###====================================================================
### Build jars and deployable zip files for external deployment
###====================================================================

targetDir=$1
biorPipelineProjDir=$2
biorCatalogProjDir=$3
pipesProjDir=$4

function usage() {
  echo "Usage:  buildExternalDeployment.sh  <ExternalDistDir>  <biorPipelineProjDir>  <biorCatalogProjDir>  <pipesProjDir>"
  echo "Example: buildExternalDeployment.sh   ~/ExternalDistribution  ~/bior_lite/bior_pipeline  ~/bior_lite/bior_catalog  ~/pipes"
}


if [ -z "$targetDir" ] || [ ! -d "$targetDir" ]  ; then
  echo "Target directory for external distribution not specified (or does not exist)"
  usage
  exit 1
fi

if [ -z "$biorPipelineProjDir" ] || [ ! -d "$biorPipelineProjDir" ] ; then
  echo "bior_pipeline project directory not specified (or does not exist)"
  usage
  exit 1
fi

if [ -z "$biorCatalogProjDir" ] || [ ! -d "$biorCatalogProjDir" ]  ; then
  echo "bior_catalog project directory not specified (or does not exist)"
  usage
  exit 1
fi

if [ -z "$pipesProjDir" ] || [ ! -d "$pipesProjDir" ]  ; then
  echo "pipes project directory not specified (or does not exist)"
  usage
  exit 1
fi


# Get name and version (ex: "bior_pipeline_2.2.2-SNAPSHOT") from pom.xml in current dir
function getNameVersion() {
  dir=$1
  # echo "dir: [$dir]"

  # Find the project name in pom.xml
  # Ex: Given "<name>bior_pipeline</name>" in pom.xml, pull out "bior_pipeline" (remove the tags and any spaces/tabs)
  name=$(grep "<name>" $dir/pom.xml | head -1 | tr -d '\t' | tr -d ' ' | sed 's/<name>//g' | sed 's/<\/name>//g')
  # echo "name: [$name]"
  
  # Find "<version>2.2.2-SNAPSHOT</version>" in pom.xml and pull out "2.2.2-SNAPSHOT" (remove the tags and any spaces/tabs)
  version=$(grep "<version>" $dir/pom.xml | head -1 | tr -d '\t' | tr -d ' ' | sed 's/<version>//g' | sed 's/<\/version>//g')
  # echo "version: [$version]"
   
  # Construct name-version  (ex: "bior_pipeline_2.2.2-SNAPSHOT")
  # Use the $nameVersion variable after calling this function
  nameVersion="${name}_${version}"
  # echo "nameVersion: [$nameVersion]"
}




### =======================================================
### Build the external zip for the bior_pipeline project
### =======================================================

### NOTE: Make sure the google doc has been downloaded as PDF and checked into SVN before running this!!! 

getNameVersion $biorPipelineProjDir

cd $targetDir
unzip -q $biorPipelineProjDir/target/bior_pipeline-*-distribution.zip
oldDir=`echo bior_pipeline-*`
newDir=`echo bior_pipeline-* | sed 's/bior_pipeline-/bior_/g' | sed 's/SNAPSHOT/beta/g'`
mv $oldDir $newDir
cd $newDir

# Change the sage "dev" to "prod"
# How to do this in one command line rather than interactively
sed -i '' 's/SAGE_ENVIRONMENT=dev/SAGE_ENVIRONMENT=prod/g' conf/cli/Global.properties 

mv conf/bior.properties.external conf/bior.properties
cp $biorPipelineProjDir/setupEnv.sh.external ./setupEnv.sh


# Copy all extra files to this directory  (do NOT copy the .svn/ subdirectories)
# use the -v flag with rsync to see what is copied
echo "Copy all doc, examples, license, scripts to temp dir"
rsync -a  $biorPipelineProjDir/doc      . --exclude .svn  --exclude .DS_Store
rsync -a  $biorPipelineProjDir/examples . --exclude .svn
rsync -a  $biorPipelineProjDir/license  . --exclude .svn
rsync -a  $biorPipelineProjDir/scripts  . --exclude .svn
cp $biorPipelineProjDir/RELEASE_NOTES.txt .
cp $biorPipelineProjDir/README.txt .
cp $biorPipelineProjDir/License.txt .


## Turn off the tar'ing of ._xxxxxx files
export COPYFILE_DISABLE=true
cd ..
echo "Tar the bior_pipeline deployment directory..."
tar -zcf ${newDir}.tar.gz  $newDir

echo "Remove the temp directory (MAKE SURE IT IS 10 chars or LONGER!!!)..."
len=${#newDir}
if [ $len -ge 10 ]; then
  echo "Yes it is >= 10 chars  (len=[$len])"
  echo "Removing dir: $newDir"
  rm -rf "$newDir"
fi


### =======================================================
### Tar-Gzip all the source code directories
### =======================================================
biorPipelineProjDir=$2
biorCatalogProjDir=$3
pipesProjDir=$4

echo "Tar the bior_pipeline source dir..."
cd $biorPipelineProjDir/..
tar  --exclude='./Dist' --exclude='./target' --exclude='.svn' --exclude='.git*' --exclude='.settings'  -zcf  $targetDir/${nameVersion}.source.tar.gz  bior_pipeline

echo "Tar the bior_catalog source dir..."
getNameVersion $biorCatalogProjDir
cd $biorCatalogProjDir/..
tar  --exclude='./DataFiles' --exclude='./target'  --exclude='.svn' --exclude='.git*' --exclude='.settings' -zcf  $targetDir/${nameVersion}.source.tar.gz  bior_catalog

echo "Tar the pipes source dir..."
getNameVersion $pipesProjDir
cd $pipesProjDir/..
tar  --exclude='./target'  --exclude='.svn' --exclude='.git*' --exclude='.settings'  -zcf  $targetDir/${nameVersion}.source.tar.gz   pipes

echo "Done."