package edu.mayo.bior.pipeline.createcatalog;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.StringWriter;

import org.apache.log4j.PatternLayout;
import org.apache.log4j.WriterAppender;
import org.apache.log4j.Logger;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import edu.mayo.bior.cli.func.BaseFunctionalTest;

public class TjsonToCatalogNoJsonParsingPipeTest extends BaseFunctionalTest {

	private StringWriter logStringWriter;

	private static Logger logger = Logger.getLogger(TjsonToCatalogNoJsonParsingPipe.class);
	
	@BeforeClass
	public static void beforeAll() {
	}
	
	@Before
	public void before() {
		// Capture the log outputs
		logStringWriter = new StringWriter();
		WriterAppender appender = new WriterAppender(new PatternLayout(), logStringWriter);
		logger.addAppender(appender);
	}
	
	public String getLog() {
		logStringWriter.flush();
		return logStringWriter.toString();
	}
	
	
	@Test
	public void testGetJsonValue() {
		TjsonToCatalogNoJsonParsingPipe t = new TjsonToCatalogNoJsonParsingPipe(1, true, false);
		// K as an int - last key-val in json - single value, within json, single value within array, multi-value within array
		assertEquals("1", 		t.getJsonVal(swapQt("{'_altAlleles':['A','ABC'],'K':1}"), 			"K"));
		assertEquals("1", 		t.getJsonVal(swapQt("{'_altAlleles':['A','ABC'],'Sub':{'K':1}}"),	"K"));
		assertEquals("[1]", 	t.getJsonVal(swapQt("{'_altAlleles':['A','ABC'],'K':[1]}"),			"K"));
		assertEquals("[1,2,3]", t.getJsonVal(swapQt("{'_altAlleles':['A','ABC'],'K':[1,2,3]}"),		"K"));

		// K as a string - last key-val in json - single value, within json, single value within array, multi-value within array
		assertEquals("1", 						t.getJsonVal(swapQt("{'_altAlleles':['A','ABC'],'K':'1'}"), 			"K"));
		assertEquals("1", 						t.getJsonVal(swapQt("{'_altAlleles':['A','ABC'],'Sub':{'K':'1'}}"),		"K"));
		assertEquals(swapQt("['1']"), 			t.getJsonVal(swapQt("{'_altAlleles':['A','ABC'],'K':['1']}"),			"K"));
		assertEquals(swapQt("['1','2','3']"),	t.getJsonVal(swapQt("{'_altAlleles':['A','ABC'],'K':['1','2','3']}"),	"K"));

		//------
		// K as an int - first key-val in json - single value, within json, single value within array, multi-value within array
		assertEquals("1", 		t.getJsonVal(swapQt("{'K':1,'J':2}"), 			"K"));
		assertEquals("1",		t.getJsonVal(swapQt("{'Sub':{'K':1},'J':2}"),	"K"));
		assertEquals("[1]", 	t.getJsonVal(swapQt("{'K':[1],'J':2}"),			"K"));
		assertEquals("[1,2,3]", t.getJsonVal(swapQt("{'K':[1,2,3],'J':2}"),		"K"));

		// K as a string - first key-val in json - single value, within json, single value within array, multi-value within array
		assertEquals("1", 						t.getJsonVal(swapQt("{'K':'1','_altAlleles':['A','ABC']}"), 			"K"));
		assertEquals("1", 						t.getJsonVal(swapQt("{'Sub':{'K':'1'},'_altAlleles':['A','ABC']}"),		"K"));
		assertEquals(swapQt("['1']"), 			t.getJsonVal(swapQt("{'K':['1'],'_altAlleles':['A','ABC']}"),			"K"));
		assertEquals(swapQt("['1','2','3']"), 	t.getJsonVal(swapQt("{'K':['1','2','3'],'_altAlleles':['A','ABC']}"),	"K"));

		//------
		// K as an int - mid key-val in json - single value, within json, single value within array, multi-value within array
		assertEquals("1", 		t.getJsonVal(swapQt("{'A':0,'K':1,'J':2}"), 		"K"));
		assertEquals("1", 		t.getJsonVal(swapQt("{'A':0,'Sub':{'K':1},'J':2}"),	"K"));
		assertEquals("[1]", 	t.getJsonVal(swapQt("{'A':0,'K':[1],'J':2}"),		"K"));
		assertEquals("[1,2,3]", t.getJsonVal(swapQt("{'A':0,'K':[1,2,3],'J':2}"),	"K"));

		// K as a string - mid key-val in json - single value, within json, single value within array, multi-value within array
		assertEquals("1", 						t.getJsonVal(swapQt("{'A':0,'K':'1','_altAlleles':['A','ABC']}"), 			"K"));
		assertEquals("1", 						t.getJsonVal(swapQt("{'A':0,'Sub':{'K':'1'},'_altAlleles':['A','ABC']}"),	"K"));
		assertEquals(swapQt("['1']"), 			t.getJsonVal(swapQt("{'A':0,'K':['1'],'_altAlleles':['A','ABC']}"),			"K"));
		assertEquals(swapQt("['1','2','3']"), 	t.getJsonVal(swapQt("{'A':0,'K':['1','2','3'],'_altAlleles':['A','ABC']}"),	"K"));

		
		assertEquals(swapQt("['A','ABC']"), t.getJsonVal(swapQt("{'_altAlleles':['A','ABC']}"), "_altAlleles"));
		
		// TODO: This will certainly mess it up for now, but finding the JSON object embedded within the JSON
		//assertEquals(swapQt("{'A':'123abc'}"), t.getJsonVal(swapQt("{'B':{'A':'123abc'}}"), 	"B"));
		
	}
	
	private String swapQt(String s) {
		return s.replace("'", "\"");
	}
	
	@Test
	public void testReplaceValueInJson() {
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(1, /*isJsonOnly=*/false, /*isModChr=*/true);
		
		// Change single key-val  (single-quote)
		assertEquals("{'chrom':'13'}",  pipe.replaceValueInJson("{'chrom':'12'}", "chrom", "13", /*isValQt=*/true));
		
		// Change single key-val  (double-quote)
		assertEquals("{\"chrom\":\"13\"}",  pipe.replaceValueInJson("{\"chrom\":\"12\"}", "chrom", "13", /*isValQt=*/true));

		// Change First value (int, so unquoted)
		assertEquals("{\"minBP\":101,\"maxBP\":101}", pipe.replaceValueInJson("{\"minBP\":100,\"maxBP\":101}", "minBP", "101", /*isValQt=*/false));

		// Change Last value (int, so unquoted)
		assertEquals("{\"minBP\":101,\"maxBP\":101}", pipe.replaceValueInJson("{\"minBP\":101,\"maxBP\":100}", "maxBP", "101", /*isValQt=*/false));

		// Change Mid value (int, so unquoted)
		assertEquals("{\"minBP\":101,\"maxBP\":101,\"chrom\":\"X\"}", pipe.replaceValueInJson("{\"minBP\":101,\"maxBP\":100,\"chrom\":\"X\"}", "maxBP", "101", /*isValQt=*/false));

		// Change Mid value (string, so quoted) - WARNING!  Comma and escaped quote in string!!!!
		assertEquals("{\"minBP\":101,\"chrom\":\"X\",\"maxBP\":101}", pipe.replaceValueInJson("{\"minBP\":101,\"chrom\":\"MT\",\"maxBP\":101}", "chrom", "X", /*isValQt=*/true));

		// Change Mid value (string, so quoted) - WARNING!  Comma and escaped quote in string!!!!
		// NOTE: We won't worry about this one for now since we are just pulling out _landmark, _minBP, and _maxBP so shouldn't have quotes and commas in the values
		//assertEquals("{\"minBP\":101,\"chrom\":\"X\",\"maxBP\":101}", pipe.replaceValueInJson("{\"minBP\":101,\"chrom\":\"MTTTT, \\\":; , < > |#$!@#$^\",\"maxBP\":101}", "chrom", "X", /*isValQt=*/true));
		
		// Insert into empty JSON
		assertEquals("{\"minBP\":100}", pipe.replaceValueInJson("{}", "minBP", "100", /*isValQt=*/false));
	}

	@Test
	/** No _landmark, _minBP, or _maxBP (should return UNKNOWN for first col, but not add/modify _landmark value) */
	public void testNoLandmark_noMinbp_noMaxbp() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/true);
		String in  = concat("1",  "2",  "{'x':3,'y':4}");
		String out = pipe.processLine(in);
		final String EXPECTED = concat("UNKNOWN",  "0",  "0",  "{'x':3,'y':4}"); 
		assertEquals(EXPECTED, out);
	}

	
	@Test
	/** No _landmark  (should return UNKNOWN for first col, but not add/modify _landmark value) */
	public void testNoLandmark() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/true);
		String in  = concat("1",  "2",  "{'_minBP':100,'_maxBP':101}");
		String out = pipe.processLine(in);
		final String EXPECTED = concat("UNKNOWN",  "100",  "101",  "{'_minBP':100,'_maxBP':101}"); 
		assertEquals(EXPECTED, out);
	}
	
	@Test
	/** Empty _landmark, isModifyChrom=true  (should return UNKNOWN for first col, and modify _landmark value since it is present) */
	public void testEmptyLandmark() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/true);
		String in  = concat("1",  "2",  "{'_landmark':'','_minBP':100,'_maxBP':101}");
		String out = pipe.processLine(in);
		final String EXPECTED = concat("UNKNOWN",  "100",  "101",  "{'_landmark':'UNKNOWN','_minBP':100,'_maxBP':101}");
		assertEquals(EXPECTED, out);
		final String EXPECTED_WARNING = "Warning: _landmark was not found (or was empty) on at least one line.  This was replaced with UNKNOWN.  Only first offence shown : {'_landmark':'','_minBP':100,'_maxBP':101}\n";
		assertEquals(EXPECTED_WARNING, getLog());
	}

	@Test
	/** Empty _landmark, isModifyChrom=false  (should return UNKNOWN for first col, but don't modify _landmark value) */
	public void testEmptyLandmark_noModify() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/false);
		String in  = concat("1",  "2",  "{'_landmark':'','_minBP':100,'_maxBP':101}");
		String out = pipe.processLine(in);
		final String EXPECTED = concat("UNKNOWN",  "100",  "101",  "{'_landmark':'UNKNOWN','_minBP':100,'_maxBP':101}"); 
		assertEquals(EXPECTED, out);
	}


	@Test
	/** Non-standard chrom, isModifyChrom=false  (should set tabix chrom to standard chrom, but not modify chrom in JSON) */
	public void testNonStandardChrom_noModify() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/false);
		String in  = concat("1",  "2",  "{'_landmark':'23','_minBP':100,'_maxBP':101}");
		String out = pipe.processLine(in);
		final String EXPECTED = concat("23",  "100",  "101",  "{'_landmark':'23','_minBP':100,'_maxBP':101}"); 
		assertEquals(EXPECTED, out);
	}

	@Test
	/** Non-standard chrom, isModifyChrom=true  (should set tabix chrom to standard chrom, and modify chrom in JSON) */
	public void testNonStandardChrom_modify() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/true);
		String in  = concat("1",  "2",  "{'_landmark':'23','_minBP':100,'_maxBP':101}");
		String out = pipe.processLine(in);
		final String EXPECTED = concat("X",  "100",  "101",  "{'_landmark':'X','_minBP':100,'_maxBP':101}"); 
		assertEquals(EXPECTED, out);
	}

	@Test
	/** Non-standard chrom, isModifyChrom=true  (should set tabix chrom to standard chrom, and modify chrom in JSON) */
	public void testNonStandardChromWithChrPrefix_modify() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/true);
		String in  = concat("1",  "2",  "{'_landmark':'chr12','_minBP':100,'_maxBP':101}");
		String out = pipe.processLine(in);
		final String EXPECTED = concat("12",  "100",  "101",  "{'_landmark':'12','_minBP':100,'_maxBP':101}"); 
		assertEquals(EXPECTED, out);
	}

	@Test
	/** Chrom not quoted, isModifyChrom=true - should quote, modify, and give warning */
	public void testChromNotQuoted_quoteModifyAndWarn() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/true);
		String in  = concat("1",  "2",  swapQuotes("{'_landmark':chr12,'_minBP':100,'_maxBP':101}"));
		String out = pipe.processLine(in);
		final String EXPECTED = concat("12",  "100",  "101",  swapQuotes("{'_landmark':'12','_minBP':100,'_maxBP':101}")); 
		assertEquals(EXPECTED, out);

		final String EXPECTED_WARNING = "Warning: _landmark was not a String.  Correcting offending JSON (only first original shown): {\"_landmark\":chr12,\"_minBP\":100,\"_maxBP\":101}\n"
									+   "Warning: _landmark was changed from chr12 to 12.  Correcting offending JSON (only first original shown): {\"_landmark\":chr12,\"_minBP\":100,\"_maxBP\":101}\n";

		assertEquals(EXPECTED_WARNING, getLog());
	}

	@Test
	/** Chrom not quoted, isModifyChrom=false - should quote, NOT modify, and give warning */
	public void testChromNotQuoted_quoteNotModifyAndWarn() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/false);
		String in  = concat("1",  "2",  "{'_landmark':chr12,'_minBP':100,'_maxBP':101}");
		String out = pipe.processLine(in);
		final String EXPECTED = concat("chr12",  "100",  "101",  "{'_landmark':'chr12','_minBP':100,'_maxBP':101}"); 
		assertEquals(EXPECTED, out);
		
		final String EXPECTED_WARNING = "Warning: _landmark was not a String.  Correcting offending JSON (only first original shown): {'_landmark':chr12,'_minBP':100,'_maxBP':101}\n";
		assertEquals(EXPECTED_WARNING, getLog());
	}

	@Test
	/** Min quoted - should modify and warn */
	public void testMinQuoted_modifyAndWarn() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/true);
		String in  = concat("1",  "2",  "{'_landmark':'12','_minBP':'100','_maxBP':101}");
		String out = pipe.processLine(in);
		final String EXPECTED = concat("12",  "100",  "101",  "{'_landmark':'12','_minBP':100,'_maxBP':101}"); 
		assertEquals(EXPECTED, out);
		
		final String EXPECTED_WARNING = "Warning: _minBP was not an integer.  Correcting offending JSON (only first original shown): {'_landmark':'12','_minBP':'100','_maxBP':101}\n";
		assertEquals(EXPECTED_WARNING, getLog());
	}

	@Test
	/** Max quoted - should modify and warn */
	public void testMaxQuoted_modifyAndWarn() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/true);
		String in  = concat("1",  "2",  "{'_landmark':'12','_minBP':100,'_maxBP':'101'}");
		String out = pipe.processLine(in);
		final String EXPECTED = concat("12",  "100",  "101",  "{'_landmark':'12','_minBP':100,'_maxBP':101}"); 
		assertEquals(EXPECTED, out);
		
		final String EXPECTED_WARNING = "Warning: _maxBP was not an integer.  Correcting offending JSON (only first original shown): {'_landmark':'12','_minBP':100,'_maxBP':'101'}\n";
		assertEquals(EXPECTED_WARNING, getLog());
	}

	@Test
	/** Min and Max quoted - should modify and warn for BOTH */
	public void testMinAndMaxQuoted_modifyAndWarn() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/true);
		String in  = concat("1",  "2",  "{'_landmark':'12','_minBP':'100','_maxBP':'101'}");
		String out = pipe.processLine(in);
		final String EXPECTED = concat("12",  "100",  "101",  "{'_landmark':'12','_minBP':100,'_maxBP':101}"); 
		assertEquals(EXPECTED, out);
		
		final String EXPECTED_WARNING = "Warning: _minBP was not an integer.  Correcting offending JSON (only first original shown): {'_landmark':'12','_minBP':'100','_maxBP':'101'}\n"
									+   "Warning: _maxBP was not an integer.  Correcting offending JSON (only first original shown): {'_landmark':'12','_minBP':'100','_maxBP':'101'}\n";
		assertEquals(EXPECTED_WARNING, getLog());
	}


	@Test
	/** Max not specified, but min and ref are, so calculate max and modify JSON and warn */
	public void testMaxNotSpecified_calcMaxModifyJsonAndWarn() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/true);
		String in  = concat("1",  "2",  swapQuotes("{'_landmark':'12','_minBP':100,'_refAllele':'AACC'}"));
		String out = pipe.processLine(in);
		final String EXPECTED = concat("12",  "100",  "103",  swapQuotes("{'_landmark':'12','_minBP':100,'_refAllele':'AACC','_maxBP':103}")); 
		assertEquals(EXPECTED, out);
		
		final String EXPECTED_WARNING = "Warning: _maxBP was not specified, so it will be calculated from _minBP and _refAllele length.\n"
									+   "First offending JSON shown: " + swapQuotes("{'_landmark':'12','_minBP':100,'_refAllele':'AACC'}\n");
		assertEquals(EXPECTED_WARNING, getLog());
	}

	@Test
	/** Ref allele length does not match positions - throw exception */
	public void testRefAlleleLengthDoesNotMatchStartEndPositions() {
		// NOTE: When calling processLine() without first calling processNextStart, it bypasses the 1-based to 0-based jsonCol conversion, so should use 0-based columns here
		TjsonToCatalogNoJsonParsingPipe pipe = new TjsonToCatalogNoJsonParsingPipe(/*jsonCol=*/2, /*isJsonOnly=*/false, /*isModifyChromosome=*/true);
		String in  = concat("1",  "2",  "{'_landmark':'12','_minBP':100,'_maxBP':100,'_refAllele':'AA'}".replace("'", "\""));
		try {
			String out = pipe.processLine(in);
			fail("Expected exception: Error:  length of the refAllele does not equal (max-min+1): {\"_landmark\":\"12\",\"_minBP\":100,\"_maxBP\":100,\"_refAllele\":\"AA\"}");
		} catch(Exception e) {
			assertTrue(e instanceof IllegalArgumentException);
			assertEquals("Error:  length of the refAllele does not equal (max-min+1): {\"_landmark\":\"12\",\"_minBP\":100,\"_maxBP\":100,\"_refAllele\":\"AA\"}", e.getMessage());
		}
	}
}
