/**
 * bior_pipeline
 *
 * <p>@author Gregory Dougherty</p>
 * Copyright Mayo Clinic, 2014
 *
 */
package edu.mayo.bior.pipeline;

import java.io.*;
import java.util.*;

import java.sql.SQLException;

import org.apache.commons.io.FileUtils;
import org.junit.*;

import com.tinkerpop.pipes.Pipe;

import edu.mayo.bior.pipes.history.MakeFirstLineHeaderPipe;
import edu.mayo.bior.util.*;
import edu.mayo.genomicutils.refassembly.*;

import com.tinkerpop.pipes.util.Pipeline;

import edu.mayo.bior.pipeline.Variant2JSONPipe;
import edu.mayo.pipes.JSON.DrillPipe;
import edu.mayo.pipes.JSON.lookup.LookupPipe;
import edu.mayo.pipes.JSON.tabix.*;
import edu.mayo.pipes.history.History;
import edu.mayo.pipes.history.HistoryInPipe;
import edu.mayo.pipes.history.HistoryOutPipe;
import edu.mayo.pipes.iterators.FileLineIterator;
import edu.mayo.pipes.util.metadata.Metadata;
import edu.mayo.pipes.util.test.PipeTestUtils;

/**
 * Class to implement JUnit tests for the {@linkplain Variant2JSONPipe}, testing code that generates JSON 
 * from a Chromosome and position, or from an rsID
 *
 * <p>@author Gregory Dougherty</p>
 */
public class Variant2JSONPipeTestITCase
{
	private static DefaultProcessor			gDefaultProcessor = new DefaultProcessor ();
	private static List<String>				gOverlapPaths = new ArrayList<String> ();
	private static List<String>				gLookupPaths = new ArrayList<String> ();
	/** Map from catalog path to lookup field */
	private static Map<String, String[]>	gLookupDrill = new HashMap<String, String[]> ();
	/** Map from lookup catalog path to associated overlap catalog path */
	private static Map<String, String>		gLookupCatalogs = new HashMap<String, String> ();
	/** Map from lookup catalog path to associated map of index paths to index files */
	private static Map<String, Map<String, String>>	gLookupIndexes = new HashMap<String, Map<String, String>> ();
	private static Map<String, FieldProcessor>	gFieldProcessorMap = new HashMap<String, FieldProcessor> ();
	
//	private static final String	kDefaultGenomeBuild = "GRCh37.p13";
//	private static final String	kDefaultGenomeBuild = "GRCh37";
//	private static final String	kPlusStrand = "+";
	
	private static final String		kLookupFile = "src/test/resources/testData/tabix/00-First1K_GRCh37.tsv.bgz";
	private static final String		kLookupIndexFile = "src/test/resources/testData/tabix/index/00-First1K_GRCh37.ID.idx.h2.db";
	private static final String		kCatalogPath = Variant2JSONPipe.getCatalogPath ();
	private static final String		kCosmicFile 	= kCatalogPath + "/cosmic/v68/CosmicCompleteExport_GRCh37.tsv.bgz";
	 // NOTE: Following is the index and not the catalog
	private static final String		kNCBIFile 		= kCatalogPath + "/dbSNP/142/00-All_GRCh37.tsv.bgz";
	private static final String		kNCBIFileIndex 	= kCatalogPath + "/dbSNP/142/index/00-All_GRCh37.ID.idx.h2.db";
	private static final String		kGenomeFile 	= kCatalogPath + "/1000_genomes/20130502_GRCh37/variants_nodups.v1/ALL.wgs.sites.vcf.tsv.bgz";
	private static final String		kHapMapFile 	= kCatalogPath + "/hapmap/2010-08_phaseII+III/allele_freqs_GRCh37.tsv.bgz";
	// !!! This requires the modified catalog !!!!
	private static final String		kHGNCFile 		= kCatalogPath + //"/hgnc/2015_02_26/hgnc_GRCh37.tsv.bgz";
																	 "/hgnc/20160422_GRCh37.p13/genes.v1/hgnc_complete_set_ensembl_gtf.tsv.bgz";
	private static final String		kHGNCPreFile 	= kCatalogPath + "/NCBIGene/GRCh37_p13/genes.tsv.bgz";
	private static final String		kBGIFile 		= kCatalogPath + "/BGI/hg19/LuCAMP_200exomeFinal.maf_GRCh37.tsv.bgz";
	private static final String		kESPFile 		= kCatalogPath + "/ESP/build37/ESP6500SI_GRCh37.tsv.bgz";
	private static final String		kMirBaseFile 	= kCatalogPath + "/mirbase/release19/hsa_GRCh37.p5.tsv.bgz";
	private static final String		kOMIMFile 		= kCatalogPath + "/../../user/v1/OMIM/2015_10_16/OMIM.10_16_15.tsv.bgz";
//	private static final String		kOMIMFile 		= kCatalogPath + "/omim/2015_10_16/OMIM.10_16_15.tsv.bgz";
//	private static final String		kOMIMFile 		= kCatalogPath + "/omim/2014_01_13/genemap_GRCh37.tsv.bgz";
	private static final String		kUCSCBRFile 	= kCatalogPath + "/ucsc/hg19/wgEncodeDacMapabilityConsensusExcludable_GRCh37.tsv.bgz";
	private static final String[]	kBaseDrill = {"_landmark", "_minBP", "_refAllele", "_altAlleles"};
	private static final String[]	kCosmicDrill = {"Mutation_ID", "Mutation_AA", "Mutation_CDS", "Mutation_GRCh37_strand"};
	private static final String[]	kNCBIDrill = {"INFO.RS", "INFO.dbSNPBuildID", "INFO.SAO", "INFO.SSR"};
	private static final String[]	kGenomeDrill = {"INFO.AFR_AF", "INFO.AMR_AF", "INFO.EAS_AF", "INFO.EUR_AF"};
	private static final String[]	kHapMapDrill = {"CEU.otherallele_freq", "CHB.otherallele_freq", "JPT.otherallele_freq", "YRI.otherallele_freq"};
	private static final String[]	kHGNCDrill = {"Approved_Name", "Approved_Symbol", "Ensembl_Gene_ID", "Entrez_Gene_ID"};
	private static final String[]	kHGNCPreDrill = {"HGNC"}; // This is the name of the "HGNC" field within the NCBIGene catalog that is drilled in preparation for hgnc catalog lookup by "hgnc_id" field
	private static final String[]	kHGNCPostDrill = null;
	private static final String[]	kBGIDrill = {"estimated_minor_allele_freq"};
	private static final String[]	kESPDrill = {"AA._maf", "EA._maf"};
	private static final String[]	kMirBaseDrill = {"ID"};
	private static final String[]	kOMIMDrill = {"name"};
	private static final String[]	kUCSCBRDrill = {"name", "score"};
	private static final String[]	kBaseHeader = {"bior.chrom", "bior.startPos", "Ref", "Alt"};
	private static final String[]	kHGNCPreHeader = {"HGNC.ID"};
	private static final String[]	kCosmicHeader = {"Cosmic.ID", "Cosmic.Amino.Acid.Change", "Cosmic.Change", "Cosmic.Strand"};
	private static final String[]	kNCBIHeader = {"dbSNP.rsID", "dbSNP.build", "dbSNP.SNP.Allele.Origin", "dbSNP.Suspect.Region"};
	private static final String[]	kGenomeHeader = {"kGenomes.AFR.MAF", "kGenomes.AMR.MAF", "kGenomes.EAS.MAF", "kGenomes.EUR.MAF"};
	private static final String[]	kHapMapHeader = {"HapMap.CEU.MAF", "HapMap.CHB.MAF", "HapMap.JPT.MAF", "HapMap.YRI.MAF"};
	private static final String[]	kHGNCHeader = {"HGNC.Approved.Gene.Name", "HGNC.Approved.Gene.Symbol", "HGNC.Ensembl.Gene.ID", 
	                             	               "HGNC.Entrez.Gene.ID"};
	private static final String[]	kBGIHeader = {"BGI200.Danish.MAF"};
	private static final String[]	kESPHeader = {"ESP6500.AFR.MAF", "ESP6500.EUR.MAF"};
	private static final String[]	kMirBaseHeader = {"mirBase.ID"};
	private static final String[]	kOMIMHeader = {"OMIM.Disease"};
//	private static final String[]	kUCSCBRHeader = {"UCSC Blacklisted Region Name", "UCSC Blacklisted Region Score"};
	private static final String[]	kUCSCBRHeader = {"UCSCBR.Name", "UCSCBR.Score"};

	// AFter performing an overlap of the variant against NCBIGene, we drill out "HGNC", then lookup "hgnc_id" in hgnc catalog
	private static final String[]	kHGNCIndexPaths = {"HGNC"}; // {"hgnc_id"};
	private static final String[]	kHGNCIndexFiles = {kCatalogPath + "/hgnc/20160422_GRCh37.p13/genes.v1/index/hgnc_complete_set_ensembl_gtf.hgnc_id.idx.h2.db"};
//	private static final String[]	kHGNCIndexPaths = {"HGNC_ID"};
//	private static final String[]	kHGNCIndexFiles = {kCatalogPath + "/hgnc/2015_02_26/index/hgnc_GRCh37.HGNC_ID.idx.h2.db"};
//	private static final String[]	kHGNCIndexFiles = {kCatalogPath + "/hgnc/2015_02_26/index/hgnc_GRCh37.HGNC.idx.h2.db"};
	
	private static final String[]	kDbSnpSuspectLookup = {"unspecified", "Paralog", "byEST", "Para_EST", "oldAlign", "other"};
	protected static final String[]	kDbSnpClinicalLookup = {"unknown", "untested", "non-pathogenic", "probable-non-pathogenic", 
	                             	                        "probable-pathogenic", "pathogenic", "drug-response", 
	                             	                        "histocompatibility", "other"};
	private static final String[]	kDbSnpAlleleLookup = {"unspecified", "Germline", "Somatic", "Both", "not-tested", 
	                             	                      "tested-inconclusive", "other"};
	
	private static final int	kNCBIRsID = 0;
	private static final int	kNCBIBuildID = kNCBIRsID + 1;
	private static final int	kNCBIAlleleOrigin = kNCBIBuildID + 1;
	private static final int	kNCBISuspect = kNCBIAlleleOrigin + 1;
	
	private static final int	kChrom = 0;
	private static final int	kStartPos = kChrom + 1;
	private static final int	kRef = kStartPos + 1;
	private static final int	kAlt = kRef + 1;
	
	
	/**
	 * Do any needed class setup
	 * @throws IOException 
	 */
	@BeforeClass
	public static void setupClass () throws IOException
	{
		verifyPathsToCatalogs();
		
		gOverlapPaths.add (kHGNCPreFile);
		gOverlapPaths.add (kMirBaseFile);
		gOverlapPaths.add (kOMIMFile);
		gOverlapPaths.add (kUCSCBRFile);
		
		gLookupPaths.add (kHGNCFile);
		
		gLookupDrill.put (kHGNCPreFile, kHGNCPreDrill);
		gLookupCatalogs.put (kHGNCFile, kHGNCPreFile);
		
		int					size = kHGNCIndexPaths.length;
		Map<String, String>	indexMap = new HashMap<String, String> ();
		
		for (int i = 0; i < size; ++i)
			indexMap.put (kHGNCIndexPaths[i], kHGNCIndexFiles[i]);
		
		gLookupIndexes.put (kHGNCFile, indexMap);
		
		
		FieldProcessor	genomesProcessor = new TextCleaner ("[", "]");
		for (String field : kGenomeHeader)
			gFieldProcessorMap.put (field, genomesProcessor);
		
		
		// Clean up Alt, making it normal, possibly delimited, strings
		FieldProcessor	altProcessor = new TextCleaner ("[\"", "\",\"", ":", "\"]");
		gFieldProcessorMap.put (kBaseHeader[kAlt], altProcessor);
		
		
		gFieldProcessorMap.put (kNCBIHeader[kNCBIRsID], new TextWrapper ("rs", null));
		gFieldProcessorMap.put (kNCBIHeader[kNCBIAlleleOrigin], new LookupProcessor (kDbSnpAlleleLookup));
//		gFieldProcessorMap.put (kNCBIHeader[kNCBIClincal], new LookupProcessor (kDbSnpClinicalLookup));
		gFieldProcessorMap.put (kNCBIHeader[kNCBISuspect], new LookupProcessor (kDbSnpSuspectLookup));
		
		
		// Return empty string as NA
		gDefaultProcessor.addLocalBlank ("");
		
		// Set the reference assembly base directory 
		File	refAssemblyBaseDir = new File ("src/test/resources/ref_assembly");
//		File	refAssemblyBaseDir = new File (kCatalogPath + "/ref_assembly");
		RefAssemblyFinder.setRefAssemblyBaseDirStatic (refAssemblyBaseDir.getCanonicalPath ());
	}
	
	
	private static void verifyPathsToCatalogs () throws IOException
	{
		List<String> necessaryCatalogs = new ArrayList<String> ( Arrays.asList (
				kCosmicFile,
				kNCBIFile,
				kGenomeFile,
				kHapMapFile,
				kHGNCFile,
				kHGNCPreFile,
				kBGIFile,
				kESPFile,
				kMirBaseFile,
				kOMIMFile,
				kUCSCBRFile,
				// Indexes:
				kNCBIFileIndex
				));
		necessaryCatalogs.addAll (Arrays.asList (kHGNCIndexFiles));
		
		StringBuilder errors = new StringBuilder ();
		for (String catalog : necessaryCatalogs)
		{
			File	catalogFile = new File (catalog);
			if (!catalogFile.exists ())
				errors.append ("ERROR: Catalog is necessary for tests: " + catalogFile.getAbsolutePath () + "\n");
						
			File	tabixFile = new File (catalog + ".tbi");
			// Omim and HGNC don't have any chromosomes or positions in the catalog - they don't
			// need tabix files
			// Use File.getAbsolutePath() here as File.getCanonicalPath() was throwing "java.io.IOException: Invalid argument"
			//   when connecting to /data5 on RCF sometimes (was this due to a later Java version???)
			boolean isOmimOrHgnc = catalogFile.getAbsolutePath().toLowerCase().contains("/hgnc/") || 
									catalogFile.getAbsolutePath().toLowerCase().contains("/omim/");
			// If the file was a catalog (ends with .bgz), and is NOT OMIM or HGNC, then make sure
			// it has a tabix file
			if (catalogFile.getName ().endsWith (".bgz") && !isOmimOrHgnc && !tabixFile.exists ())
				errors.append ("ERROR: Tabix file for catalog is necessary for tests: " + tabixFile.getAbsolutePath() + "\n");
		}
		
		if (errors.length () > 0)
		{
			System.err.println (errors);
			System.err.println ("You can mount the RCF drives on your Mac by doing the following:");
			System.err.println ("1) Create a link to the location where the directories will be mounted (only needs to be done once)");
			System.err.println ("     cd /");
			System.err.println ("     sudo ln -s /Volumes/data5 /data5");
			System.err.print ("2) Mount the RCF folders by opening a Finder window, then clicking 'Go' -> ");
			System.err.println ("'Connect to server...', then entering this path");
			System.err.println ("     cifs://rcfcluster-cifs/data5");
			System.err.println ("The drive should now be mounted to the /data5 path");
			Assert.fail (errors.toString ());
		}
	}


	/**
	 * Do any needed setup.  If History still statically stored MetaData, this would create it
	 */
	@Before
	public void setup ()
	{
		// Do any needed setup here
	}
	
	
	/**
	 * Do any needed tear down.  If History still statically stored MetaData, this would tear it down
	 */
	@After
	public void teardown()
	{
		// Do any needed tear down here
	}
	
	
	/**
	 * Test that validates that the bim file with all its issues will be processed successfully
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testJSONGenerationBimFile () throws IOException, SQLException
	{
		String						bimFile = "src/test/resources/testData/VariantToJson/bb.test.bim";
		String						bimResultsFile = "src/test/resources/testData/VariantToJson/bimResults.txt";
		Pipeline<String, String>	thePipe = getVariant2JsonPipeline ("testJSONGenerationBimFile");
		FileLineIterator			input = new FileLineIterator (bimFile);
		List<String>				expected = FileUtils.readLines (new File (bimResultsFile));
		
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
    	
    	System.out.println("\n\n\nExpected: ---------------------------------------------");
    	PipeTestUtils.printLines (expected);
     	System.out.println ("\n\n\nActual: ---------------------------------------------");
    	PipeTestUtils.printLines (actual);

    	PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	@Ignore("This is just testing additional annotations attached to the end of the line, and not really the Variant2JSONPipe")
	/**
	 * Test that validates that the bim file with all its issues will be processed successfully
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testBimFileAnnotation () throws IOException, SQLException
	{
		String					bimFile = "src/test/resources/testData/VariantToJson/bb.test.bim";
		String					bimResultsFile = "src/test/resources/testData/VariantToJson/bimAnnotatedResults.txt";
		String					lookupFile = kNCBIFile;
		String					lookupIndexFile = kNCBIFileIndex;
		Metadata				metadata = new Metadata ("testBimFileAnnotation");
		Pipe<String, String>	addHeader = new MakeFirstLineHeaderPipe();
		Pipe<String, History>	setup = new HistoryInPipe (metadata);
		LookupPipe				lookup = new LookupPipe (lookupFile, lookupIndexFile);
		List<String>			ctgPaths = Arrays.asList(
									kHGNCFile,
									kCosmicFile,
									kNCBIFile,
									kGenomeFile,
									kHapMapFile,
									kBGIFile,
									kESPFile,
									kMirBaseFile,
									kOMIMFile,
									kUCSCBRFile
								);

		Variant2JSONPipe			logic = new Variant2JSONPipe (lookup, null);
		Pipeline<String, History>	preLogic = new Pipeline<String, History> (addHeader, setup);
		Pipeline<History, History>	annotate = makeFindPipes (ctgPaths, logic);
		Pipe<History, String>		postLogic = new HistoryOutPipe ();
		Pipeline<String, String>	thePipe = new Pipeline<String, String> (preLogic, annotate, postLogic);
		
		FileLineIterator	input = new FileLineIterator (bimFile);
		List<String>		expected = FileUtils.readLines (new File (bimResultsFile));
		
		thePipe.setStarts (input);
		
    	List<String>	actual = PipeTestUtils.getResults (thePipe);
    	
//    	System.out.print (TabixReader.getBufferCount ());
//    	System.out.println (" Line Buffers created");
//    	System.out.print (TabixReader.getBufferCallCount ());
//    	System.out.println (" Line Buffer calls");
//    	System.out.print (TabixReader.getMaxBufferCapacity ());
//    	System.out.println (" Final buffer size");
    	System.out.println("\n\n\nExpected: ---------------------------------------------");
    	PipeTestUtils.printLines (expected);
     	System.out.println ("\n\n\nActual: ---------------------------------------------");
    	PipeTestUtils.printLines (actual);
		PipeTestUtils.assertListsEqual (expected, actual);
	}
	
	
	@Ignore("This is just testing additional annotations attached to the end of the line (and drilled columns), and not really the Variant2JSONPipe")
	/**
	 * Test that validates that the bim file with all its issues will be processed successfully
	 * 
	 * @throws IOException	For any file problems
	 */
	@Test
	public void testBimFileAnnotations () throws IOException, SQLException
	{
		String					bimFile = "src/test/resources/testData/VariantToJson/bb.test.bim";
		String					bimResultsFile = "src/test/resources/testData/VariantToJson/bimOldDrilledResults.txt";
		String					lookupFile = kNCBIFile;
		String					lookupIndexFile = kNCBIFileIndex;
		Metadata				metadata = new Metadata ("testBimFileAnnotations");
		Pipe<String, String>	addHeader = new MakeFirstLineHeaderPipe ();
		Pipe<String, History>	setup = new HistoryInPipe (metadata);
		LookupPipe				lookup = new LookupPipe (lookupFile, lookupIndexFile);
		
		List<String>			catalogPaths = Arrays.asList(
									kHGNCFile,
									kCosmicFile,
									kNCBIFile,
									kGenomeFile,
									kHapMapFile,
									kBGIFile,
									kESPFile,
									kMirBaseFile,
									kOMIMFile,
									kUCSCBRFile
									);
		
		List<String[]>			drillPaths = Arrays.asList(
									kHGNCDrill,
									kHGNCPostDrill,
									kCosmicDrill,
									kNCBIDrill,
									kGenomeDrill,
									kHapMapDrill,
									kBGIDrill,
									kESPDrill,
									kMirBaseDrill,
									kOMIMDrill,
									kUCSCBRDrill,
									kBaseDrill
									);
		
		Variant2JSONPipe			logic = new Variant2JSONPipe (lookup, null);
		Pipeline<String, History>	preLogic = new Pipeline<String, History> (addHeader, setup);
		Pipeline<History, History>	annotate = makeFindPipes (catalogPaths, logic, true);
		Pipeline<History, History>	postLogic = makeDrillPipes (drillPaths);
		Pipeline<String, History>	thePipe = new Pipeline<String, History> (preLogic, annotate, postLogic);
		
		FileLineIterator	input = new FileLineIterator (bimFile);
		
		thePipe.setStarts (input);
		
    	List<History>	actual = getResults (thePipe);
    	
		List<String[]>	drillHeaders = Arrays.asList(
									kBaseHeader,
									kHGNCPreHeader,
									kHGNCHeader,
									kCosmicHeader,
									kNCBIHeader,
									kGenomeHeader,
									kHapMapHeader,
									kBGIHeader,
									kESPHeader,
									kMirBaseHeader,
									kOMIMHeader,
									kUCSCBRHeader
									);
    	
		List<String>		output = dumpHistory (actual, drillHeaders, kBaseHeader);
		List<String>		expected = FileUtils.readLines (new File (bimResultsFile));
		
//    	System.out.print (TabixReader.getBufferCount ());
//    	System.out.println (" Line Buffers created");
//    	System.out.print (TabixReader.getBufferCallCount ());
//    	System.out.println (" Line Buffer calls");
//    	System.out.print (TabixReader.getMaxBufferCapacity ());
//    	System.out.println (" Final buffer size");
//    	PipeTestUtils.printLines (output);
		PipeTestUtils.assertListsEqual (expected, output);
	}
	
	
	/**
	 * Parse a List of {@linkplain History} into Strings, using the passed in headers as well as the 
	 * History's "Original Headers"
	 * 
	 * @param theHistory	List of History to parse.  Must not be null, if empty will return 
	 * empty list
	 * @param addedHeaders	Headers added for the annotations, in the order of the annotations
	 * @param lastToFirst	Headers for items appearing at the end, which need to get moved starting 
	 * at {@code firstCol}.  Must be first in addedHeaders
	 * @return	List of Strings, one per line of output
	 */
	private List<String> dumpHistory (List<History> theHistory, List<String[]> addedHeaders, String[] lastToFirst)
	{
		List<String>	results = new ArrayList<String> ();
		int				numCols = -1;
		int				firstCol = -1;
		StringBuilder	outLine = new StringBuilder (1000);
		FieldProcessor[]	processors = null;
		
		if (theHistory.isEmpty ())
			return results;
		
		int		numAddedCols = addHeader (theHistory.get (0), addedHeaders, results, outLine);
		int		numLast = (lastToFirst == null) ? 0 : lastToFirst.length;
		boolean	first = true;
		
		for (History history : theHistory)
		{
			if (first)
			{
				numCols = history.size ();
				firstCol = numCols - numAddedCols;
				processors = getProcessors (firstCol, numCols, addedHeaders, lastToFirst);
				numCols -= numLast;	// Won't get these in the normal way
				numLast += numCols;	// Make this now the real end
			}
			
			first = true;
			for (int i = 0; i < firstCol; ++i)
			{
				if (first)
					first = false;
				else
					outLine.append ('\t');
				outLine.append (processors[i].process (history.get (i)));
			}
			
			for (int i = numCols; i < numLast; ++i)
			{
				outLine.append ('\t');
				outLine.append (processors[i].process (history.get (i)));
			}
			
			for (int i = firstCol; i < numCols; ++i)
			{
				outLine.append ('\t');
				outLine.append (processors[i].process (history.get (i)));
			}
			
			results.add (outLine.toString ());
			outLine.delete (0, outLine.length ());
		}
		
		return results;
	}
	
	
	/**
	 * Determine the {@linkplain FieldProcessor} for each field
	 * 
	 * @param firstCol		First added column
	 * @param numCols		Size of returned array
	 * @param addedHeaders	Where to get the headers from to do the {@linkplain FieldProcessor} lookup
	 * @param lastToFirst	Headers for items appearing at the end, whose output starts at 
	 * {@code firstCol}, but which is actually at the end.  Must be first in addedHeaders
	 * @return	Array of length {@code numCols}, each position having a processor
	 */
	private FieldProcessor[] getProcessors (int firstCol, int numCols, List<String[]> addedHeaders, String[] lastToFirst)
	{
		FieldProcessor[]	processors = new FieldProcessor[numCols];
		int					i;
		
		for (i = 0; i < firstCol; ++i)
			processors[i] = gDefaultProcessor;
		
		for (String[] headers : addedHeaders)
		{
			if (headers == lastToFirst)
				continue;
			
			for (String header : headers)
			{
				FieldProcessor	processor = gFieldProcessorMap.get (header);
				if (processor == null)
					processor = gDefaultProcessor;
				
				processors[i] = processor;
				++i;
			}
		}
		
		if (lastToFirst != null)
		{
			for (String header : lastToFirst)
			{
				FieldProcessor	processor = gFieldProcessorMap.get (header);
				if (processor == null)
					processor = gDefaultProcessor;
				
				processors[i] = processor;
				++i;
			}
		}
		
		return processors;
	}
	
	
	/**
	 * Add history headers to {@code results}, using {@code outLine} as scratch space
	 * 
	 * @param history		{@linkplain History} object holding the headers to start with
	 * @param addedHeaders	{@linkplain List} of String[] holding all the other headers to add
	 * @param results		{@linkplain List} of String to add the headers to
	 * @param outLine		Scratch space, will be empty on entry, and on exit
	 * @return	Number of header lines added from addedHeaders
	 */
	private int addHeader (History history, List<String[]> addedHeaders, List<String> results, StringBuilder outLine)
	{
		List<String>	headers = history.getMetaData ().getOriginalHeader ();
		String			headerLine = null;
		
		for (String aHeader : headers)
		{
			if (headerLine != null)
				results.add (headerLine);	// Had a header w/ more than one line, put breaks between the lines
			
			headerLine = aHeader;
		}
		
		if (headerLine != null)
			outLine.append (headerLine);
		
		int	numAddedCols = 0;
		
		for (String[] theHeaders : addedHeaders)
		{
			numAddedCols += theHeaders.length;
			for (String theHeader : theHeaders)
			{
				outLine.append ("\t");
				outLine.append (theHeader);
			}
		}
		
		results.add (outLine.toString ());
		outLine.delete (0, outLine.length ());
		
		return numAddedCols;
	}
	
	
	/**
	 * Create a {@linkplain History} to {@linkplain History} {@linkplain Pipeline} from a List of paths as Strings
	 * 
	 * @param paths	{@linkplain List} of paths to database files
	 * @param start	{@linkplain Pipe} from History to History to go at beginning of Pipeline, or null
	 * @return	{@linkplain History} to {@linkplain History} {@linkplain Pipeline}, possibly empty, never null
	 * @throws IOException 
	 */
	private Pipeline<History, History> makeFindPipes (List<String> paths, Pipe<History, History> start) throws IOException, SQLException
	{
		return makeFindPipes (paths, start, false);
	}
	
	
	/**
	 * Create a {@linkplain History} to {@linkplain History} {@linkplain Pipeline} from a List of paths as Strings
	 * 
	 * @param paths			{@linkplain List} of paths to database files
	 * @param start			{@linkplain Pipe} from History to History to go at beginning of Pipeline, or null
	 * @param reverseList	If true, add files as pipes in reverse order, so parse order matches path order
	 * @return	{@linkplain History} to {@linkplain History} {@linkplain Pipeline}, possibly empty, never null
	 * @throws IOException 
	 */
	private Pipeline<History, History> makeFindPipes (List<String> paths, Pipe<History, History> start, boolean reverseList) throws IOException, SQLException
	{
		Pipeline<History, History>	annotate = new Pipeline<History, History> ();
		
		if (start != null)
			annotate.addPipe (start);
		
		if (paths == null)
			return annotate;
		
		if (reverseList)
		{
			int				numPaths = paths.size ();
			List<String>	hold = new ArrayList<String> (numPaths);
			for (int i = numPaths - 1; i >= 0; --i)
				hold.add (paths.get (i));
			
			paths = hold;
		}
		
		int	historyPosition = -1;
		for (String path : paths)
		{
			if (gLookupPaths.contains (path))
			{
				String				overlapPath = gLookupCatalogs.get (path);
				String[]			drillField = gLookupDrill.get (overlapPath);
				Map<String, String>	indexMap = gLookupIndexes.get (path);
				
				if ((overlapPath == null) || (drillField == null) || (indexMap == null))
					continue;	// Do nothing, add nothing
				
				int		numDrills = drillField.length;
				String	indexFile = indexMap.get (drillField[numDrills - 1]);	// Use last drill
				if (indexFile == null)
					continue;	// Do nothing, add nothing
				
				annotate.addPipe (new OverlapPipe (overlapPath, historyPosition));
				annotate.addPipe (new DrillPipe (false, drillField));
				annotate.addPipe (new LookupPipe (path, indexFile));
				historyPosition -= numDrills;	// Each drill adds to end of history, pushing target back one
			}
			else if (gOverlapPaths.contains (path))
				annotate.addPipe (new OverlapPipe (path, historyPosition));
			else
				annotate.addPipe (new SameVariantPipe (path, false, historyPosition));
			--historyPosition;	// Each pipe adds to end of history, pushing target back one
		}
		
		return annotate;
	}
	
	
	/**
	 * Create a {@linkplain History} to {@linkplain History} {@linkplain Pipeline} from a List of 
	 * JSON "Paths" to drill for 
	 * 
	 * @param paths	{@linkplain List} of JSON paths of fields we're interested in, in reverse order 
	 * that their databases were added to the annotation logic
	 * @param end	{@linkplain Pipe} from History to String to go at end of Pipeline
	 * @return	{@linkplain History} to {@linkplain History} {@linkplain Pipeline}, possibly empty, never null
	 * @throws IOException 
	 */
	protected Pipeline<History, History> makeDrillPipes (List<String[]> paths) throws IOException
	{
		Pipeline<History, History>	driller = new Pipeline<History, History> ();
		
		if (paths != null)
		{
			int	drillColumn = -1;
			for (String[] drillSet : paths)
			{
				if (drillSet == null)	// Way to force a skip back
				{
					--drillColumn;
				}
				else
				{
					driller.addPipe (new DrillPipe (false, drillSet, drillColumn));
					drillColumn -= drillSet.length;
				}
			}
		}
		
		return driller;
	}
	
	
	/**
	 * Create a list of {@linkplain SameVariantPipe}s from a List of paths as Strings
	 * 
	 * @param paths	{@linkplain List} of paths to database files
	 * @return	List of SameVariantPipe, possibly empty, never null
	 * @throws IOException 
	 */
	protected List<TabixParentPipe> makeFindPipes (List<String> paths) throws IOException
	{
		List<TabixParentPipe>	chain = new ArrayList<TabixParentPipe> ();
		
		if (paths == null)
			return chain;
		
		int	historyPosition = -1;
		for (String path : paths)
		{
			chain.add (new SameVariantPipe (path, false, historyPosition));
			--historyPosition;	// Each pipe adds to end of history, pushing target back one
		}
		
		return chain;
	}
	
	
	/**
	 * Find the valid ranges for M and XY
	 * @throws IOException 
	 * @throws NumberFormatException 
	 * @throws AssemblyNotSupportedException 
	 */
//	@Test
	public void testBoundaries () throws NumberFormatException, IOException, AssemblyNotSupportedException
	{
		RefBasePairLookup	genome = Variant2JSONPipe.getRefAssemblyFinder ("GRCh37");
		int					end = 1000000;
		
		testChromosome (genome, "M", 1, 16571);
		testChromosome (genome, "XY", 1, end);
//		testChromosome (genome, "MT", 1, 16571);
		testChromosome (genome, "Y", 1, end);
		testChromosome (genome, "X", 1, end);
	}
	
	
	/**
	 * Find which locations in a chromosome return valid bases
	 * 
	 * @param genome	{@linkplain RefBasePairLookup} for getting the bases
	 * @param chrom		Chromosome of interest
	 * @param start		Where to start looking in the Chromosome
	 * @param end		Where to stop looking in the Chromosome
	 * @throws NumberFormatException
	 * @throws IOException 
	 */
	private void testChromosome (RefBasePairLookup genome, String chrom, int start, int end) throws NumberFormatException, IOException
	{
		boolean	isDot = false;
		boolean	isN = false;
		
		System.out.print ("Chromosome ");
		System.out.print (chrom);
		System.out.print (" from ");
		System.out.print ("" + start);
		System.out.print (" from ");
		System.out.println ("" + end);
		for (int i = start; i <= end; ++i)
		{
			String	pos = "" + i;
			String	ref = genome.getBasePairAtPosition (chrom, pos, pos);
			
			if (ref.equals ("."))
			{
				if (!isDot)
				{
					if (isN)
					{
						isN = false;
						System.out.print ("Ended N at ");
						System.out.println ("" + i);
					}
					
					isDot = true;
					System.out.print ("Started invalid at ");
					System.out.println ("" + i);
				}
			}
			else if (ref.equals ("N"))
			{
				if (!isN)
				{
					if (isDot)
					{
						isDot = false;
						System.out.print ("Ended invalid at ");
						System.out.println ("" + i);
					}
					
					isN = true;
					System.out.print ("Started N at ");
					System.out.println ("" + i);
				}
			}
			else
			{
				if (isDot)
				{
					isDot = false;
					System.out.print ("Ended invalid at ");
					System.out.println ("" + i);
				}
				
				if (isN)
				{
					isN = false;
					System.out.print ("Ended N at ");
					System.out.println ("" + i);
				}
			}
		}
		
		if (isDot)
		{
			isDot = false;
			System.out.print ("Invalid went to ");
			System.out.println ("" + end);
		}
		
		if (isN)
		{
			isN = false;
			System.out.print ("N went to ");
			System.out.println ("" + end);
		}
	}
	
	
	/**
	 * Create the {@linkplain Pipeline} to be used in the tests
	 *  
	 * @param meta	Name of the test being run
	 * @return	A String in, String out {@linkplain Pipeline}
	 */
	private Pipeline<String, String> getVariant2JsonPipeline (String meta) throws IOException, SQLException
	{
		String						lookupFile = kLookupFile;
		String						lookupIndexFile = kLookupIndexFile;
		Metadata					metadata = new Metadata (meta);
		Pipe<String, String>		addHeader = new MakeFirstLineHeaderPipe ();
		Pipe<String, History>		setup = new HistoryInPipe (metadata);
		Pipe<String, History>		preLogic = new Pipeline<String, History> (addHeader, setup);
		LookupPipe					lookup = new LookupPipe (lookupFile, lookupIndexFile);
		
		Variant2JSONPipe			logic = new Variant2JSONPipe (lookup, null);
		Pipe<History, String>		postLogic = new HistoryOutPipe ();
		Pipeline<String, String>	thePipe = new Pipeline<String, String> (preLogic, logic, postLogic);
		
		return thePipe;
	}
	
	
	/**
	 * Take a {@linkplain Pipe} that emits {@linkplain History}, and run it to completion, 
	 * saving all its results to a List
	 * 
	 * @param pipe	Pipe to use.  If null will return an empty list
	 * @return	List of History, possibly empty, never null
	 */
	public static <T> List<History> getResults (Pipe<T, History> pipe)
	{
		List<History> results = new ArrayList<History> ();
		
		if (pipe != null)
		{
			while (pipe.hasNext ())
			{
				Object	obj = pipe.next ();
				
				if (obj instanceof History)
					results.add ((History) obj);
				else
					results.add (new History (obj.toString ()));
			}
		}
		
		return results;
	}
	
}
