package edu.mayo.bior.cli.func;

import static org.junit.Assert.assertEquals;

import java.io.File;
import java.io.IOException;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.cli.cmd.Variant2JSONCommand;
import edu.mayo.pipes.util.BiorProperties;
import edu.mayo.pipes.util.BiorProperties.Key;

/** NOTE: This will require access to the ref assembly and dbSNP 142 catalog files */
public class Variant2JSONCommandITCase extends BaseFunctionalTest
{
	
	@Rule
	public TemporaryFolder mTempFolder = new TemporaryFolder();
	
	private File mTempDir;
	
	@Before
	public void beforeEach() throws IOException {
		mTempFolder.create();
		mTempDir = mTempFolder.newFolder();
		
		// Set the BIOR_LITE_HOME environment variable and the PATH to it
		setBiorToolkitCmdsRequired(true);
	}
	
	private String getInput() {
		return concat("id", "chrom", "pos", 		"allele1", 	"allele2") + "\n"
			+  concat("a",  "1",     "101046742", 	"C", 		"T") + "\n"
			+  concat("a",  "1",     "101077318", 	"T", 		"C") + "\n"
			+  concat("a",  "1",     "106440499", 	"T", 		"C") + "\n";
	}
	
	private String getOutput37() {
		return "##BIOR=<ID='bior.ToTJson',Operation='bior_variant_to_tjson',DataType='JSON',ShortUniqueName='ToTJson'>".replace("'", "\"") + "\n"
			+  concat("#id", "chrom", 	"pos", 		"allele1", 	"allele2", 	"bior.ToTJson") + "\n"
			+  concat("a",   "1",		"101046742","C",		"T",		"{'_landmark':'1','_refAllele':'C','_altAlleles':['T'],'_minBP':101046742,'_maxBP':101046742}".replace("'", "\"")) + "\n"
			+  concat("a",   "1",		"101077318","T",		"C",		"{'_landmark':'1','_refAllele':'A','_altAlleles':['G'],'_minBP':101077318,'_maxBP':101077318,'WARNING':'Neither of the alleles supplied matched the reference, so swapped strands'}".replace("'", "\"")) + "\n"
			+  concat("a",   "1", 		"106440499","T",		"C",		"{'_landmark':'1','_refAllele':'C','_altAlleles':['T'],'_minBP':106440499,'_maxBP':106440499}".replace("'", "\"")) + "\n";
	}

	private String getOutput38() {
		// NOTE: For some reason, the first row's position comes back with a REF allele of 'c' instead of 'C'
		return "##BIOR=<ID='bior.ToTJson',Operation='bior_variant_to_tjson',DataType='JSON',ShortUniqueName='ToTJson'>".replace("'", "\"") + "\n"
				+  concat("#id", "chrom", 	"pos", 		"allele1", 	"allele2", 	"bior.ToTJson") + "\n"
				+  concat("a",   "1",		"101046742","C",		"T",		"{'_landmark':'1','_refAllele':'c','_altAlleles':['T'],'_minBP':101046742,'_maxBP':101046742}".replace("'", "\"")) + "\n"
				+  concat("a",   "1",		"101077318","T",		"C",		"{'_landmark':'1','_refAllele':'T','_altAlleles':['C'],'_minBP':101077318,'_maxBP':101077318}".replace("'", "\"")) + "\n"
				+  concat("a",   "1", 		"106440499","T",		"C",		"{'_landmark':'1','_refAllele':'T','_altAlleles':['C'],'_minBP':106440499,'_maxBP':106440499}".replace("'", "\"")) + "\n";
	}

	
	@Test
	/** Test default GRCh37 build and dbSNP GRCh37  */
	public void testDefaultsGrch37() throws IOException {
		String inputData = getInput();

		System.out.println("\n\n\n\nBIOR_LITE_HOME: " + System.getenv("BIOR_LITE_HOME"));
		System.out.println("PATH: " + System.getenv("PATH") + "\n\n\n\n");
		
		// Run the command with the -v flag to get the version
		CommandOutput out0 = runCmdApp(
				inputData,
				new Variant2JSONCommand(),
				"bior_variant_to_tjson",
				"-v"
				);
		System.out.println("STDERR: " + out0.stderr);
		System.out.println("STDOUT: " + out0.stdout);
		System.out.println("exit:   " + out0.exit);
			
		
		CommandOutput out = runCmdApp(
				inputData,
				new Variant2JSONCommand(),
				"bior_variant_to_tjson"
				);
		
		String warningNotUsingBuildFlag = "\nWARNING!: Specifying the --build flag is highly recommended to ensure input data matches the correct ref assembly!\n\n";
		assertEquals(warningNotUsingBuildFlag, out.stderr);
		assertEquals(getOutput37(), out.stdout);
		assertEquals("Error: Exit code should be 0", out.exit, 0);
	}

	
	@Test
	/** Test "GRCh37" build and dbSNP GRCh37  */
	public void testGrch37Build() throws IOException {
		String inputData = getInput();

		CommandOutput out = runCmdApp(
				inputData,
				new Variant2JSONCommand(),
				"bior_variant_to_tjson",
				"-B", "GRCh37"
				);
		
		assertEquals("", out.stderr);
		assertEquals(getOutput37(), out.stdout);
		assertEquals("Error: Exit code should be 0", out.exit, 0);
	}

	@Test
	/** Test "hg19" build (for GRCh37) and dbSNP GRCh37  */
	public void testHg19Build() throws IOException {
		String inputData = getInput();

		CommandOutput out = runCmdApp(
				inputData,
				new Variant2JSONCommand(),
				"bior_variant_to_tjson",
				"-B", "hg19"
				);
		
		assertEquals("", out.stderr);
		assertEquals(getOutput37(), out.stdout);
		assertEquals("Error: Exit code should be 0", out.exit, 0);
	}

	
	@Test
	/** Test GRCh38-based ref assembly (Specify "GRCh38" as the build) and dbSNP, and reading input from a file */
	public void testGrch38() throws IOException {
		String inputData = getInput();
		File inputFile = new File(mTempDir, "my.txt");
		FileUtils.write(inputFile, inputData);
		
		BiorProperties props = new BiorProperties();
		String catalogBaseDir = props.get(Key.fileBase);
		String dbsnpPath = catalogBaseDir + "dbSNP/142_GRCh38/variants_nodups.v1/00-All.vcf.tsv.bgz";

		CommandOutput out = runCmdApp(
				inputData, 
				new Variant2JSONCommand(), 
				"bior_variant_to_tjson",
				"-f", inputFile.getCanonicalPath(),
				"-B", "GRCh38",
				"-d", dbsnpPath
				);
		
		assertEquals("", out.stderr);
		assertEquals(getOutput38(), out.stdout);
		assertEquals("Error: Exit code should be 0", out.exit, 0);
	}
	
	
	@Test
	/** Test GRCh38-based ref assembly (Specify "hg38" as the build) and dbSNP, and reading input from a file */
	public void testHg38() throws IOException {
		String inputData = getInput();
		File inputFile = new File(mTempDir, "my.txt");
		FileUtils.write(inputFile, inputData);
		
		BiorProperties props = new BiorProperties();
		String catalogBaseDir = props.get(Key.fileBase);
		String dbsnpPath = catalogBaseDir + "dbSNP/142_GRCh38/variants_nodups.v1/00-All.vcf.tsv.bgz";

		CommandOutput out = runCmdApp(
				inputData, 
				new Variant2JSONCommand(), 
				"bior_variant_to_tjson",
				"-f", inputFile.getCanonicalPath(),
				"-B", "hg38",
				"-d", dbsnpPath
				);
		
		assertEquals("", out.stderr);
		assertEquals(getOutput38(), out.stdout);
		assertEquals("Error: Exit code should be 0", out.exit, 0);
	}
}
