package edu.mayo.bior.cli.func;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.pipeline.createcatalog.TabixCmd;
import edu.mayo.exec.Command;

/** Test the portion of the bior scripts that check the java version for minimum required java version */
public class JavaVersionCheckITCase {

	@Rule
	public TemporaryFolder mTempFolder = new TemporaryFolder();
	
	private File mTempDir;
	
	@Before
	public void beforeEach() throws IOException {
		mTempFolder.create();
		mTempDir = mTempFolder.newFolder();
	}
	
	
	@Test
	public void testOracleJdk() throws IOException, InterruptedException {
		File javaFake = getJavaFakeExec(/*isOracleJdk=*/true, /*minorVersion=*/"8");
		CommandOutput out = exec(javaFake);
		
		assertEquals(0,   out.exit);
		assertEquals("", out.stderr);
		String biorDrillHelp = FileUtils.readFileToString(new File("src/test/resources/help/bior_drill.help"));
		assertEquals(biorDrillHelp, out.stdout);
	}
	
	@Test
	/** Note: There was a bug here where trying to extract the java version would throw this error:
	 *    "bin/bior_drill: line 31: [: -lt: unary operator expected"
	 *  because it tried to grep for the line containing "java version", but openJdk returns "openjdk version",
	 *  resulting in a blank version and minor version coming back, which causes the 'if' statement to think it only has one arg
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public void testOpenJdk() throws IOException, InterruptedException {
		File javaFake = getJavaFakeExec(/*isOracleJdk=*/false, /*minorVersion=*/"8");
		CommandOutput out = exec(javaFake);
		
		assertEquals(0,   out.exit);
		assertEquals("", out.stderr);
		String biorDrillHelp = FileUtils.readFileToString(new File("src/test/resources/help/bior_drill.help"));
		assertEquals(biorDrillHelp, out.stdout);
	}
	
	@Test
	public void minorVersionTooLow() throws IOException, InterruptedException {
		File javaFake = getJavaFakeExec(/*isOracleJdk=*/true, /*minorVersion=*/"4");
		CommandOutput out = exec(javaFake);
		
		assertEquals(1,   out.exit);
		final String EXPECTED_ERROR_MSG = "Invalid Java version 1.4.0_67.  Java 1.6 or higher is required." + "\n"
										+ "You can check your java version by running: java -version" + "\n";
		assertEquals("", out.stdout);
		assertEquals(EXPECTED_ERROR_MSG, out.stderr);
	}
	
	@Test
	public void minorVersion10() throws IOException, InterruptedException {
		File javaFake = getJavaFakeExec(/*isOracleJdk=*/true, /*minorVersion=*/"10");
		CommandOutput out = exec(javaFake);
		
		assertEquals(0,   out.exit);
		assertEquals("", out.stderr);
		String biorDrillHelp = FileUtils.readFileToString(new File("src/test/resources/help/bior_drill.help"));
		assertEquals(biorDrillHelp, out.stdout);
	}
	
	
	
	
	//================================================================================================================
	/**
	 * Create a java script that will print a version if required, then call the other java under the covers
	 * @param isOracleJdk  If true, then use the Oracle version string, else use openJdk
	 * @param minorVersion 5,6,7,8,9,10 etc
	 * @throws IOException 
	 */
	private File getJavaFakeExec(boolean isOracleJdk, String minorVersion) throws IOException {
		File javaFileBefore = TabixCmd.getFromPath("java");
		List<String> versionOracleJdk =  Arrays.asList(
				//"java version \"1.7.0_67\"",
				"java version \"1." + minorVersion + ".0_67\"",
				//"Java(TM) SE Runtime Environment (build 1.7.0_67-b01)", 
				"Java(TM) SE Runtime Environment (build 1." + minorVersion + ".0_67-b01)", 
				"Java HotSpot(TM) 64-Bit Server VM (build 24.65-b04, mixed mode)"
				);

		List<String> versionOpenJdk =  Arrays.asList(
				"openjdk version \"1." + minorVersion + ".0_91\"",
				"OpenJDK Runtime Environment (Zulu 7.12.0.3-linux64) (build 1." + minorVersion + ".0_91-b15)",
				"OpenJDK 64-Bit Server VM (Zulu 7.12.0.3-linux64) (build 24.91-b15, mixed mode)"
				);

		StringBuilder javaExec = new StringBuilder(
				"if [ \"$2\" == \"-version\" ] ; then" + "\n");
		for( String versionLine : (isOracleJdk ? versionOracleJdk : versionOpenJdk) ) {
				javaExec.append("    echo '" + versionLine + "'" + "\n");
		}
		javaExec.append("else" + "\n"
					+   "    " + javaFileBefore.getCanonicalPath() + " $*" + "\n"
				    +   "fi" + "\n");
		
		File javaFileNew = new File(mTempDir, "java");
		FileUtils.writeStringToFile(javaFileNew, javaExec.toString());
		javaFileNew.setExecutable(true);
		return javaFileNew;
	}
	
	private CommandOutput exec(File javaFakeExec) throws IOException, InterruptedException {
		// Add our fake java exec to the front of the path
		Map<String,String> envVars = new HashMap<String,String>(System.getenv());
		String biorHome = getBiorHome();
		String biorBin = biorHome + "/bin";
		String path = javaFakeExec.getParentFile().getCanonicalPath() + ":"
				+ biorBin + ":"
				+ envVars.get("PATH");
		envVars.put("PATH", path);
		envVars.put("BIOR_LITE_HOME", getBiorHome());
		String cmdArgs[] = new String[] { biorBin + "/bior_drill", "-h" };
		Command cmd = new Command(cmdArgs, envVars, true);
		cmd.execute();
		
		CommandOutput out = new CommandOutput();
		out.exit = cmd.getExitCode();
		out.stderr = cmd.getStderr();
		out.stdout = cmd.getStdout();
		
		return out;
	}
	
	private String getBiorHome() throws IOException {
		File targetFolder = new File("target");
		for (File f: targetFolder.listFiles()) {
			if (f.isDirectory() && (f.getName().startsWith("bior-toolkit")) && ! f.getName().endsWith("-distribution")) {
				return f.getCanonicalPath();
			}
		}
		throw new IOException("bior-toolkit dir not found");
	}
}
