package edu.mayo.bior.cli.func;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import edu.mayo.bior.buildcatalog.BuildCatalogTestUtils;

import org.apache.commons.io.FileUtils;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import com.ibm.icu.text.SimpleDateFormat;

import edu.mayo.bior.buildcatalog.BuildCatalog;
import edu.mayo.bior.buildcatalog.BuildCatalogStepInputException;
import edu.mayo.bior.buildcatalog.BuildInfo;
import edu.mayo.bior.buildcatalog.BuildInfoKey;
import edu.mayo.bior.buildcatalog.BuildStepKey;
import edu.mayo.bior.buildcatalog.MergeMetadataStep;
import edu.mayo.bior.buildcatalog.MergeMetadataStepTest;
import edu.mayo.bior.catalog.HumanBuildAssembly;
import edu.mayo.bior.cli.cmd.BuildCatalogCommand;
import edu.mayo.bior.cli.cmd.OverlapPipelineCommand;
import edu.mayo.bior.pipeline.createcatalog.TabixCmd;


public class BuildCatalogCommandITCase extends BaseFunctionalTest
{
   private File mTempDir;
   private File mTargetDir;
   private File mBuildInfoFile;

   private Map<String, String> mBuildInfoMap = new HashMap<String, String>();
   private BuildInfo mBuildInfo = null;

   
   private PrintStream   mSysoutOrig = System.out;
   private PrintStream   mSyserrOrig = System.err;
   private ByteArrayOutputStream mSysoutBytes;
   private ByteArrayOutputStream mSyserrBytes;
   private PrintStream mSysoutTempStream;
   private PrintStream mSyserrTempStream;
   
   
   class PropertiesFilesContents {
	   public String columnsTsv;
	   public String columnsBlacklist;
	   public String columnsBlacklistBiorweb;
	   public String datasourceProperties;
	   
	   // These should be filled in by writePropsFiles()
	   public File columnsTsvFile;
	   public File columnsBlacklistFile;
	   public File columnsBlacklistBiorwebFile;
	   public File datasourcePropertiesFile;
   }
   

   @BeforeClass
   public static void beforeAll() throws FileNotFoundException {
	   BaseFunctionalTest.setBiorToolkitCmdsRequired(true);
   }
   
   	@Before
   	public void beforeEach() throws IOException, BuildCatalogStepInputException
   	{
   		mSysoutBytes = new ByteArrayOutputStream();
   		mSyserrBytes = new ByteArrayOutputStream();
   		mSysoutTempStream = new PrintStream(mSysoutBytes);
   		mSyserrTempStream = new PrintStream(mSyserrBytes);
   		

	   
   		TemporaryFolder tempFolder = new TemporaryFolder();
   		tempFolder.create();
   		mTempDir = tempFolder.newFolder();
   		mTargetDir = tempFolder.newFolder();

   		// NOTE: This is not used in all tests - some supply their own
   		mBuildInfoFile = BuildCatalogTestUtils.substituteInITCaseBuildInfo(mTargetDir, mTempDir);

 
   		File compileScript = new File("src/test/resources/buildCatalog/ITCase_makeJson.sh");
   		compileScript.setExecutable(true);
   		
   		File biorDrill = TabixCmd.getFromPath("bior_drill");
   		String biorDrillPath = ( biorDrill == null )
   				?  "(unknown)"
   				:  biorDrill.getAbsolutePath(); 
   		System.out.println("\n\nBioR path (to bior_drill):\n  " + biorDrillPath + "\n\n");
   	}
   	
   	
	@AfterClass
	public static void afterAll() {
		// Files to delete after verify run
		List<String> filesToRemove = Arrays.asList(
				"IT_Test_BuildCatalog_verify.txt",
				"buildCatalog.allSteps_verify.txt",
				"buildCatalogCommandITCase.logfile.txt",
				"catalog378197_verify.txt",
				"catalogUnknown_verify.txt",
				"verifyCatalog_TestCatalog_verify.txt"
				);
		for(String file : filesToRemove) {
			File f = new File(file);
			if( f.exists() )
				f.delete();
		}
	}


   @Test
   public void test_viaJavaObjOnly()
   {
      try
      {
         String buildInfoFileName = "src/test/resources/buildCatalog/ITCase_buildInfo.txt";
         BuildCatalog buildCat = new BuildCatalog();
         buildCat.build(mBuildInfoFile.getCanonicalPath(), BuildStepKey.MAKE_JSON);
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail("Unexpected exception occurred while building catalog. Exception: " + e.getMessage());
      }
   }

   @Test
   public void test_viaCLICommand()
   {
      try
      {
    	 File logFile = new File(mTempDir, "buildCatalogCommandITCase.logfile.txt");
         CommandOutput out = executeScript(
            "bior_build_catalog",
            null,
            "-b", mBuildInfoFile.getCanonicalPath(),
            "-s", BuildStepKey.MAKE_JSON.name(),
            "--logfile", logFile.getCanonicalPath());

         System.out.println("Standard Output: " + out.stdout);
         System.err.println("Standard Error: " + out.stderr);
         System.out.println("Standard Exit Status: " + out.exit);

         assertContainsAll(out.stdout, Arrays.asList(
        		 "Step make_json SUCCEEDED",
        		 "Step make_catalog SUCCEEDED",
        		 "Step make_prop_files SUCCEEDED",
        		 "Step merge_prop_files SUCCEEDED",
        		 "Step make_indexes SUCCEEDED",
        		 "Step verify SUCCEEDED"
        		 ));
         // These two steps removed for now
         //assertContains(out.stderr, "Step docs SUCCEEDED");
         //assertContains(out.stderr, "Step summary SUCCEEDED");

         assertTrue(out.exit == 0);

         String logFileContents = FileUtils.readFileToString(logFile);
         String verifyTxt = FileUtils.readFileToString(new File("IT_Test_BuildCatalog_verify.txt"));
         assertContains(logFileContents, "Warning: _maxBP was not specified");
         // Columns.tsv
         assertContains(out.stdout, "IT_Test_BuildCatalog.columns.tsv' has some issues.");
         assertContains(verifyTxt,  "IT_Test_BuildCatalog.columns.tsv' has the following WARNING messages:\n"
        		     + "WARNING: Description not found for these columns: [ALTS, CHR, GENENAME, RSID]\n"
        		     + "WARNING: HumanReadableName is set to the same value as the ColumnName for these columns: [ALTS, CHR, GENENAME, RSID]");
         // Datasource.properties
         assertContains(out.stdout, "IT_Test_BuildCatalog.datasource.properties' has some issues.");
         assertContains(verifyTxt, "WARNING: Property 'Description' in '" + mTargetDir.getCanonicalPath()
         			  + "/IT_Test_BuildCatalog.datasource.properties' should have a value");

      }
      catch (Exception e)
      {
         fail("Unexpected exception occurred while building catalog. Exception: " + e.getMessage());
      }
   }
   
   
   @Test
   /** Test with double-quotes in one of the descriptions in both columns.tsv and datasource.properties
    *  Verify that the resulting files do not have double-quotes.    */
   public void testDoubleQuotesInColsTsvAndDatasrcProps()
   {
      try
      {
    	  //-----------------------------------------------------------------------------------
    	  // Setup previous catalog to have some fields with double-quotes
    	  //-----------------------------------------------------------------------------------
    	  // Create a previous catalog that will have:
    	  // - datasource.properties with double-quote in description
    	  // - columns.tsv with double-quotes in the description and HumanReadableName fields
    	  File previousCtgDir = new File(mTempDir, "PreviousCatalog");
    	  previousCtgDir.mkdir();
    	  
    	  File previousCtg  = new File(previousCtgDir, "junk.tsv.bgz");
    	  previousCtg.createNewFile();
    	  
    	  // For datasource.properties:
    	  // Embed double-quotes within the "Description" - this is NOT ok
    	  // Surround the value for "Source" with quotes - this should be ok as quotes should be removed
    	  File prevDatasrcProps = new File(previousCtgDir, "junk.datasource.properties");
    	  FileUtils.write(prevDatasrcProps, "Description=A description with \"double-quotes\" in it" + "\n"
    			  						+   "Source=\"dbSNP\"");
    	  
    	  File prevColsTsv      = new File(previousCtgDir, "junk.columns.tsv");
    	  FileUtils.write(prevColsTsv, concat("#Key",     "Type",   "Count", "Description",              "HumanReadableName") + "\n"
    			  				+  concat("GENENAME", "String", "1",     "The \"name\" of the gene", "Gene \"Name\"") );
    	  
    	  FileUtils.copyFile(prevColsTsv, new File(previousCtgDir, "junk.columns.tsv.blacklist"));
    	  FileUtils.copyFile(prevColsTsv, new File(previousCtgDir, "junk.columns.tsv.blacklist.biorweb"));

    	  // Since the build_info.txt has the previous catalog commented out, we want to add it back in
    	  String buildInfoWithPrevCtg = FileUtils.readFileToString(mBuildInfoFile).replaceAll("#PREVIOUS_CATALOG.*", "PREVIOUS_CATALOG_PATH=" + previousCtg.getCanonicalPath());
    	  FileUtils.write(mBuildInfoFile, buildInfoWithPrevCtg);
    	  //-----------------------------------------------------------------------------------
    	  

    	  File logFile = new File(mTempDir, "buildCatalogCommandITCase.logfile.txt");
    	  CommandOutput out = runCmdApp(new BuildCatalogCommand(), "bior_build_catalog", 
    			  "-b", mBuildInfoFile.getCanonicalPath(),
    			  "-s", BuildStepKey.MAKE_JSON.name(),
    			  "--logfile", logFile.getCanonicalPath());

         System.out.println("Standard Output: " + out.stdout);
         System.err.println("Standard Error: " + out.stderr);
         System.out.println("Standard Exit Status: " + out.exit);

         assertContainsAll(out.stdout, Arrays.asList(
        		 "Step make_json SUCCEEDED",
        		 "Step make_catalog SUCCEEDED",
        		 "Step make_prop_files SUCCEEDED",
        		 "Step merge_prop_files SUCCEEDED",
        		 "Step make_indexes SUCCEEDED",
        		 "Step verify SUCCEEDED"
        		 ));
         
         // Check for errors - two from double-quote in datasource.properties, and two from columns.tsv
         assertContains(out.stderr, "Warning: the Description value in datasource.properties contains a double-quote which could mess up any downstream VCF validators.  Converting these to single-quotes.  Offending value: [A description with \"double-quotes\" in it] in file: ");
         // Just ignore quotes around values (this should not be a warning)
         //assertContains(out.stderr, "Warning: the Source value in datasource.properties contains a double-quote which could mess up any downstream VCF validators.  Converting these to single-quotes.  Offending value: [\"dbSNP\"] in file: ");
         assertContains(out.stderr, "Warning: the HumanReadableName in columns.tsv contains a double-quote which would mess up any downstream VCF validators.  Converting these to single-quotes.  Offending Line: [GENENAME	String	1	The \"name\" of the gene	Gene \"Name\"].  File: ");
         assertContains(out.stderr, "Warning: the Description in columns.tsv contains a double-quote which would mess up any downstream VCF validators.  Converting these to single-quotes.  Offending Line: [GENENAME	String	1	The \"name\" of the gene	Gene \"Name\"].  File: ");
         
         assertTrue(out.exit == 0);
         
         // Verify that the columns.tsv and datasource.properties do NOT have the double-quotes
         String colsTsvContent = FileUtils.readFileToString(new File(mTargetDir, "IT_Test_BuildCatalog.columns.tsv"));
         String dataSrcContent = FileUtils.readFileToString(new File(mTargetDir, "IT_Test_BuildCatalog.datasource.properties"));
         assertFalse(colsTsvContent.contains("\""));
         assertFalse(dataSrcContent.contains("\""));
      }
      catch (Exception e)
      {
         fail("Unexpected exception occurred while building catalog. Exception: " + e.getMessage());
      }
   }
   
   
   
   
    @Test
   public void testErrorCode_viaCLICommand_InvalidBuildInfoFile()
   {
      try
      {
         CommandOutput out = executeScript(
            "bior_build_catalog",
            null,
            "-b", mBuildInfoFile.getCanonicalPath() + ".INVALID_BUILD_INFO_FILE",
            "-s", BuildStepKey.MAKE_JSON.name(),
            "--logfile", "buildCatalogCommandITCase.logfile.txt");

         //System.out.println("Standard Output: " + out.stdout);
         //System.err.println("Standard Error: " + out.stderr);
         //System.out.println("Standard Exit Status: " + out.exit);

         assertTrue("System.out did not match expected.  Actual:\n" + out.stdout,
        		    out.stdout.length() == 0);
         assertTrue("System.err did not match expected.  Actual:\n" + out.stderr,
     		        out.stderr.startsWith("Application error bior_build_catalog:\n: Error: Specified build information file does not exist: "));
         assertContains(out.stderr, "/build_info.txt.INVALID_BUILD_INFO_FILE");
         assertTrue(out.exit == 1);
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail("Unexpected exception occurred while building catalog. Exception: " + e.getMessage());
      }
   }
   
   
   
   
   @Test
   /** There were errors when using relative paths defined in the build_info.txt file */ 
   public void test_relativePathToCompileScriptInBuildInfoTxt() throws IOException, InterruptedException
   {
	   // Create build_info.txt and put it into temp folder
	   System.out.println("Target dir: " + mTargetDir.getCanonicalPath());
	   System.out.println("Temp dir:   " + mTempDir.getCanonicalPath());
	   System.out.println("BuildInfo:  " + mBuildInfoFile.getCanonicalPath());
	   
	   // Create a script that will build a catalog based on the number of lines passed in as an argument,
	   // and output to the path specified in the variable $MAKE_JSON_OUTPUT_FILE_PATH
	   // Lines will be similar to:  
	   //		"1  100  100  {'key':1}"
	   //       "2  200  200  {'key':2}"
	   String makeJsonScriptContents = 
			   "#!/bin/bash\n" +
			   "numLines=$1\n" +
			   "echo \"Current directory: $PWD\"" + "\n" +
			   "echo \"MAKE_JSON_OUTPUT_FILE_PATH: $MAKE_JSON_OUTPUT_FILE_PATH\"" + "\n" +
			   "echo \"TEMP_DIR: $TEMP_DIR\"" + "\n" +
			   "for num in `seq 1 $numLines` ; do" + "\n" +
			   "  echo -e \"$num\\t${num}00\\t${num}00\\t{'key':$num,'src':$num}\" >> $MAKE_JSON_OUTPUT_FILE_PATH" + "\n" +
			   "done" + "\n" + 
			   "touch $TEMP_DIR/junk.txt";

	   File makeJsonScript  = new File(mTargetDir, "makeJson.sh");
	   FileUtils.writeStringToFile(makeJsonScript, makeJsonScriptContents);
	   makeJsonScript.setExecutable(true);
	   
	   // Create a subdirectory under the target that has the previous catalog and copy all catalog files to that
	   File prevCtgDir = new File(mTargetDir, "PreviousCatalog");
	   prevCtgDir.mkdir();
	   final String PREVIOUS_CATALOG_DIR = "src/test/resources/buildCatalog/mergeProps_previousCatalog";
	   FileUtils.copyFileToDirectory(new File(PREVIOUS_CATALOG_DIR, "x.tsv.bgz"), prevCtgDir);
	   FileUtils.copyFileToDirectory(new File(PREVIOUS_CATALOG_DIR, "x.columns.tsv"),   prevCtgDir);
	   FileUtils.copyFileToDirectory(new File(PREVIOUS_CATALOG_DIR, "x.datasource.properties"),  prevCtgDir);
	   
	   // Make a temp directory relative to the target dir
	   File tempSubdir = new File(mTargetDir, "Temp");
	   tempSubdir.mkdir();
	   
	   // The build_info.txt file should be in the temp directory - load it and change the following lines
	   String buildInfoContents = 
			   "MAKE_JSON_SCRIPT_PATH=makeJson.sh" + "\n" +				// RELATIVE PATH
			   "MAKE_JSON_ARGS=10" + "\n" +
			   "MAKE_JSON_OUTPUT_FILE_PATH=output.json" + "\n" +		// RELATIVE PATH
			   "TARGET_DIR=." + "\n" + 									// RELATIVE PATH
			   "PREVIOUS_CATALOG_PATH=PreviousCatalog/x.tsv.bgz" + "\n" +//RELATIVE PATH
			   "CATALOG_PREFIX=mycatalog" + "\n" +
			   "TEMP_DIR=Temp" + "\n" +									// RELATIVE PATH
			   "DATA_SOURCE=testSource" + "\n" +
			   "DATA_SOURCE_VERSION=142" + "\n" +
			   "DATA_SOURCE_BUILD=GRCh37" + "\n" +
			   "DATA_SOURCE_RELEASE_DATE=2018_05_03" + "\n" +
			   "INDEXES=key,src" + "\n";
	   FileUtils.writeStringToFile(new File(mTargetDir, "build_info.txt"), buildInfoContents);
	   
	   String batchFileContents = "cd " + mTargetDir.getCanonicalPath() + "\n"
			   				+     "bior_build_catalog  -b  build_info.txt  --log";
	   File buildScript = new File(mTargetDir, "build.sh");
	   FileUtils.writeStringToFile(buildScript, batchFileContents);
	   buildScript.setExecutable(true);
	   
	   //--------------------------------------- Execute them! -----------------------------
	   String stdin = null;
	   CommandOutput out = executeScript( buildScript.getCanonicalPath(), stdin );
	
	   System.out.println("Standard Output: " + out.stdout);
	   System.err.println("Standard Error: " + out.stderr);
	   System.out.println("Standard Exit Status: " + out.exit);
	
	   assertContainsAll(out.stdout, Arrays.asList( 
			   "Step make_json SUCCEEDED",
			   "Step make_catalog SUCCEEDED",
			   "Step make_prop_files SUCCEEDED",
			   "Step merge_prop_files SUCCEEDED",
			   "Step make_indexes SUCCEEDED",
			   "Step verify SUCCEEDED"
			   // These two have been removed for now since we are not doing anything with them yet
			   //"docs",
			   //"summary"
			   ));
	   
	   // Should finish with exitcode==0
	   assertEquals(0, out.exit);
   }
   
   

   
   @Test
   /** Test scenario where the user runs with a limited catalog first which does NOT have all the keys,
    *  but has edited the description and HumanReadableName columns. 
    *  Then runs bior_build_catalog against the full catalog and those columns are added (should show warnings)
    *  We point to the previous catalog which has the Description and HumanReadableName filled in and updates to those (also warning),
    *  as well as a couple fields from the datasource.properties file which were missing, but then filled in from previous catalog */
   public void testPropertiesFiles_manuallyFilledVsDefaultsVsPreviousCatalogSettings() throws Exception {
	   mBuildInfoMap = getBuildInfoMap();
	   
	   // First create the columns.tsv and datasource.properties files 
	   // and edit them so they have a modified description and HumanReadableName for at least a couple items
	   // Then run the full build process and verify that the values in the modified files are NOT overwritten
	   PropertiesFilesContents currProps = getProps0Input();
	   writePropsFiles(currProps, mTargetDir, "buildCatalog.allSteps");

	   // Write the previous catalog files
	   PropertiesFilesContents prevCtg = getProps1Input();
	   // Replace "key2" line with a new key: "xxxx" in previous catalog's columns.tsv
	   // Should get a warning about the missing row and one about the additional row
	   prevCtg.columnsTsv = removeRowsAfterId(prevCtg.columnsTsv, "key") + "\n"
			   + concat("xxxx", "String", "1", "The X factor",               "XFactor") + "\n";
	   File prevCtgDir = new File(mTargetDir, "PreviousCatalog");
	   prevCtgDir.mkdir();
	   String catalogPrefix =  "buildCatalog.allSteps";
	   writePropsFiles(prevCtg, prevCtgDir, catalogPrefix);
	   File prevCtgFile = new File(prevCtgDir, catalogPrefix + ".tsv.bgz");
	   prevCtgFile.createNewFile();
	   mBuildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), prevCtgFile.getCanonicalPath());


	   //--------- Run the build script ---------------------------------
	   runBuildScript();

	   // Verify that the contents have been modified as expected ------------------------------------------
	   PropertiesFilesContents propsActual = loadProps(mTargetDir, catalogPrefix, /*isDefaults=*/false);
	   
	   // Same as props1 except that key2 will have blank description and humanReadableName==ID
	   //   because it was not in the current columns.tsv and was not in the previous catalog
	   PropertiesFilesContents propsExpected = getProps1Expected();
	   propsExpected.columnsTsv = removeRowsAfterId(propsExpected.columnsTsv, "aaa") + "\n"
		   // "bbb" should not have updated type, description nor humanReadableName.  Also, warn about type being different from default 
		   + concat("bbb",  "String", "1",  "bbb desc",  "bbbHRN") + "\n"
		   // "key" should not have updated description but SHOULD have updated humanReadableName 
		   + concat("key",  "String", "1",  "Some key",   "keyUpdated")	+ "\n"
		   // "key2" should have blank description, and default humanReadableName "key2" because it was not in the current, but will be added from default
		   + concat("key2", "String", "1",  ".",           "key2") + "\n"
		   // "xyz" is not in the catalog or previous catalog, so should remain unchanged
	   	   + concat("xyz",	"String", "1",  "A field not actually in the catalog", "ANonExistentField") + "\n";

	   // Verify the props
	   assertPropsSame(propsExpected, propsActual);

	   // Verify stderr ----------------------------------------------------------------------------------
	   String sysout = getSysout();
	   // Summary output: ---------------------------------------------------------------
	   String mergeLogPath = getLogFile(mTargetDir, BuildStepKey.MERGE_PROP_FILES).getCanonicalPath();
	   assertContains(sysout,
			   	"Step merge_prop_files STARTED\n" +
			   	"Some changes made to '" + mTargetDir.getCanonicalPath() + "/buildCatalog.allSteps.datasource.properties'\n" +
			   	"Some changes made to '" + mTargetDir.getCanonicalPath() + "/buildCatalog.allSteps.columns.tsv'\n" +
			   	"Columns.tsv file '"     + mTargetDir.getCanonicalPath() + "/buildCatalog.allSteps.columns.tsv' has some issues. See verify output for details.\n" +
			   	"Current blacklist file has been previously modified by user.  No changes made.\n" +
			   	"Current blacklist.biorweb file has been previously modified by user.  No changes made.\n" + 
			   	"Step merge_prop_files SUCCEEDED" );

	   // Log output --------------------------------------------------------------------
	   String verifyTxt = FileUtils.readFileToString(new File("buildCatalog.allSteps_verify.txt"));
	   String mergeLogContents = FileUtils.readFileToString(new File(mergeLogPath));
	   System.out.println("Log:-----------" + mergeLogContents);
	   // Columns.tsv
	   // Ex timestamp:  "2016-08-29 09:54:54" - make sure we can parse it
	   final String TIMESTAMP_FORMAT = "yyyy-MM-dd HH:mm:ss ";
	   SimpleDateFormat dateFormat = new SimpleDateFormat(TIMESTAMP_FORMAT);
	   String timestamp = mergeLogContents.substring(0, mergeLogContents.indexOf("Step merge_prop_files STARTED"));
	   dateFormat.parse(timestamp);
	   String actual = removeFirstXCharsFromEachLine(mergeLogContents, TIMESTAMP_FORMAT.length());
	   String expected = "Step merge_prop_files STARTED\n"
			   		 // Datasource.properties
			   		 +   "Warning: (datasource.properties): Key 'Description' missing, or value is empty.  Attempting to add key and value...\n"
			   		 +   "    Using the previous catalog key and value: A short catalog\n"
			   		 +   "Description/comment missing for key: 'Description'\n"
			   		 +   "    Using default description/comment: ## Description of catalog.  Ex: NCBI's dbSNP Variant Database\n"
			   		 +   "Description/comment missing for key: 'Source'\n"
			   		 +   "    Using default description/comment: ## Source of data, without point release, etc.  Ex: dbSNP\n"
			   		 +   "Warning: (datasource.properties): Key 'Dataset' missing, or value is empty.  Attempting to add key and value...\n"
			   		 +   "    Using the previous catalog key and value: Misc\n"
			   		 +   "Description/comment missing for key: 'Dataset'\n"
			   		 +   "    Using default description/comment: ## Type of data.  Ex: Variants\n"
			   		 +   "Note: (datasource.properties): Value for key 'Build' coming from build_info.txt file with value: GRCh37\n"
			   		 +   "Some changes made to '" + mTargetDir.getCanonicalPath() + "/buildCatalog.allSteps.datasource.properties'\n"
			   		 // Columns.tsv
			   		 +   "Warning: (columns.tsv): Using the previous catalog's description for column 'aaa'\n"
			   		 +   "Warning: (columns.tsv): Using the previous catalog's HumanReadableName for column 'aaa'\n"
			   		 +   "Warning: (columns.tsv): Column bbb:  The current *Type* is different from the default that was calculated from crawling all data in the catalog.\n"
			   		 +   "    current Type: String\n"
			   		 +   "    default Type: Integer\n"
			   		 +   "Warning: (columns.tsv): Column bbb: The current *Type* is different from the same column in the previous catalog.\n"
			   		 +   "    current  Type: String\n"
			   		 +   "    previous Type: Integer\n"
			   		 +   "Warning: (columns.tsv): Using the previous catalog's HumanReadableName for column 'key'\n"
			   		 +   "Warning: (columns.tsv): Extra key?  'xyz' is in the current columns.tsv, but was not detected in the catalog (not in columns.tsv.defaults).\n"
			   		 +   "Warning: (columns.tsv): Extra key?  'xyz' is in the current columns.tsv, but not in the previous catalog's columns.tsv.\n"
			   		 +   "Warning: (columns.tsv): Some columns were detected in the catalog data (default columns.tsv) but not in the current columns.tsv.  These will be added to the current list.\n"
			   		 +   "    key2	String	1	.	key2\n"
			   		 +   "Warning: (columns.tsv): These columns were present in the previous catalog's columns.tsv but not in the current columns.tsv:\n"
			   		 +   "    xxxx	String	1	The X factor	XFactor\n"
			   		 +   "Some changes made to '" + mTargetDir.getCanonicalPath() + "/buildCatalog.allSteps.columns.tsv'\n"
			   		 +   "Columns.tsv file '" + mTargetDir.getCanonicalPath() + "/buildCatalog.allSteps.columns.tsv' has some issues. See verify output for details.\n"
			   		 +   "Current blacklist file has been previously modified by user.  No changes made.\n"
			   		 +   "Current blacklist.biorweb file has been previously modified by user.  No changes made.\n"
			   		 +   "Step merge_prop_files SUCCEEDED\n";
	   assertEquals(expected, actual);
   }
   
   private String removeFirstXCharsFromEachLine(String fullStr, int firstXCharsToCut) {
	   String[] lines = fullStr.split("\n");
	   StringBuilder out = new StringBuilder();
	   for(String s : lines) {
		   int lenToCut = firstXCharsToCut >= s.length()  ?  s.length() : firstXCharsToCut;
		   out.append(s.substring(lenToCut+1) + "\n");
	   }
	   return out.toString();
   }

   private void assertContains(String fullText, List<String> listOfStringsToCheck) {
	   for(String s : listOfStringsToCheck) {
		   assertContains(fullText, s);
	   }
   }

   private String loadLog(File catalogDir, BuildStepKey buildStep) throws IOException {
	   return FileUtils.readFileToString(getLogFile(catalogDir, buildStep));
   }
   
   private File getLogFile(File catalogDir, BuildStepKey buildStep) {
	   return new File(catalogDir, BuildCatalog.BUILD_SUBDIR + "/progress/" + buildStep.getDisplayName() + ".log");
   }

/** Find strToFind, then cut off all rows after the newline following that string */
   private String removeRowsAfterId(String fullStr, String idToFind) {
	   int idx = fullStr.indexOf("\n" + idToFind + "\t");
	   if( idx == -1 )
		   return fullStr;
	   
	   int idxNewline = fullStr.indexOf("\n", idx+1);
	   if( idxNewline == -1 )
		   return fullStr;
	   
	   return fullStr.substring(0, idxNewline);
   }

   @Test
   /** If the user edits the columns.tsv file (or datasource.properties), 
    *  then re-runs the build script, it should NOT overwrite the values in there    */
   public void testPropertiesFiles_allStepsAfterColumnsTsvEdited() throws IOException, Exception
   {
	   mBuildInfoMap = getBuildInfoMap();
	   
	  // First create the columns.tsv and datasource.properties files 
	  // and edit them so they have a modified description and HumanReadableName for at least a couple items
	  // Then run the full build process and verify that the values in the modified files are NOT overwritten
	  PropertiesFilesContents propsIn = getProps1Input();
	  String ctgPrefix = "buildCatalog.allSteps";
	  writePropsFiles(propsIn, mTargetDir, ctgPrefix);
	  
	  
	  //--------- Run the build script ---------------------------------
	  runBuildScript();

	  
      // Verify that the contents have not been modified ------------------------------------------
	  PropertiesFilesContents propsActual = loadProps(mTargetDir, ctgPrefix, /*isDefaults=*/false);
	  PropertiesFilesContents propsExpected = getProps1Expected();
	  assertPropsSame(propsExpected, propsActual);
      

	  // Make sure the defaults are NOT equal to the current files------------------------------------
      PropertiesFilesContents propsDefaultsActual = loadProps(mTargetDir, ctgPrefix, /*isDefaults=*/true);
      PropertiesFilesContents propsDefaultExpected = getProps1ExpectedDefaults();
      assertPropsSame(propsDefaultExpected, propsDefaultsActual);

      String sysout = getSysout();
      String verifyTxt = FileUtils.readFileToString(new File("buildCatalog.allSteps_verify.txt"));
      assertContains(sysout,
    		  "Step merge_prop_files STARTED\n"
    		+ "Some changes made to '" + mTargetDir.getCanonicalPath() + "/buildCatalog.allSteps.datasource.properties'\n"
    		+ "No changes made to '" + mTargetDir.getCanonicalPath() + "/buildCatalog.allSteps.columns.tsv'\n"
    		+ "Current blacklist file has been previously modified by user.  No changes made.\n"
    		+ "Current blacklist.biorweb file has been previously modified by user.  No changes made.\n"
    		+ "Step merge_prop_files SUCCEEDED\n");
   }

   
   @Test
   /** If the user edits the columns.tsv, datasource.properties, and blacklist files 
    *  then re-runs the build script, it should NOT overwrite the values in there.
    *  Test with no previous catalog. */
   public void testPropertiesFiles_editAfterFirstRun_noPrevCtg() throws IOException, Exception  {
	   runEditRun(/*isWithPreviousCatalog=*/false);
   }

   @Test
   /** If the user edits the columns.tsv, datasource.properties, and blacklist files 
    *  then re-runs the build script, it should NOT overwrite the values in there.
    *  Test with no previous catalog. */
   public void testPropertiesFiles_editAfterFirstRun_withPrevCtg() throws IOException, Exception  {
	   runEditRun(/*isWithPreviousCatalog=*/true);
   }


   
   /** Run the build script, edit all properties files, then run the build script again */
   private void runEditRun(boolean isWithPreviousCatalog) throws Exception {
	   mBuildInfoMap = getBuildInfoMap();

	   
	  // First create the columns.tsv and datasource.properties files 
	  // and edit them so they have a modified description and HumanReadableName for at least a couple items
	  // Then run the full build process and verify that the values in the modified files are NOT overwritten
	  PropertiesFilesContents propsIn = getProps1Input();
	  String ctgPrefix = "buildCatalog.allSteps";
	  writePropsFiles(propsIn, mTargetDir, ctgPrefix);
	  
	  
	  //--------- Run the build script ---------------------------------
	  if( isWithPreviousCatalog ) {
		  mBuildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), new File("src/test/resources/buildCatalog/mergeProps_previousCatalog/x.tsv.bgz").getCanonicalPath());
	  }
	  runBuildScript();

	  
	  // Modify the properties files
	  String columnsTsv = FileUtils.readFileToString(propsIn.columnsTsvFile);
	  String dataSource = FileUtils.readFileToString(propsIn.datasourcePropertiesFile);
	  // columns.tsv - change a description and a separate human-readable name; and add extra row to columns tsv
	  columnsTsv = columnsTsv.replace("ALTS	String	.	.	ALTS", "ALTS	String	.	New description	ALTS")
			  				 .replace("CHR	String	1	.	CHR",  "CHR	String	1		Chromosome")
			  				 + "TestId	String	1	.	Test Id" + "\n";
	  // datasource.properties - change a value, and add extra field to datasource.properties
	  dataSource = dataSource.replace("Format=1.1.1", "Format=1.1.1.0")
			  				 + "TestField=something" + "\n";
	  // Set blacklist
	  String blacklist = MergeMetadataStepTest.BLACKLIST_HEADER + "\n" + "_altAlleles" + "\n" + "_id";
	  String blacklistBiorweb = MergeMetadataStepTest.BLACKLIST_HEADER + "\n" + "_altAlleles" + "\n" + "_id" + "\n" + "_landmark" + "\n" + "_maxBP";
	  // Write the files back out
	  FileUtils.writeStringToFile(propsIn.columnsTsvFile, columnsTsv);
	  FileUtils.writeStringToFile(propsIn.datasourcePropertiesFile, dataSource);
	  FileUtils.writeStringToFile(propsIn.columnsBlacklistFile, blacklist);
	  FileUtils.writeStringToFile(propsIn.columnsBlacklistBiorwebFile, blacklistBiorweb);
	  

	  
	  // Now, run the script again
	  runBuildScript();

	  
      // Verify that the contents have not been modified ------------------------------------------
	  String columnsTsvAfter = FileUtils.readFileToString(propsIn.columnsTsvFile);
	  String dataSourceAfter = FileUtils.readFileToString(propsIn.datasourcePropertiesFile);
	  String blacklistAfter  = FileUtils.readFileToString(propsIn.columnsBlacklistFile);
	  String blacklistBiorwebAfter = FileUtils.readFileToString(propsIn.columnsBlacklistBiorwebFile);
	  
	  assertEquals(columnsTsv, columnsTsvAfter);
	  assertEquals(dataSource, dataSourceAfter);
	  assertEquals(blacklist,  blacklistAfter);
	  assertEquals(blacklistBiorweb, blacklistBiorwebAfter);
	  
	  String sysout = getSysout();
      assertContains(sysout, "Step make_json SUCCEEDED");
      assertContains(sysout, "Step make_catalog SUCCEEDED");
      assertContains(sysout, "Step make_prop_files SUCCEEDED");
      assertContains(sysout, "Step merge_prop_files SUCCEEDED");
      assertContains(sysout, "Step make_indexes SUCCEEDED");
      assertContains(sysout, "Step verify SUCCEEDED");
   }

   
   
	/** Create expected properties from the base props created by getProps1Input() method */
   private PropertiesFilesContents getProps1Expected() {
	   PropertiesFilesContents props = getProps1Input();
	   // Some headers should be added from the hard-coded headers since the inputs don't have any
	   props.datasourceProperties = props.datasourceProperties
			  .replace("ShortUniqueName=", "### Datasource properties file for Catalog - buildCatalog.allSteps.  Please fill in the descriptions to the keys below.\n"
					  					 + "## Short name that should be unique (or mostly unique except for fixes to existing catalog) across all catalogs. Ex: dbSNP_142_GRCh37p13\n"
					  					 + "ShortUniqueName=")
			  .replace("Description=", "## Description of catalog.  Ex: NCBI's dbSNP Variant Database\n" + "Description=")
			  .replace("Source=",      "## Source of data, without point release, etc.  Ex: dbSNP\n"     + "Source=")
			  .replace("Dataset=",     "## Type of data.  Ex: Variants\n" + "Dataset=");
	   return props;
   }

   private void assertPropsSame(PropertiesFilesContents propsExpected, PropertiesFilesContents propsActual) {
	      assertEquals(propsExpected.columnsTsv, 				propsActual.columnsTsv);
	      assertEquals(propsExpected.datasourceProperties, 		propsActual.datasourceProperties);
	      assertEquals(propsExpected.columnsBlacklist,  		propsActual.columnsBlacklist);
	      assertEquals(propsExpected.columnsBlacklistBiorweb, 	propsActual.columnsBlacklistBiorweb);
   }

   private PropertiesFilesContents loadProps(File catalogDir, String catalogPrefix, boolean isDefaults) throws IOException {
	      PropertiesFilesContents props = new PropertiesFilesContents();
	      String buildDir = isDefaults ? "/" + BuildCatalog.BUILD_SUBDIR + "/" : "";
	      String defaultExt = isDefaults ? ".default" : "";
	      props.columnsTsvFile           	= new File(catalogDir,  buildDir + catalogPrefix + ".columns.tsv" + defaultExt);
	      props.columnsBlacklistFile     	= new File(catalogDir,  buildDir + catalogPrefix + ".columns.tsv.blacklist" + defaultExt);
	      props.columnsBlacklistBiorwebFile = new File(catalogDir,  buildDir + catalogPrefix + ".columns.tsv.blacklist.biorweb" + defaultExt);
	      props.datasourcePropertiesFile 	= new File(catalogDir,  buildDir + catalogPrefix + ".datasource.properties" + defaultExt);
	      
	      props.columnsTsv 				= FileUtils.readFileToString(props.columnsTsvFile); 
	      props.columnsBlacklist 		= FileUtils.readFileToString(props.columnsBlacklistFile); 
	      props.columnsBlacklistBiorweb	= FileUtils.readFileToString(props.columnsBlacklistBiorwebFile);
	      props.datasourceProperties 	= FileUtils.readFileToString(props.datasourcePropertiesFile); 
	      
	      return props;
   }

   private void redirectSysoutAndSyserrToStrings() {
	   System.setOut(mSysoutTempStream);
	   System.setErr(mSyserrTempStream);
   }
   
   private void resetSysoutAndSyserr() {
	   System.setOut(mSysoutOrig);
	   System.setErr(mSyserrOrig);	   
   }
   
   private String getSysout() {
	   return mSysoutBytes.toString();
   }

   private String getSyserr() {
	   return mSyserrBytes.toString();
   }

   /** Used by test:   testAllStepsAfterColumnsTsvEdited */
   private PropertiesFilesContents getProps1Input() {
	   PropertiesFilesContents props1 = new PropertiesFilesContents();
	   props1.columnsTsv = MergeMetadataStepTest.COLUMNS_TSV_HEADER +
				  concat("_landmark",    "String", "1",     "Updated Description - _landmark", "landmarkUpdated") + "\n" +
				  concat("_maxBP",       "Integer","1",     "Updated Description - _maxBP",    "maxBpUpdated") + "\n" +
				  concat("_minBP",       "Integer","1",     "Updated Description - _minBP",    "minBpUpdated") + "\n" +
				  concat("aaa",          "Integer","1",     "Updated Description - aaa",       "aaaUpdated") + "\n" +
				  concat("bbb",          "Integer","1",     "Updated Description - bbb",       "bbbUpdated") + "\n" +
				  concat("key",          "String", "1",     "Updated Description - key",       "keyUpdated") + "\n" +
				  concat("key2",         "String", "1",     "Updated Description - key2",      "key2Updated") + "\n";
		  
	   props1.datasourceProperties =
				  "ShortUniqueName=xcatalog" + "\n" +
				  "Description=A short catalog" + "\n" +
				  "Source=x" + "\n" +
				  "Dataset=Misc" + "\n" +
				  "## Version of the data source" + "\n" +
				  "Version=1" + "\n" +
				  "## The Genome build/assembly" + "\n" +
				  "Build=GRCh37" + "\n" +
				  "## The BioR catalog compatibility format" + "\n" +
				  "Format=1.1.0" + "\n" +
	   			  "## The release date of the data source from the provider (not the BioR build date).  Ex: 2018-07-10" + "\n" +
	   			  "DataSourceReleaseDate=2018-06-22" + "\n";

	   props1.columnsBlacklist = "_landmark" + "\n" +
				  				 "_maxBP" + "\n"; 
	   
	   props1.columnsBlacklistBiorweb = props1.columnsBlacklist;
	   
	   return props1;
   }
   
   
   /** Used by test:   testAllStepsAfterColumnsTsvEdited */
   private PropertiesFilesContents getProps1ExpectedDefaults() {
	   PropertiesFilesContents props1 = new PropertiesFilesContents();
	   props1.columnsTsv = MergeMetadataStepTest.COLUMNS_TSV_HEADER +
	    		  concat("_landmark",   "String",  "1",       "Provides a context for the genomic coordinates _minBP and _maxBP.  Most often this is the chromosome where the feature appears, but could be a known genetic marker, gene, or other item (BioR field)", "Chromosome or landmark (BioR)") + "\n" +
	    		  concat("_maxBP",      "Integer", "1",       "The maximum (ending) 1-based base pair position within the chromosome (BioR field)", "Max base-pair position (BioR)") + "\n" +
	    		  concat("_minBP",      "Integer", "1",       "The minimum (starting) 1-based base pair position within the chromosome (BioR field)", "Min base-pair position (BioR)") + "\n" +
	    		  concat("aaa",         "Integer", "1",		  ".",  "aaa") + "\n" +
	    		  concat("bbb",         "Integer", "1",       ".",  "bbb") + "\n" +
	    		  concat("key",         "String",  "1",       ".",  "key") + "\n" +
	    		  concat("key2",        "String",  "1",       ".",  "key2") + "\n";

	   props1.datasourceProperties = 
	    		  "### Datasource properties file for Catalog - buildCatalog.allSteps.  Please fill in the descriptions to the keys below.\n" +
	    		  "## Short name that should be unique (or mostly unique except for fixes to existing catalog) across all catalogs. Ex: dbSNP_142_GRCh37p13\n" +
	    		  // Built from: DATA_SOURCE + _ + DATA_SOURCE_VERSION + _ + DATA_SOURCE_BUILD 
	    		  // Then remove all non-alphanumeric and underscore characters
	    		  // Ex: ShortUniqueName=dbSNP_142_GRCh37p13
	    		  "ShortUniqueName=\n" +
	    		  "## Description of catalog.  Ex: NCBI's dbSNP Variant Database\n" +
	    		  // Will need to be manually filled in
	    		  "Description=\n" +
	    		  "## Source of data, without point release, etc.  Ex: dbSNP\n" +
	    		  // From DATA_SOURCE
	    		  "Source=\n" +
	    		  "## Type of data.  Ex: Variants\n" +
	    		  // Will need to be manually filled in
	    		  "Dataset=\n" +
	    		  "## Version of the data source.  Ex: 142\n" +
	    		  // From DATA_SOURCE_VERSION
	    		  "Version=\n" +
	    		  "## The Genome build/assembly.  Ex: GRCh37.p13\n" +
	    		  // From DATA_SOURCE_BUILD
	    		  "Build=\n" +
	    		  "## The BioR catalog compatibility format.  Ex: 1.1.1\n" +
	    		  "Format=1.1.1\n" + 
	   			  "## The release date of the data source from the provider (not the BioR build date).  Ex: 2018-07-10" + "\n" +
	   			  "DataSourceReleaseDate=\n";


	   props1.columnsBlacklist = MergeMetadataStepTest.BLACKLIST_HEADER + 
	    		  "_landmark\n" +
	    		  "_maxBP\n" +
	    		  "_minBP\n";
	   
	   props1.columnsBlacklistBiorweb = MergeMetadataStepTest.BLACKLIST_BIORWEB_HEADER +
	    		  "_landmark\n" +
	    		  "_maxBP\n" +
	    		  "_minBP\n";
	   
	   return props1;
   }
   
   /** Used by test:   testInitialSubset_filledInByDefaultsAndPreviousCatalogSettings */
   private PropertiesFilesContents getProps0Input() {
	   PropertiesFilesContents props = new PropertiesFilesContents();
	   props.columnsTsv = MergeMetadataStepTest.COLUMNS_TSV_HEADER +
			concat("_landmark",    "String", "1",     "Updated Description - _landmark", "landmarkUpdated") + "\n" +
			concat("_maxBP",       "Integer","1",     "Updated Description - _maxBP",    "maxBpUpdated") + "\n" +
			concat("_minBP",       "Integer","1",     "Updated Description - _minBP",    "minBpUpdated") + "\n" +
			// Description and HumanReadableName should be filled in by previous catalog (getProps1Input())
			concat("aaa",          "Integer","1",     ".",       						 "aaa") + "\n" +
			// NOTE: "bbb" has a different type ("String" instead of "Integer") and description and HumanReadableName from the previous catalog - it should NOT be changed
			concat("bbb",          "String", "1",     "bbb desc",       				 "bbbHRN") + "\n" +
			// "key" is not in previous catalog - should be left alone here
			concat("key",          "String", "1",     "Some key",  						 "key") + "\n" +
	   		// NOTE: missing "key2" - should be filled in by bior_build_catalog because it occurs in the "default" catalog crawl
	   	    //   key2.....
	   		// "xyz" is a key that will NOT occur in the catalog, and will not be in the defaults.
	   		//   However, it should not be removed, as the user may be working from a small subset first, and the key only occurs in the full catalog
	   		concat("xyz",		   "String", "1",     "A field not actually in the catalog", "ANonExistentField") + "\n";
	   
	   props.datasourceProperties =
			"### Datasource properties file for Catalog - buildCatalog.allSteps.  Please fill in the descriptions to the keys below." + "\n" +
			"## Short name that should be unique (or mostly unique except for fixes to existing catalog) across all catalogs. Ex: dbSNP_142_GRCh37p13" + "\n" +
			"ShortUniqueName=xcatalog" + "\n" +
			"Description=" + "\n" +		// NOTE: Description should be filled in by bior_build_catalog from previous catalog
			"Source=x" + "\n" +
			"Dataset=" + "\n" +			// NOTE: Dataset should be filled in by bior_build_catalog from previous catalog
			"## Version of the data source" + "\n" +
			"Version=1" + "\n" +
			"## The Genome build/assembly" + "\n" +
			"Build=" + "\n" +			// NOTE: Build should be filled in by bior_build_catalog from previous catalog
			"## The BioR catalog compatibility format" + "\n" +
			"Format=1.1.0" + "\n" +
			"## The release date of the data source from the provider (not the BioR build date).  Ex: 2018-07-10" + "\n" +
			"DataSourceReleaseDate=2018-06-22" + "\n";
	   
	   props.columnsBlacklist = 
			"_landmark" + "\n" +
			"_maxBP" + "\n"; 
	   
	   props.columnsBlacklistBiorweb = props.columnsBlacklist;
	   
	   return props;
   }

   
  
   private Map<String, String> getBuildInfoMap() throws IOException {
	   String catalogPrefix = "buildCatalog.allSteps";

	   mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE.name(), "dbSNP");
	   mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE_VERSION.name(), "142");
	   mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE_BUILD.name(), HumanBuildAssembly.GRCh37.name());
	   mBuildInfoMap.put(BuildInfoKey.CATALOG_PREFIX.name(), catalogPrefix);
	   mBuildInfoMap.put(BuildInfoKey.INDEXES.name(), "key, bbb");
	   mBuildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), "");
	   mBuildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), mTargetDir.getCanonicalPath());
	   mBuildInfoMap.put(BuildInfoKey.TEMP_DIR.name(), mTempDir.getCanonicalPath());
	   mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), "");
	   mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), "--inputFile  inputFile  --column  4");
	   mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name(), new File("src/test/resources/buildCatalog/makeJsonStep.sh").getCanonicalPath());
	   File compileOutput = new File(mTempDir, "compileOutput.tsv");
	   mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), compileOutput.getCanonicalPath());

	   return mBuildInfoMap;
   }
   
   private void writePropsFiles(PropertiesFilesContents propsToWrite, File targetDir, String catalogPrefix) throws IOException {
	   propsToWrite.columnsTsvFile   			= new File(targetDir, catalogPrefix + ".columns.tsv");
	   propsToWrite.columnsBlacklistFile		= new File(targetDir, catalogPrefix + ".columns.tsv.blacklist");
	   propsToWrite.columnsBlacklistBiorwebFile = new File(targetDir, catalogPrefix + ".columns.tsv.blacklist.biorweb");
	   propsToWrite.datasourcePropertiesFile    = new File(targetDir, catalogPrefix + ".datasource.properties");
	   
	   FileUtils.writeStringToFile(propsToWrite.columnsTsvFile, 			propsToWrite.columnsTsv);
	   FileUtils.writeStringToFile(propsToWrite.datasourcePropertiesFile, 	propsToWrite.datasourceProperties);
	   FileUtils.writeStringToFile(propsToWrite.columnsBlacklistFile,		propsToWrite.columnsBlacklist);
	   FileUtils.writeStringToFile(propsToWrite.columnsBlacklistBiorwebFile,propsToWrite.columnsBlacklistBiorweb);
   }

   
   private void runBuildScript() throws Exception {
	   // Catalog contents:
	   //echo "1	2	3	{'key':'val','aaa':2,'_landmark':1,'_minBP':100,'_maxBP':100}" >> $fileOutputPath
	   //echo "2	3	4	{'key2':'val2','bbb':3,'_landmark':2,'_minBP':200,'_maxBP':200}" >> $fileOutputPath
	   //mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name(), new File("src/test/resources/buildCatalog/makeJsonStep.sh").getCanonicalPath());
	   //mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), "--inputFile  inputFile  --column  4");
//	   File compileOutput = new File(mTempDir, "compileOutput.tsv");
//	   mBuildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), compileOutput.getCanonicalPath());
//	   
//	   mBuildInfoMap.put(BuildInfoKey.CATALOG_PREFIX.name(),    "my.ctg.prefix");
//	   mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE.name(),       "dbSNP");
//	   mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE_BUILD.name(), "GRCh37.p13");
//	   mBuildInfoMap.put(BuildInfoKey.DATA_SOURCE_VERSION.name(),"142");

	   mBuildInfo = new BuildInfo(mBuildInfoMap);
	   mBuildInfo.save(mBuildInfoFile.getCanonicalPath());

	   
	   BuildCatalog build = new BuildCatalog();
	   redirectSysoutAndSyserrToStrings();
	   build.build(mBuildInfoFile.getCanonicalPath(), BuildStepKey.AUTO);
	   resetSysoutAndSyserr();
	   System.out.println(mSysoutBytes.toString());
	   System.err.println(mSyserrBytes.toString());
	   //---------------------------------------------------------------------
   }



}
