package edu.mayo.bior.cli.func;

import java.io.BufferedReader;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Level;

import org.apache.log4j.Logger;

import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;

import com.tinkerpop.pipes.Pipe;
import com.tinkerpop.pipes.util.Pipeline;

import edu.mayo.bior.cli.cmd.VerifyCommandTest;
import edu.mayo.bior.pipeline.createcatalog.TabixCmd;
import edu.mayo.bior.pipeline.createcatalog.TjsonToCatalogPipe;
import edu.mayo.bior.util.StreamConnector;
import edu.mayo.cli.CommandLineApp;
import edu.mayo.cli.CommandPlugin;
import edu.mayo.exec.Command;
import edu.mayo.pipes.MergePipe;
import edu.mayo.pipes.String2HistoryPipe;
import edu.mayo.pipes.WritePipe;
import edu.mayo.pipes.JSON.tabix.BgzipWriter;
import edu.mayo.pipes.UNIX.CatAnythingPipe;
import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.History;
import edu.mayo.pipes.history.HistoryMetaData;
import htsjdk.samtools.util.BlockCompressedInputStream;
import htsjdk.samtools.util.BlockCompressedOutputStream;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;



public abstract class BaseFunctionalTest {

	private static final String ENV_VAR_BIOR_LITE_HOME = "BIOR_LITE_HOME";
	protected static final String TABIX_PATH1 = System.getProperty("user.home") + "/tabix";
	protected static final String TABIX_PATH2 = "/projects/bsi/bictools/apps/alignment/tabix/0.2.5";

	private static boolean isBiorToolkitCmdsRequired = false;

	// NOTE: This static block will be called BEFORE any class that extends this class
	//       (Even before the RemoteFunctionalTest class that extends this one
	// isBiorToolkitCmdsReq - sets the BIOR_LITE_HOME system variable and adds it to the PATH
	public static void setBiorToolkitCmdsRequired(boolean isBiorToolkitCmdsReq) throws FileNotFoundException {
		disableLogging();
		isBiorToolkitCmdsRequired = isBiorToolkitCmdsReq;
		try {
			setup();
		} catch(Exception e) {
			e.printStackTrace();
		}
	}
	
	
	// Disable the logging so it doesn't pollute the test logs and stdout output
	private static void disableLogging() {
	    // For the code below to work, it must be executed before the logger is created!!!
	    //System.setProperty(org.slf4j.impl.Log4jMSimpleLogger.DEFAULT_LOG_LEVEL_KEY, "TRACE");
	    //​org.slf4j.Logger log = LoggerFactory.getLogger(App.class);		

	    //Logger root = (Logger) LoggerFactory.getLogger(Logger.ROOT_LOGGER_NAME);
	    //root.setLevel(Level.INFO);
	    //Logger.getRootLogger().setLevel(Level.FATAL);
	    
		// Prevent Sage info msgs from appearing in the test case outputs
		// Ex:  "INFO edu.mayo.sage.UsageLoggerBuilder - Getting usage logger using server:bsu-sage.mayo.edu port:33124 failOnError:true protocol:UDP"
	    Logger.getLogger(edu.mayo.sage.UsageLogger.class).setLevel(Level.ERROR);
	    Logger.getLogger(edu.mayo.sage.UsageLoggerBuilder.class).setLevel(Level.ERROR);
	}


	public String removeAllLinesStartingWithSlf4j(String stderr) {
		return stderr.replaceAll("SLF4J: .*\n", "");
	}


	/** Given a multi-line output, remove all headers beginning with "#" */
	public String removeHeaders(String stdout) {
		StringBuilder s = new StringBuilder();
		for(String line : stdout.split("\n")) {
			if( s.length() > 0 )
				s.append("\n");
			if( ! line.startsWith("#"))
				s.append(line);
		}
		return s.toString();
	}

	
	// stores the $BIOR_LITE_HOME value
	protected static String sHomePath;

	// UNIX environment variables
	private static Map<String, String> sEnvVars = new HashMap<String, String>();

	
	private static boolean isDebuggingOn = false;
	
	public static void beforeAllForDebugging() {
		if( ! isDebuggingOn )
    		System.out.println("Debugging for Command testing has been disabled.  Set BaseFunctionalTest.isDebuggingOn flag to true to print stdout, stderr, exit, verifyOutput.");
	}

	
	public static void setup() throws Exception {
		System.out.println("\n\n=============== BaseFunctionalTest.setup() ====================\n\n");
		
		// pass along current environment
		sEnvVars.putAll(System.getenv());
		
		final String SEPARATOR = System.getProperty("path.separator");
		String path = sEnvVars.get("PATH") + SEPARATOR
			     + TABIX_PATH1 + SEPARATOR
			     + TABIX_PATH2;
		
		if( isBiorToolkitCmdsRequired ) {
			System.out.println("\n---------- BioR toolkit commands are REQUIRED! --------------\n");
			sHomePath = getBiorDeploymentFolder().getAbsolutePath();
			// setup UNIX environment variables required by bior lite
			sEnvVars.put(ENV_VAR_BIOR_LITE_HOME, sHomePath);
			
			// Add two possible paths for tabix/bgzip to the PATH
			// Also, set the BIOR_LITE_HOME path to the front of the PATH
			path  =  sHomePath + "/bin" + SEPARATOR
				  + path;
			sEnvVars.put("PATH", path);
			
			setEnv(sEnvVars);
		} else {
			System.out.println("\n---------- BioR toolkit commands are NOT required ---------------\n");
		}
		
		System.out.println("Path = " + path);
		System.out.println("sHomePath (BIOR_LITE_HOME) = " + sHomePath);
		System.out.println("sEnvVars = " + sEnvVars);
	}

	
	// From: https://stackoverflow.com/questions/318239/how-do-i-set-environment-variables-from-java
	protected static void setEnv(Map<String, String> newenv) throws Exception {
		try {
			Class<?> processEnvironmentClass = Class.forName("java.lang.ProcessEnvironment");
		    Field theEnvironmentField = processEnvironmentClass.getDeclaredField("theEnvironment");
		    theEnvironmentField.setAccessible(true);
		    Map<String, String> env = (Map<String, String>) theEnvironmentField.get(null);
		    env.putAll(newenv);
		    Field theCaseInsensitiveEnvironmentField = processEnvironmentClass.getDeclaredField("theCaseInsensitiveEnvironment");
		    theCaseInsensitiveEnvironmentField.setAccessible(true);
		    Map<String, String> cienv = (Map<String, String>)     theCaseInsensitiveEnvironmentField.get(null);
		    cienv.putAll(newenv);
		} catch (NoSuchFieldException e) {
		    Class[] classes = Collections.class.getDeclaredClasses();
		    Map<String, String> env = System.getenv();
		    for(Class cl : classes) {
		    	if("java.util.Collections$UnmodifiableMap".equals(cl.getName())) {
		    		Field field = cl.getDeclaredField("m");
		    		field.setAccessible(true);
		    		Object obj = field.get(env);
		    		Map<String, String> map = (Map<String, String>) obj;
		    		map.clear();
		    		map.putAll(newenv);
		    	}
		    }
		}
	}
	
	@Before
	@After
	public void clearHistory() {
	}
	
	/**
	 * Gets array of environment variables in the format "var1=value1" that will
	 * be used in the script to be executed.
	 * 
	 * @return
	 */
	private String[] getEnvironmentVariables() {
		List<String> list = new ArrayList<String>();
		for (String varName: sEnvVars.keySet()) {
			String varValue = sEnvVars.get(varName);
			
			list.add(varName+"="+varValue);
		}
			
		return (String[]) list.toArray(new String[0]);
	}
	
	
	/**
	 * Locates the unzipped distribution folder under target built by  maven.
	 * 
	 * @return
	 * @throws FileNotFoundException thrown if the folder is not found
	 */
	private static File getBiorDeploymentFolder() throws FileNotFoundException {
		
		File deploymentFolder = null;

		// auto-detect inside maven target folder
		File targetFolder = new File("target");
		for (File f: targetFolder.listFiles()) {
			// Directory should be similar to "bior-toolkit_3.0.0", but NOT "bior-toolkit_3.0.0-distribution"
			// (disregard the distribution directory)
			if (f.isDirectory()  &&  f.getName().startsWith("bior-toolkit")  &&  ! f.getName().endsWith("distribution") ) {
				deploymentFolder = f;
			}
		}

		// If we could not get it from the target directory, then figure out $BIOR_LITE_HOME value from system variable
		if( deploymentFolder == null ) {
			String envValue = System.getenv(ENV_VAR_BIOR_LITE_HOME);
			if ((envValue != null) && (envValue.trim().length() > 0)) {
				// use UNIX environment variable if available
				deploymentFolder = new File(envValue);
				System.out.println(
						"WARNING: found $" + ENV_VAR_BIOR_LITE_HOME + " in your environment.  "+
				        "Running functional test against: " + deploymentFolder.getAbsolutePath());
			}
		}
		
		if (deploymentFolder == null) {
			String msg = "Error!  Unable to locate target/bior-toolkit-<version> distribution folder\n"
					+ "Try running:\n"
					+ "  cd bior-toolkit\n"
					+ "  mvn clean package\n"
					+ "  source setupEnv.sh";
			System.err.println(msg);
			throw new FileNotFoundException(msg);
		}
		
		return deploymentFolder;
	}
	
	/**
	 * Executes the specified shell script as a Linux cmd line
	 * 
	 * @param scriptName
	 *            Name of the shell script
	 * @param stdin
	 *            Content typed to STDIN. Set to NULL if not used
	 * @param scriptArgs
	 *            Zero or more arguments sent to the script
	 * @return CommandOutput bean.
	 * @throws IOException
	 * @throws InterruptedException
	 */
	protected CommandOutput executeScript(String scriptName, String stdin,
			String... scriptArgs) throws IOException, InterruptedException
	{
		String[] cmdArgs = getCmdArgs(scriptName, scriptArgs);
		Command cmd = new Command(cmdArgs, sEnvVars, true);
		String EOL = System.getProperty("line.separator");
		List<String> stdInLines = new ArrayList<String>();
		if( stdin != null && stdin.length() > 0 )
			stdInLines.addAll(Arrays.asList(stdin.split(EOL)));
		
		cmd.execute(stdInLines);
		CommandOutput cmdOut = new CommandOutput();
		cmdOut.exit = cmd.getExitCode();
		cmdOut.stderr = cmd.getStderr();
		cmdOut.stdout = cmd.getStdout();

		if( cmdOut.stderr.contains("/bin/sh: " + scriptName + ": command not found") ) {
			System.err.println("\nError: Could not find BioR command.  You may need to create a @BeforeClass method\n"
					+  "in your integration test and call this method to add the BioR commands to the path:\n"
					+  "  BaseFunctionalTest.setBiorToolkitCmdsRequired(true)\n");
		}
		
		return cmdOut;
	}

	
	/** If scriptName is "bior_drill" and scriptArgs are [ "-h", "-l" ], 
	 *  this will return 3 strings where the toolkit command and args are one string:
	 *  [ "/bin/sh", "-c", "bior_drill -h -l" ]
  		See pipeline/Treat/TestCommandLinePipes.java
		Should make this platform agnostic
	 * @param scriptName
	 * @param scriptArgs
	 * @return
	 */
	private String[] getCmdArgs(String scriptName, String[] scriptArgs) {
		StringBuilder str = new StringBuilder(scriptName);
		for(String arg : scriptArgs) {
			str.append("  " + arg);
		}
		return new String[] { "/bin/sh", "-c", str.toString() };
	}

	/**
	 * Executes the specified shell script that may contain pipes.
	 * Ex:  "cat myfile.vcf | bior_vcf_to_json | bior_drill -k -p CHROM" 
	 * 
	 * @param cmdWithPipes  Command to run (may contain pipes)
	 * @return CommandOutput Output from command - contains ALL stderr lines, but only stdout from last pipe operation.
	 * @throws IOException
	 * @throws InterruptedException
	 */
	protected CommandOutput executeScriptWithPipes(String cmdWithPipes) throws IOException, InterruptedException {
		// Split the command into the pipe-able pieces (ex: "cat junk.vcf | bior_vcf_to_json" 
		// will be broken up into two commands and run as two separate commands)
		// NOTE: leading and trailing spaces around the pipe char will be removed
		String[] cmds = cmdWithPipes.split(" +\\| +");
		
		// Create the CommandOutput and set it's stderr to "" so we don't have a "null" in there
		CommandOutput out = new CommandOutput();
		out.stderr = "";
		
		String stdin = null;
		
		for(String cmd : cmds) {
			// If the command is a bior command (begins with "bior_"), the prepend the path to the cmd
			if(cmd.startsWith("bior_"))
				cmd = sHomePath + "/bin/" + cmd;

			
			Process p = Runtime.getRuntime().exec(cmd, getEnvironmentVariables());
	
			// connect STDERR from script process and store in local memory
			// STDERR [script process] ---> local byte array
			ByteArrayOutputStream stderrData = new ByteArrayOutputStream();
			StreamConnector stderrConnector = new StreamConnector(p.getErrorStream(), stderrData, 1024);
			Thread stderrThread = new Thread(stderrConnector);
			stderrThread.start();
	
			// connect STDOUT from script process and store in local memory
			// STDOUT [script process] ---> local byte array
			ByteArrayOutputStream stdoutData = new ByteArrayOutputStream();		
			StreamConnector stdoutConnector = new StreamConnector(p.getInputStream(), stdoutData, 1024);
			Thread stdoutThread = new Thread(stdoutConnector);
			stdoutThread.start();
	
			// feed STDIN into process if necessary
			if (stdin != null) {
				p.getOutputStream().write(stdin.getBytes());
				p.getOutputStream().close();
			}
			
			// block until process ends
			int exitCode = p.waitFor();

			// wait for threads to finish
			stdoutThread.join();
			stderrThread.join();
			
			String stderr = stderrData.toString("UTF-8");
			String stdout = stdoutData.toString("UTF-8");

			out.stderr += stderr;
			out.stdout = stdout;
			out.exit = exitCode;
			
			if(exitCode != 0) {
				return out;
			}
			
			// Set the stdin for the next pipe to the output from the previous one
			stdin = out.stdout;
		}
		return out;
	}
	
	/**
	 * Loads contents of a file into a String object.
	 * 
	 * @param f
	 * @return
	 * @throws IOException
	 */
	protected String loadFile(File f) throws IOException {
		StringWriter sWtr = new StringWriter();
		PrintWriter pWtr = new PrintWriter(sWtr);
		BufferedReader br = new BufferedReader(new FileReader(f));

		String line = br.readLine();
		while (line != null) {
			pWtr.println(line);
			line = br.readLine();
		}
		pWtr.close();

		return sWtr.toString();
	}

	/**
	 * Extracts any rows that begin with a '#' character
	 * 
	 * @param s
	 * @return
	 * @throws IOException
	 */
	protected String getHeader(String s) throws IOException {
		
		StringReader sRdr = new StringReader(s);
		BufferedReader bRdr = new BufferedReader(sRdr);
		
		StringWriter sWtr = new StringWriter();
		PrintWriter pWtr = new PrintWriter(sWtr);
		
		String line = bRdr.readLine();
		while (line != null) {
			if (line.startsWith("#")) {
				pWtr.println(line);
			}
			line = bRdr.readLine();
		}
		
		pWtr.close();
		return sWtr.toString();
	}
	
	
	public static String concat(String... col) {
		StringBuilder str = new StringBuilder();
		for(int i=0; i < col.length; i++) {
			if( i > 0 )
				str.append("\t");
			str.append(col[i]);
		}
		return str.toString();
	}
	
	public static String swapQuotes(String s) {
		return s.replaceAll("'", "\"");
	}
	
	protected void setStdin(String s) {
		InputStream fakeIn = new ByteArrayInputStream(s.getBytes());
		System.setIn(fakeIn);
	}

	public static void assertContains(String fullText, String substr) {
		String msg = "\n" 
				+    "Could not find substring: ====================================================\n"
				+    substr  + "\n\n"
				+    "Within: ======================================================================\n"
				+    fullText + "\n"
				+    "==============================================================================\n";
		assertTrue(msg, fullText.contains(substr));
	}
	
	public static void assertContainsAll(String fullText, List<String> substrings) {
		StringBuilder msg = new StringBuilder("Assertion error: Output did NOT contain these lines (but should have):\n");
		msg.append("------------- Expected lines ------------------------\n");
		boolean isMissingAny = false;
		for(String s : substrings) {
			if( ! fullText.contains(s) ) {
				isMissingAny = true;
				msg.append(s + "\n");
			}
		}
		msg.append("------------- Actual output: ------------------------\n");
		msg.append(fullText + "\n");
		msg.append("-----------------------------------------------------\n");

		assertFalse(msg.toString(), isMissingAny);
	}

	public static void assertContainsNone(String fullText, List<String> substrings) {
		StringBuilder msg = new StringBuilder("Assertion error: Output contained these lines (but should NOT have):\n");
		msg.append("------------- Unexpected lines ----------------------\n");
		boolean isContainsAny = false;
		for(String s: substrings) {
			if( fullText.contains(s) ) {
				isContainsAny = true;
				msg.append(s + "\n");
			}
		}
		msg.append("------------- Actual output: ------------------------\n");
		msg.append(fullText + "\n");
		msg.append("-----------------------------------------------------\n");
		
		assertFalse(msg.toString(), isContainsAny);
	}


	
	/**
	 * Execute the equivalent of the Linux cmd line script, but purely in Java so we can debug it
	 * @param commandClassInstance  A class that implements the CommandPlugin class (ex: CreateCatalogCommand)
	 * @param cmdName The name of the BioR script (ex: "bior_create_catalog")
	 * @param cmdArgs Any flags or arguments to the command
	 * @return
	 * @throws UnsupportedEncodingException
	 */
	public static CommandOutput runCmdApp(String stdin, CommandPlugin commandClassInstance, String cmdName, String... cmdArgs) throws UnsupportedEncodingException {
		CommandLineApp app = new CommandLineApp();
        app.captureSystemOutAndErrorToStrings();
        CommandPlugin mockPlugin = commandClassInstance;
    	
        if( stdin != null )
        	app.setStdIn(stdin);
        
        CommandOutput output = new CommandOutput();
        output.exit = app.runApplication(mockPlugin.getClass().getName(), /*MOCK_SCRIPT_NAME=*/cmdName, cmdArgs);
        // Set SYSOUT and SYSERR back to their original output streams
        app.resetSystemOutAndError();

        output.stdout = app.getSystemOutMessages();
        output.stderr = app.getSystemErrorMessages();
    
        return output;
	}

	/**
	 * Mock-call to the command line script, but does not actually require us to call command line scripts
	 * @param commandClassInstance  A class that implements the CommandPlugin class (ex: CreateCatalogCommand)
	 * @param cmdName The name of the BioR script (ex: "bior_create_catalog")
	 * @param cmdArgs Any flags or arguments to the command
	 * @return
	 * @throws UnsupportedEncodingException
	 */
	public static CommandOutput runCmdApp(CommandPlugin commandClassInstance, String cmdName, String... cmdArgs) throws UnsupportedEncodingException {
		return runCmdApp(null, commandClassInstance, cmdName, cmdArgs);
	}
 
	
	/** Create a History object with the merged header (separated by tabs) */
	public static History createHistoryWithHeader(String mergedHeader) {
		History history = new History();
		history.setMetaData(new HistoryMetaData(null));
		history.getMetaData().setOriginalHeader(null);
		if( mergedHeader == null || mergedHeader.length() == 0 )
			return history;
		
		List<String> header = new ArrayList<String>();
		header.addAll(Arrays.asList(mergedHeader.split("\t")));
		HistoryMetaData historyMetaData = new HistoryMetaData(header);
		historyMetaData.setOriginalHeader(header);
		history.setMetaData(historyMetaData);
		for(String headerName : header) {
			historyMetaData.getColumns().add(new ColumnMetaData(headerName));
		}
		return history;
	}

	
	
	protected void createCatalog(List<String> inputLines, File catalogBgzipFile) throws IOException, InterruptedException {
		createCatalog(inputLines, catalogBgzipFile, /*isCreateTabixIndex=*/true);
	}

		
	protected void createCatalog(List<String> inputLines, File catalogBgzipFile, boolean isCreateTabixIndex) throws IOException, InterruptedException {
        Pipeline pipeline = new Pipeline(
        		new String2HistoryPipe("\t"),								// String  -> History
        		new BgzipWriter(catalogBgzipFile.getCanonicalPath())
        		);
        
        pipeline.setStarts(inputLines);
        while( pipeline.hasNext() ) {
        	pipeline.next();
        }
        
        if( isCreateTabixIndex )
        	TabixCmd.createTabixIndex(catalogBgzipFile.getCanonicalPath());
	}
	
	public static void writeBgzip(List<String> strList, File bgzipOutputFile) throws IOException {
		BlockCompressedOutputStream fout = null;
		try {
			fout = new BlockCompressedOutputStream(bgzipOutputFile);
			for(String s : strList)
				fout.write((s+"\n").getBytes());
		} finally {
			if( fout != null )
				fout.close();
		}
	}
	
	public static List<String> readBgzip(File bgzipFile) throws IOException {
		List<String> strList = new ArrayList<String>();
		BufferedReader fin = null;
		try {
			if( bgzipFile.getName().endsWith(".gz")  ||  bgzipFile.getName().endsWith(".bgz") ) {
				fin = new BufferedReader(new InputStreamReader(new BlockCompressedInputStream(bgzipFile)));
			} else {
				fin = new BufferedReader(new FileReader(bgzipFile));
			}
			String line = null;
			while( (line = fin.readLine()) != null ) {
				strList.add(line);
			}
		} finally {
			if( fin != null )
				fin.close();
		}
		return strList;
	}
	
    //=============================================================================================================
    public static void printOutputsForDebugging(CommandOutput output, String logText) {
    	if( ! isDebuggingOn )
    		return;
    		
    	
    	System.out.println("exitCode: " + output.exit);

    	System.out.println("stdout:------------------------------------");
    	System.out.println(output.stdout);
    	
    	System.out.println("stderr:------------------------------------");
    	System.out.println(output.stderr);
    	
    	System.out.println("logText output:-----------------------------");
    	System.out.println(logText);
	}

}
