package edu.mayo.bior.cli.cmd;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.FileFilter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;

import org.apache.commons.io.FileUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import com.jayway.jsonpath.Criteria;
import com.jayway.jsonpath.DocumentContext;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.Predicate;

import edu.mayo.bior.buildcatalog.BuildStepKey;
import edu.mayo.bior.cli.func.BaseFunctionalTest;
import edu.mayo.bior.cli.func.CommandOutput;
import edu.mayo.pipes.util.BiorProperties;

/** Test the BuildCatalogCommand class used by bior_build_catalog */
public class BuildCatalogCommandTest  extends  BaseFunctionalTest{

	@Rule
	public TemporaryFolder tempFolder = new TemporaryFolder();
	
	private File tempDir;
	private File targetDir;
	private File sharedClusterDir;
	private File makeJsonFile;
	private File buildInfoTextFile;
	private File logFile;
	
	private File biorLiteHome;
	private static String biorPropertiesPathBefore;
	
	@Before
	public void beforeEach() throws IOException {
		tempFolder.create();
		tempDir = tempFolder.newFolder("Temp");
		targetDir = tempFolder.newFolder("Target");
		sharedClusterDir = tempFolder.newFolder("Shared");
		BaseFunctionalTest.setBiorToolkitCmdsRequired(false);
		
		logFile = new File(tempDir, "bior.log");
		//biorLiteHome = getBiorLiteHome();
		
		makeJsonFile = createMakeJsonFile(tempDir);
		buildInfoTextFile = createBuildInfoFile(tempDir, makeJsonFile);
		
		// Override the paths to RCF system /data5 dirs, and use local ones for testing
		biorPropertiesPathBefore = BiorProperties.getFile();
		BiorProperties.setFile("src/test/resources/bior.properties.test");
	}
	
	@After
	public void afterEach() {
		BiorProperties.setFile(biorPropertiesPathBefore);
	}
	
	@Test
	public void testBuildCatalogCommand_makeJsonThruPropsSteps() throws IOException {
		CommandOutput out = runMakeJsonToMergePropsSteps();
		printCmdOutput(out);
		assertEquals("STDERR:\n" + out.stderr + "\n\nSTDOUT:\n" + out.stdout, 0,  out.exit);
		
		assertContainsAll(out.stdout, Arrays.asList(
				"Step make_json SUCCEEDED",
				"Step make_catalog SUCCEEDED",
				"Step make_prop_files SUCCEEDED",
				"Step merge_prop_files SUCCEEDED"
				) );
		assertTrue(out.stdout.trim().endsWith("Step merge_prop_files SUCCEEDED\n-------------------------------------"));
	}

	
	@Test
	public void testBuildCatalogCommand_verifyStepOnly() throws IOException {
		CommandOutput makeJsonOut = runMakeJsonToVerifySteps();
		CommandOutput verifyCmdOut = runVerifyStep();
		printCmdOutput(verifyCmdOut);
		String verifyTxtOutput = getVerifyTxtContents(verifyCmdOut.stdout);
		assertContains(verifyTxtOutput, "ERROR 501: _refAllele [A] does not match reference sequence value [C] for:  chr=5 position=11621");
		assertContains(verifyTxtOutput, "ERROR 504: Chromosome 5: _refAllele values that DO NOT match the reference sequence is [1].");

		assertEquals(0,  verifyCmdOut.exit);

		assertContainsAll(verifyCmdOut.stdout, Arrays.asList(
				"Verified 2 of total 2 rows of catalog",
				"Verify #ERROR: 2, #WARNING: 3.",
				"Step verify SUCCEEDED"
				) );
	}
	
	private String getVerifyTxtContents(String verifyCmdStdout) throws IOException {
		final String PREFIX = "See details in '";
		int idx1 = verifyCmdStdout.indexOf(PREFIX);
		int idx2 = verifyCmdStdout.indexOf("'", idx1 + PREFIX.length());
		if( idx1 == -1 || idx2 == -1 ) {
			System.err.println("Could not find the verify.txt filename in the output:\n" + verifyCmdStdout);
			return "";
		}
		String filename = verifyCmdStdout.substring(idx1 + PREFIX.length(), idx2);
		return FileUtils.readFileToString(new File(filename));
	}

	@Test
	public void testJaywayJson() {
		Predicate pred = new Predicate() {
			public boolean apply(PredicateContext ctx) {
				return true;
			}
		};
		
		JsonPath xJsonPath = JsonPath.compile("x");
        String out = ((String)xJsonPath.read("{\"x\":\"chr1\"}")).toLowerCase();
        assertEquals("chr1", out);
        
		JsonPath xJsonPath2 = JsonPath.compile("x", pred);
        String out2 = ((String)xJsonPath2.read("{\"x\":\"chr2\"}")).toLowerCase();
        assertEquals("chr2", out2);

		Predicate predNo = new Predicate() {
			public boolean apply(PredicateContext ctx) {
				return false;
			}
		};
        JsonPath xJsonPath3 = JsonPath.compile("x", predNo);
        String out3 = ((String)xJsonPath3.read("{\"x\":\"chr3\"}")).toLowerCase();
        // NOTE: Why is this still "chr3" when the predicate returns FALSE?
        assertEquals("chr3", out3);

	}
	
	
	private CommandOutput runMakeJsonToMergePropsSteps() throws IOException {
		CommandOutput out = runCmdApp(new BuildCatalogCommand(), "bior_build_catalog",
				"-b", buildInfoTextFile.getCanonicalPath(),
				"-lf",logFile.getAbsolutePath(),
				"-s", BuildStepKey.MAKE_JSON.name(),  		// Start step
				"-e", BuildStepKey.MERGE_PROP_FILES.name()  // Stop  step
				);
		return out;
	}

	private CommandOutput runMakeJsonToVerifySteps() throws IOException {
		CommandOutput out = runCmdApp(new BuildCatalogCommand(), "bior_build_catalog",
				"-b", buildInfoTextFile.getCanonicalPath(),
				"-lf",logFile.getAbsolutePath(),
				"-s", BuildStepKey.MAKE_JSON.name(),  		// Start step
				"-e", BuildStepKey.VERIFY.name()  // Stop  step
				);
		return out;
	}

	private CommandOutput runVerifyStep() throws IOException {
		CommandOutput out = runCmdApp(new BuildCatalogCommand(), "bior_build_catalog",
				"-b", buildInfoTextFile.getCanonicalPath(),
				"-lf",logFile.getAbsolutePath(),
				"-s", BuildStepKey.VERIFY.name(), // Start step
				"-e", BuildStepKey.VERIFY.name()  // Stop  step
				);
		return out;
	}
	
	private void printCmdOutput(CommandOutput out) {
		System.out.println("BuildCatalogCommandTest - Command output:");
		System.out.println("Exit code:-------------------------");
		System.out.println(out.exit);
		System.out.println("Stdout:-------------------------");
		System.out.println(out.stdout);
		System.out.println("Stderr:-------------------------");
		System.out.println(out.stderr);
	}

	private File createMakeJsonFile(File tmpDir) throws IOException {
		makeJsonFile = new File(tmpDir, "make_json.sh");
		FileUtils.write(makeJsonFile,
				"#!/bin/bash\n"
			  + "echo MAKE_JSON_OUTPUT_FILE_PATH = $MAKE_JSON_OUTPUT_FILE_PATH\n"
			  // NOTE: The ref alleles at these coordinates should match those in src/test/resources/ref_assembly/GRCh37/genome_GRCh37.tsv
			  // OR at least the first should match, but we'll make the one on chr5 NOT match to throw a warning  (ref is 'C' in the ref assembly at that position)
			  + "echo -e '1\t10011\t10011\t" + "{'_landmark':'1','_minBP':10011,'_maxBP':10011,'_refAllele':'C','_altAlleles':['A'],'X':1}".replace("'", "\"") + "' >  $MAKE_JSON_OUTPUT_FILE_PATH\n"
			  + "echo -e '5\t11621\t11621\t" + "{'_landmark':'5','_minBP':11621,'_maxBP':11621,'_refAllele':'A','_altAlleles':['C'],'X':2}".replace("'", "\"") + "' >> $MAKE_JSON_OUTPUT_FILE_PATH\n"
			 );
		makeJsonFile.setExecutable(true);
		System.out.println("BuildCatalogCommandTest.testBuildCatalogCommand_verifyStepOnly(): makeJsonFile path: " + makeJsonFile.getAbsolutePath());
		return makeJsonFile;
	}

	//=========================================================
	private File createBuildInfoFile(File tmpDir, File mkJsonFile) throws IOException {
		File buildInfoFile = new File(tmpDir, "build_info.txt");
		FileUtils.write(buildInfoFile, 
				"DATA_SOURCE_BUILD=GRCh37\n"
			 +  "DATA_SOURCE=ClinVar\n"
			 +  "MAKE_JSON_ARGS=\n"
			 +  "MAKE_JSON_SCRIPT_PATH=" + mkJsonFile.getAbsolutePath() + "\n"
			 +  "CATALOG_PREFIX=macarthur-lab_xml\n"
			 +  "DATA_SOURCE_VERSION=2018-03\n"
			 +  "TARGET_DIR=" + this.targetDir.getAbsolutePath() + "\n"
			 +  "TEMP_DIR=" + this.tempDir.getAbsolutePath() + "\n"
			 +  "SHARED_CLUSTER_DIR=" + this.sharedClusterDir.getAbsolutePath() + "\n"
			 //+  "BIOR_LITE_HOME=" + this.biorLiteHome.getAbsolutePath() + "\n"
			 );
		System.out.println("BuildCatalogCommandTest.testBuildCatalogCommand_verifyStepOnly(): buildInfoFile path: " + buildInfoFile.getAbsolutePath());
		return buildInfoFile;
	}
	
//	private File getBiorLiteHome() throws IOException {
//		String env = System.getenv("BIOR_LITE_HOME");
//		String prop = System.getProperty("BIOR_LITE_HOME");
//		
//		System.out.println("BIOR_LITE_HOME (env):  " + env);
//		System.out.println("BIOR_LITE_HOME (prop): " + prop);
//		
//		String targetBiorDir = System.getProperty("user.dir") + "/target";
//		
//		File[] files = new File(targetBiorDir).listFiles(new FileFilter() {
//			public boolean accept(File file) {
//				return   file.getName().startsWith("bior-toolkit-")
//				    && ! file.getName().endsWith("-distribution")
//				    &&   file.isDirectory()
//				    &&   file.exists();
//			}
//		});
//		if( files == null  ||  files.length == 0 )
//			throw new IOException("Cannot find the BIOR_LITE_HOME directory from the current project.  Looking for: " + targetBiorDir + "/bior-toolkit-*/");
//		File biorHome = files[0];
//		System.out.println("BIOR_LITE_HOME (target): " + biorHome.getAbsolutePath());
//		return biorHome;
//	}
}
