package edu.mayo.bior.buildcatalog;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.IOException;

import org.apache.commons.io.FileUtils;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

/** Tests for Logger for each step in the Catalog build process
 * @author Michael Meiners (m054457)
 * 2016-04-27  */
public class StepLoggerTest {

	
	@Test
	public void test() throws IOException {
		TemporaryFolder folder = new TemporaryFolder();
		folder.create();
		File tempFolder = folder.newFolder();
		
		StepLogger log = new StepLogger(BuildStepKey.MAKE_CATALOG.toString(), tempFolder);

		// Summary and log should be blank before writing to them  (though they will exist because the printwriters will create them
		assertTrue(log.getSummaryFile().exists());
		assertTrue(log.getLogFile().exists());
		assertEquals("", log.getSummary());
		assertEquals("", FileUtils.readFileToString(log.getLogFile()));
		
		log.log("Count to 10");
		log.log("1,2,3,4,5,6,7,8,9,10");
		log.summary("summary:");
		log.summary("1,2,skipAFew,10");
		

		// Done writing
		log.closeLogWriters();

		// Strip the date-timestamp off the front of each line to make it easier to compare
		String summaryStr = removeFirstXCharsFromEachLine(log.getSummary(), StepLogger.DATE_FORMAT_STR.length());
		assertEquals("summary:\n1,2,skipAFew,10\n", summaryStr);
		
		// Strip the date-timestamp off the front of each line
		String logStr = removeFirstXCharsFromEachLine(FileUtils.readFileToString(log.getLogFile()), StepLogger.DATE_FORMAT_STR.length());
		assertEquals("Count to 10\n1,2,3,4,5,6,7,8,9,10\n", logStr);
		
	}
	
	@Test
	public void testRemovalOfFirstXChars() {
		String in = "1\n123\n12345\n34567\n";
		String out = removeFirstXCharsFromEachLine(in, 3);
		// NOTE: The 1 is maintained because the line is not long enough
		assertEquals("1\n\n45\n67\n", out);
	}
	
	/** Removes the first X number of characters off the beginning of each line.
	 *  This is useful for removing the first 25 characters that make up the timestamp in the log and summary files 
	 * @param line  The line to remove the first X chars from
	 * @param numChars  Number of characters to remove from the beginning of each line
	 * @return
	 */
	protected static String removeFirstXCharsFromEachLine(String line, int numChars) {
		int idxNewline = line.indexOf("\n");
		// Remove the first x chars of the first line
		if( idxNewline != -1  &&  idxNewline >= numChars )
			line = line.substring(numChars);
		// Remove the first x chars of each subsequent line
		line = line.replaceAll("\n.{" + numChars + "}", "\n");
		return line;
	}
}
