package edu.mayo.bior.util.validation;

import edu.mayo.bior.catalog.GoldenAttribute;
import edu.mayo.cli.InvalidDataException;
import edu.mayo.pipes.JSON.modify.JsonTypeExtended;

public class GoldenValidator {

	/** getGoldenAttribute: for the input string, looks for the golden attribute
	 *                      that matches the string, to verify it, and then return
	 *                      the GoldenAttributes enum object for that string.
	 *                      Returns null if not a valid golden attribute value.
	 * @param String that is a BIOR golden JSON attribute key
	 * @return Returns null if not a valid golden attribute value.
	 * @throws InvalidDataException
	 */
	public static GoldenAttribute getGoldenAttribute(String goldenJsonKey) throws InvalidDataException {
		try {
			return GoldenAttribute.valueOf(goldenJsonKey);
		} catch(Exception e) {
			return null;
		}
	}
	
	public static boolean isValidGolden(String inputGoldenKey) throws InvalidDataException {
		return null != getGoldenAttribute(inputGoldenKey);
	}

	/** validateGoldenKey: for the input string, looks for the golden attribute
	 *                      that matches the string, to verify it, and then returns
	 *                      if that string matches one of the BIOR golden attribute 
	 *                      key names. 
	 *                      Throws an InvalidDataException exception if is not a valid 
	 *                      golden attribute value.
	 * @param String that is a BIOR golden JSON attribute key
	 * @return 
	 * @throws InvalidDataException
	 */
	public static void validateGoldenKey(String goldenJsonKey) throws InvalidDataException  {
		if( null == getGoldenAttribute(goldenJsonKey) ) {
			throw new InvalidDataException("Not a valid BioR golden attribute name: " + goldenJsonKey);
		}
	}
	
	/** validateDataType: throws an exception if the data type is not valid
	 *  data type for the input golden attribute.
	 * 
	 */
	public static void validateDataType(String goldenJsonKey, String dataType) throws InvalidDataException {
		GoldenAttribute goldenKey = getGoldenAttribute(goldenJsonKey);
		validateDataType(goldenKey, dataType);
	}
	
	/** validateDataType: throws an exception if the data type is not valid
	 *  data type for the input golden attribute.
	 * 
	 */
	public static void validateDataType(GoldenAttribute goldenJsonKey, String dataType) throws InvalidDataException {
		switch(goldenJsonKey) {
		case _minBP:      // Same as _maxBP
		case _maxBP: 
			// In the TAB_TO_TJSON config, the data type should be "NUMBER" according to help text.
			if ( ! isJsonTypeNumeric(dataType) ) { 
				throw new InvalidDataException(goldenJsonKey.name() + " golden attribute must be of type 'Integer'.");
			}
			break;
		case _landmark:   // Same as _strand
		case _id:         // Same as _strand
		case _refAllele:  // Same as _strand
		case _altAlleles: // Same as _strand
		case _type:       // Same as _strand
		case _strand: 
			if ( ! dataType.equalsIgnoreCase("STRING")) { 
				throw new InvalidDataException(goldenJsonKey.name() + " golden attribute must be of type String.");
			}
			break;
		}
	}
	
	public static boolean isJsonTypeNumeric(String jsonType) {
		return JsonTypeExtended.FLOAT.toString().equalsIgnoreCase(jsonType)
			|| JsonTypeExtended.INTEGER.toString().equalsIgnoreCase(jsonType)
			|| "NUMBER".equalsIgnoreCase(jsonType);  // "NUMBER" for backwards compatibility
	}
}
