/**
 * bior_pipeline
 *
 * <p>@author Gregory Dougherty</p>
 * Copyright Mayo Clinic, 2014
 *
 */
package edu.mayo.bior.util;

/**
 * Interface for call to process a field from a catalog.  Example use cases:<br/>
 * Have data that need to clean up (i.e. 1000 Genomes MAFs come back surrounded by [])<br/>
 * Have data that's an enumeration, and want the text returned, not the number (i.e. 0 = "Unknown")
 *
 * <p>@author Gregory Dougherty</p>
 */
public interface FieldProcessor
{
	/** String value that will be ignored / treated as null is treated */
	public static final String	kBlank = ".";
	/** Default String value to return when input is {@code kBlank} */
	public static final String	kBlankValue = "NA";
	
	/**
	 * Add a String to be recognized as a blank by all {@linkplain FieldProcessor}s, 
	 * in addition to all existing blanks
	 * 
	 * @param value	String to be recognized as blank.  Null will be ignored
	 */
	public void addBlank (String value);
	
	
	/**
	 * Set a String to be recognized as the only non-null blank by all {@linkplain FieldProcessor}s, 
	 * deleting all existing blanks
	 * 
	 * @param value	String to be recognized as blank.  If null will keep anything but null from 
	 * being recognized as blank
	 */
	public void setBlank (String value);
	
	
	/**
	 * Set a String that all {@linkplain FieldProcessor}s will returned when find a blank, can be null
	 * 
	 * @param value	String to be returned when find a blank, can be null
	 */
	public void setBlankValue (String value);
	
	
	/**
	 * Get all the strings that all {@linkplain FieldProcessor}s recognize as blank, other than null
	 * 
	 * @return	Array of non-null strings
	 */
	public String[] getBlanks ();
	
	
	/**
	 * Process a String input, returning the String translation, or null if the input should be NA
	 * 
	 * @param value	Input to parse, can be null
	 * @return	Processed output
	 */
	public String process (String value);
}
