package edu.mayo.bior.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;

import edu.mayo.cli.InvalidDataException;
import edu.mayo.pipes.history.History;

/** Utilities to convert one-based and negative column numbers to zero-based columns, 
 *  as well as converting column ranges from a format such as "2..3,5,10,-1" to a list of one-based columns
 */
public class ColumnResolver {
	
	private static final String TAB_DELIMITER = "\t";
	
	
	/** 
	 * Resolve column ranges to zero-based indexes
	 *  Ex: If we have 10 columns (0 to 9 internally), then the range "1..3,2..4,-1..-3" will resolve to: 0,1,2,3,7,8,9
	 *  Likewise, column names can be used and will be resolved to 0-based indexes
	 *  
	 * @param history	{@linkplain History} object to provide column data
	 * @param ranges	String to parse for column data
	 * @return	{@linkplain List} of 0-based columns
	 * @throws Exception
	 */
	static public List<Integer> rangesToZeroBasedIndexes(History history, String ranges) throws Exception {
		String header = history.getMetaData().getColumnHeaderRow(history, TAB_DELIMITER);
		List<Integer> positiveCols = new ColumnResolver().getColumnsFromRanges(header, ranges);
		
		// subtract 1 from all positive columns to make them zero-based
		List<Integer> zeroBasedCols = new ArrayList<Integer>();
		for(int i=0; i < positiveCols.size(); i++)
			zeroBasedCols.add(positiveCols.get(i)-1);
		return zeroBasedCols;
	}
	
	/**
	 * Routine to take a column name and give you a column number
	 * 
	 * @param history		{@linkplain History} object to provide column data
	 * @param columnName	Name of column we're interested in
	 * @return	Nothing, it's not yet implemented
	 * @throws Exception	Always
	 */
	static public Integer nameToZeroBasedIndex(History history, String columnName) throws Exception {
		throw new Exception("Not implemented yet");
	}
	
	/** Return the 0-based positive column number for the one passed in. 
	 *  Input column could be negative or positive.
     *    -1 is last column, (so if 10 columns, this returns 9) 
     *     1 is first column (returns 0)
     *     0 is not allowed  (throws error)
	 * @param history 
	 * @param columnNumber_oneBased 
	 * @param lineNum 
     *  @return  Returns the 0-based index into the history object for use by java     */
	static public Integer numberToZeroBasedIndex(History history, int columnNumber_oneBased, long lineNum)  {
		// Throw a readable error if col is 0, less than -size, or greater than/equal to +size
		if( columnNumber_oneBased == 0  ||  columnNumber_oneBased < -history.size()  ||  columnNumber_oneBased > history.size() ) {
			throw new IndexOutOfBoundsException("Error on line " + lineNum 
					+ ".  Specified column index is beyond the range of the columns in the data row"
					+ ".  Specified column:  " + columnNumber_oneBased
					+ ".  Size of data row: "  + history.size());
		}

		// Now, convert it into a positive column, 0-based
		int col = columnNumber_oneBased;
		if( col < 0 )
			col = history.size() + columnNumber_oneBased;
		else if( col > 0)
			col = columnNumber_oneBased - 1;

		return col;
	}
	
	
	
	/** Convert a string like "2..3,5,10,-1" into a list of positive columns.
	 *  These columns will later be pulled into the INFO column and those columns removed.
	 */
	protected List<Integer> getColumnsFromRanges(String header, String ranges) throws InvalidDataException {
		if( header == null || header.trim().length() == 0 || header.trim().equals("#") ) 
			throw new InvalidDataException("Error: Invalid column - no header found");
		
		if( ranges == null || ranges.trim().length() == 0 )
			return new ArrayList<Integer>();
		
		int numColsInHeader = header.split("\t").length;
		String[] rangesArray = ranges.split(",");
		List<Integer> cols = new ArrayList<Integer>();
		for(String range : rangesArray) {
			// Skip any blank range (so if user left a starting or trailing comma)
			if( range.trim().length() == 0 )
				continue;
			cols.addAll(getColsFromSingleRange(range, numColsInHeader));
		}
		removeDuplicates(cols);
		Collections.sort(cols);
		return cols;
	}
	
	private void removeDuplicates(List<Integer> cols) {
		HashSet<Integer> colsHashSet = new HashSet<Integer>();
		for(int i=cols.size()-1; i>=0; i--) {
			if( colsHashSet.contains(cols.get(i)) )
				cols.remove(i);
			else
				colsHashSet.add(cols.get(i));
		}
	}

	/** Get all columns from a single range (no commas separating multiple ranges)  Ex: 2..3 (not 2..3,4,5..6)
	 * @throws InvalidDataException 
	 * @throws NumberFormatException */
	protected List<Integer> getColsFromSingleRange(String range, int numColsInHeader) throws NumberFormatException, InvalidDataException {
		List<Integer> cols = new ArrayList<Integer>();
		if( isInteger(range) ) {
			cols.add(getPositiveCol(range, Integer.parseInt(range), numColsInHeader));
		}
		// Check for sequential range (ex: 8..10)
		else if( range.contains("..") ) {
			try {
				String[] vals = range.trim().split("\\.\\.");
				// Could have "..3" for range (meaning cols 1,2,3)
				if( range.trim().startsWith("..") )
					vals = new String[] { "1", vals[1] };
				// Could have "3.." for range (meaning columns 3,4,5...end)
				// In this case, there will be only one element in the array
				if( range.trim().endsWith("..") )
					vals = new String[] { vals[0], numColsInHeader + "" };
					
				Integer int1 = getPositiveCol(range, Integer.parseInt(vals[0]), numColsInHeader);
				Integer int2 = getPositiveCol(range, Integer.parseInt(vals[1]), numColsInHeader);
				
				Integer first = Math.min(int1, int2);
				Integer last  = Math.max(int1, int2);
				
				for(int i=first; i <= last; i++)
					cols.add(i);
			} catch(InvalidDataException e) {
				// Just rethrow InvalidDataExceptions since the values are probably out of range
				throw e;
			} catch(Exception e) {
				throw new InvalidDataException("Error: Invalid column or range [" + range + "] - range could not be parsed");
			}
		} else {
			throw new InvalidDataException("Error: Invalid column or range [" + range + "] - range could not be parsed");			
		}
		
		return cols;
	}

	protected int getPositiveCol(String range, int colPosOrNeg, int numColsInHeader) throws InvalidDataException {
		if( colPosOrNeg > numColsInHeader )
			throw new InvalidDataException("Error: Invalid column or range [" + range + "] - header only contains " + numColsInHeader + " columns");
		
		if( colPosOrNeg < 0 )
			colPosOrNeg = numColsInHeader + colPosOrNeg + 1;
		
		// If it's STILL negative after trying to make it positive...
		if( colPosOrNeg < 0 )
			throw new InvalidDataException("Error: Invalid column or range [" + range + "] - header only contains " + numColsInHeader + " columns");

		if( colPosOrNeg == 0 )
			throw new InvalidDataException("Error: Invalid column or range [" + range + "] - 0 is not a valid column number.  Column numbers start at 1");

		return colPosOrNeg;
	}
	
	protected boolean isInteger(String str) {
		try {
			Integer.parseInt(str);
			return true;
		} catch(Exception e) {
			return false;
		}
	}
	
}
