package edu.mayo.bior.pipeline.createcatalog;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.zip.GZIPInputStream;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.code.externalsorting.ExternalSort;
import com.jayway.jsonpath.InvalidPathException;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.Predicate;
import com.tinkerpop.pipes.Pipe;
import com.tinkerpop.pipes.IdentityPipe;
import com.tinkerpop.pipes.util.Pipeline;

import edu.mayo.bior.catalog.HumanBuildAssembly;
import edu.mayo.pipes.MergePipe;
import edu.mayo.pipes.String2HistoryPipe;
import edu.mayo.pipes.WritePipe;
import edu.mayo.pipes.JSON.tabix.BgzipWriter;
import edu.mayo.pipes.UNIX.CatAnythingPipe;
import edu.mayo.pipes.util.GenomicObjectUtils;
import htsjdk.samtools.util.BlockCompressedInputStream;
import net.minidev.json.JSONArray;

/** Given a plain-text file (or gzip or bgzip file) that has a JSON column, convert that file to a catalog that has the format:
 *   [CHROM]  [MIN_BP]  [MAX_BP]  [JSON]
 *  Or, if the flag is passed in to create a JSON-only catalog, then format is:
 *   [JSON]
 *    and
 *     - don't extract the chrom,min,max;
 *     - ignore sort
 *     - don't do tabix index
 * @author Michael Meiners (m054457)
 *
 */
public class TjsonToCatalog {

	private static final Logger sLogger = LoggerFactory.getLogger(TjsonToCatalog.class);

	protected String[] mUserChromList = new String[0];
	protected String[] mHumanChromList = new String[0];
	
	private final JsonPath	mJsonPathChrom = JsonPath.compile ("_landmark");//, 	getJsonPredicate());
	private final JsonPath	mJsonPathMinBp = JsonPath.compile ("_minBP");
	private final JsonPath	mJsonPathMaxBp = JsonPath.compile ("_maxBP");
	private final JsonPath	mJsonPathRef   = JsonPath.compile ("_refAllele");
	private final JsonPath	mJsonPathAlts  = JsonPath.compile ("_altAlleles");

	/** Report every X lines (only set this externally within tests) */
	public static long REPORT_EVERY_X_LINES = 100000;
	/** Line to start at (only set this externally within tests) */
	public static long LINE_TO_START_AT = 0;
	
	public enum FileType { TEXT, GZIP, BGZIP };
	
	private Predicate getJsonPredicate() {
		return new Predicate() {
			public boolean apply(PredicateContext ctx) {
				return true;
			}
		};
	}

	/** Given an input file with the JSON column that is the focus of the new catalog,
	 *  build the new catalog, bgzip it, and put add a tabix index to it 
	 * @param inputFile   The input file to read that contains the JSON object from which to build the catalog
	 * @param bgzipCatalogOutFile  The full name (ending in ".tsv.bgz") of the resulting catalog
	 * @param isSort      True if the catalog is to be sorted, or false if none of the lines should be sorted
	 * @param chromSortOrderFilePath  A user-specified file that outlines how the chromosomes should be sorted
	 * @param jsoncol     The 0-based index of the column where the JSON is located
	 * @param isJsonOnly  True if the catalog should contain only the JSON column, and not the three tabix columns for chrom,minbp,maxbp
	 * @param tempDirectory  A temporary directory to work in while building the catalog.
	 *                       Or null to use the default Java temp directory.
	 *                       This should have enough space for 2-3x the size of the original input file. 
	 * @param buildAssembly 
	 * @param isModifyChrom  True if the chromosome should be automatically modified to normalize to human chromosomes (Ex: "chr1" to "1"; "MT" to "M")
	 * @param isInputFileAlreadyInFinalCatalogFormat  If the input file is already in final format and we do not have to sort, 
	 * 						 then just copy that file verbatim to the correct location.
	 * 						 This can happen if the make_json step produces a bgzip file with the correct tabix and JSON columns
	 * @throws IOException 
	 * @throws InterruptedException */
	public void createCatalog( String inputFile, String bgzipCatalogOutFile, boolean isSort, 
			String chromSortOrderFilePath, int jsonCol, boolean isJsonOnly, String tempDirectory, 
			HumanBuildAssembly buildAssembly, boolean isModifyChrom,
			boolean isInputFileAlreadyInFinalCatalogFormatAndBgzipped) throws IOException, InterruptedException
	{
        
        // If sorting is required, then:
		//   1) Copy input to a plain-text file
        //   2) Sort the file based on default chromosome order (unless user provided a file in which case we sort by that order)
        //   3) Bgzip the temp file to the bgzipCatalogOutFile (and add tabix index IF it is not to be a JSON-only catalog)
		// Else:
		//   1) Write directly to a bgzip file (and create a tabix index IF the file is NOT to be a JSON-only catalog)
		sLogger.info("Create catalog"
				+ "\n    inputFile: " + inputFile
				+ "\n    bgzipCatalogOutFile: " + bgzipCatalogOutFile
				+ "\n    chromSortOrderFilePath: " + chromSortOrderFilePath
				+ "\n    isSort: " + isSort
				+ "\n    jsonCol: " + jsonCol
				+ "\n    isJsonOnly: " + isJsonOnly
				+ "\n    tempDirectory: " + tempDirectory
				+ "\n    buildAssembly: " + buildAssembly
				+ "\n    isModifyChrom: " + isModifyChrom
				+ "\n    isInputFileAlreadyInFinalCatalogFormatAndBgzipped:  " + isInputFileAlreadyInFinalCatalogFormatAndBgzipped
				);
        if( isSort ) {
        	sLogger.info("Sort file: write to a temp file first, split that into multiple files which are then sorted and merged back into one, then bgzip the result and add tabix index if it is not just a JSON-only catalog");
            // Create the temp file in case we have to use it for sorting
        	File tempDir  = tempDirectory == null  ?  null  :  new File(tempDirectory);
    		File tempFile = File.createTempFile("myTempFileToSort", ".tmp", tempDir);
    		tempFile.deleteOnExit();

    		// Convert to catalog format and write to a plain-text file
    		writeCatalogLines_jsonParsing(inputFile, tempFile.getCanonicalPath(), /*isBgzipOut=*/false, jsonCol, isJsonOnly, isModifyChrom);

            // Sort by chrom,min,max
            sortByChrom(tempFile, chromSortOrderFilePath, buildAssembly);
            
            // Now write to bgzip and apply tabix index as necessary
            writeBgzipFile(tempFile.getCanonicalPath(), bgzipCatalogOutFile);
        } else {
        	// NO SORT
        	// Write the input file directly to a bgzip file
        	sLogger.info("Write directly to the bgzip catalog file and apply tabix index after (if not a JSON_only catalog)");
        	sLogger.info("No sort will be performed on the catalog.");
        	if( isInputFileAlreadyInFinalCatalogFormatAndBgzipped ) {
        		sLogger.info("The output from make_json is already BGZipped and in the correct format, so just copy the file to the final output location");
        		FileUtils.copyFile(new File(inputFile), new File(bgzipCatalogOutFile));
        	} else {
        		sLogger.info("Extract the first 3 tabix columns (if needed), and format the catalog lines, then write to bgzip output");
        		writeCatalogLines_jsonParsing(inputFile, bgzipCatalogOutFile, /*isBgzipOut=*/true, jsonCol, isJsonOnly, isModifyChrom);
        	}
        }
        
		boolean isCreateTabixIndex = ! isJsonOnly;
        sLogger.info("isCreateTabixIndex: " + isCreateTabixIndex);
        if( isCreateTabixIndex ) {
    		TabixCmd.createTabixIndex(bgzipCatalogOutFile);
        }
        
        sLogger.info("Done building catalog");
	}




	/** Given an input file with a JSON field, convert this JSON field to a catalog in the correct format (could be single column JSON), then write to either bgzip or plain-text file
	 *  When writing catalog:
	 * 		1) Need to sort:
	 * 			- read from bgz or plain-text, convert to catalog format, write to plain-text
	 * 			- Sort plain-text
	 * 			- read plain-text and write bgzip and tabix index it
	 * 		2) No sort:
	 * 			- read from bgz or plain-text, convert to catalog format, write to bgzip and tabix index it 
	 * @param inputFile
	 * @param outputFile
	 * @param isBgzipOut
	 * @param isCreateTabixIndex
	 * @param jsonCol
	 * @param isJsonOnly
	 * @param isModifyChrom
	 */
	protected void writeCatalogLines_jsonParsing(String inputFile, String outputFile,  boolean isBgzipOut,
		int jsonCol, boolean isJsonOnly, boolean isModifyChrom)
	{
		sLogger.info("Writing catalog format to a temp file (JSON PARSING): " + outputFile);

		boolean isGzip = inputFile.endsWith(".gz") || inputFile.endsWith(".bgz")  ||  isFileGzip(new File(inputFile));
		String inputFileType = isGzip  ?  "gzip"  :  "text";
        
        Pipe outputWriter = isBgzipOut
        		?  new BgzipWriter(outputFile)   // History -> (output)
        		:  new Pipeline(				 // History -> (output)
        				new MergePipe("\t"),														 // History -> String
        				new WritePipe(outputFile, /*isAppendToFile=*/false, /*isAddNewlines=*/true)  // String -> (output)
        				);
        	
        Pipeline pipeline = new Pipeline(
        		new CatAnythingPipe(inputFileType),							// (input) -> String
        		new String2HistoryPipe("\t"),								// String  -> History
        		new TjsonToCatalogPipe(jsonCol, isJsonOnly, isModifyChrom),	// History -> History
        		outputWriter												// History -> (output)
        		);
        
        pipeline.setStarts(Arrays.asList(inputFile));
        
        long numLinesWritten = LINE_TO_START_AT;
        while( pipeline.hasNext() ) {
        	pipeline.next();
            numLinesWritten++;
        	if( numLinesWritten % REPORT_EVERY_X_LINES == 0 )
        		sLogger.info("Num lines written to catalog format: " + numLinesWritten);
        }
	}
	
	protected void writeCatalogLines_textParsingOnly(String inputFile, String outputFile,  boolean isBgzipOut,  int jsonCol, boolean isJsonOnly, boolean isModifyChrom) {
		sLogger.info("Writing catalog format to a temp file (NO JSON PARSING): " + outputFile);

		boolean isGzip = inputFile.endsWith(".gz") || inputFile.endsWith(".bgz")  ||  isFileGzip(new File(inputFile));
		String inputFileType = isGzip  ?  "gzip"  :  "text";
        
        Pipe outputWriter = isBgzipOut
        		?  new BgzipWriterString(outputFile)   // String -> (output)
        		:  new Pipeline(				 	   // String -> (output)
        				new WritePipe(outputFile, /*isAppendToFile=*/false, /*isAddNewlines=*/true)  // String -> (output)
        				);
        		
        Pipeline pipeline = new Pipeline(
        		new CatAnythingPipe(inputFileType),											// (input) -> String
        		new TjsonToCatalogNoJsonParsingPipe(jsonCol, isJsonOnly, isModifyChrom),	// String  -> String
        		outputWriter																// String  -> (output)
        		);
        
        pipeline.setStarts(Arrays.asList(inputFile));
        
        long numLinesWritten = LINE_TO_START_AT;
        while( pipeline.hasNext() ) {
        	pipeline.next();
            numLinesWritten++;
        	if( numLinesWritten % REPORT_EVERY_X_LINES == 0 )
        		sLogger.info("Num lines written to catalog format: " + numLinesWritten);
        }
	}

	/** Gets the file type
	  * See: http://stackoverflow.com/questions/30507653/how-to-check-whether-file-is-gzip-or-not-in-java  */

	public static FileType getFileType(File file) {
		if( isFileGzip(file) ) {
			if( isFileBgzip(file) ) {
				return FileType.BGZIP;
			} else {
				return FileType.GZIP;
			}
		} else {
			return FileType.TEXT;
		}	
	}
	
	public static  boolean isFileBgzip(File inputFile) {
		// BGzip is a subset of Gzip, so if it's not GZ then it cannot be BGZ 
		if( ! isFileGzip(inputFile) )
			return false;
		
		try {
			BlockCompressedInputStream zipInput = new BlockCompressedInputStream(inputFile);
			zipInput.readLine();
			return true;
		} catch(Exception e) {
			return false;
		}
	}

	/** Checks if a file is gzipped.
	 *  NOTE:  A Bgzip file is ALSO a gzip file!
	  * From: http://stackoverflow.com/questions/30507653/how-to-check-whether-file-is-gzip-or-not-in-java  */
	public static boolean isFileGzip(File f) {
		int magic = 0;
		try {
			RandomAccessFile raf = new RandomAccessFile(f, "r");
			magic = raf.read() & 0xff | ((raf.read() << 8) & 0xff00);
			raf.close();
		} catch (Throwable e) {
			e.printStackTrace(System.err);
		}
		return magic == GZIPInputStream.GZIP_MAGIC;
	}

	public static BufferedReader getBufferedReader(File file) throws IOException {
		BufferedReader fin = null;
		FileType type = getFileType(file);
		if( FileType.BGZIP.equals(type) ) {
			fin = new BufferedReader(new InputStreamReader(new BlockCompressedInputStream(file)));
		} else if( FileType.GZIP.equals(type)) {
			fin = new BufferedReader(new InputStreamReader(new GZIPInputStream(new FileInputStream(file))));
		} else { // FileType.TEXT.equals(
			fin = new BufferedReader(new FileReader(file));
		}
		return fin;
	}




	protected void writeBgzipFile(String inputFile, String bgzipOutputFile) {
		sLogger.info("Write input file to a bgzip catalog");
        String inputFileType = (inputFile.endsWith(".gz") || inputFile.endsWith(".bgz"))  ?  "gzip"  :  "text";

        Pipeline pipeline = new Pipeline(
        		new CatAnythingPipe(inputFileType),				// (input) -> String
        		new String2HistoryPipe("\t"),					// String  -> History
        		new BgzipWriter(bgzipOutputFile)    // History -> (output)
        		);
        
        pipeline.setStarts(Arrays.asList(inputFile));
        
        while( pipeline.hasNext() )
        	pipeline.next();
	}
	
	
	
	/** Sort the tempInputFile by chromosome, minBP, maxBP, ref, alt.  The chromosome sort order can be defined by
	 *  passing in an optional chromSortOrderTextFile that has all chromosomes that occur in the tempInputFile
	 *  in the order that they should be sorted (one chromosome per line).  For example:<br>
	 *   1 <br>
	 *   9 <br>
	 *   10 <br>
	 *   12 <br>
	 *   23 <br.
	 *   1_random <br>
	 *  If chromSortOrderTextFile is null, then the sort order is by numeric value, then by numeric order 
	 * @param tempInputFile
	 * @param chromSortOrderTextFile
	 * @param buildAssembly 
	 * @throws IOException 
	 */
	public void sortByChrom(File tempInputFile, String chromSortOrderTextFile, HumanBuildAssembly buildAssembly) throws IOException {
		sLogger.info("Load user and human chromosome files");
		loadUserChromList(chromSortOrderTextFile);
		loadHumanChromList(buildAssembly);
		Comparator<String> lineComparator = getLineComparator();
		File tempDir = tempInputFile.getParentFile();

		// Split the file up into sorted partial files
		// NOTE: Make sure to use gzip temporary files!!!
		sLogger.info("Break the input file up into multiples and sort, then merge them");
		sLogger.info("Input file: " + tempInputFile.getCanonicalPath() + "   " + tempInputFile.length() + " bytes");
		boolean isDistinct_pruneDuplicateLines = false;
		boolean isUseGzipForTempFiles = true;
		int numHeaderLines = 0;
		List<File> sortedPartialFiles = ExternalSort.sortInBatch(
				tempInputFile,
				lineComparator, 
				ExternalSort.DEFAULTMAXTEMPFILES,
				Charset.defaultCharset(),
				tempDir,
				isDistinct_pruneDuplicateLines,
				numHeaderLines,
				isUseGzipForTempFiles
				);
		sLogger.info("Sorted temp files: " + sortedPartialFiles.size());
		for(File f : sortedPartialFiles) {
			sLogger.info("  " + f.getCanonicalPath() + "   " + f.length() + " bytes");
		}
		deleteAllFilesOnExit(sortedPartialFiles);
		
		// Merge the partial files back into the tempInputFile
		sLogger.info("Merge all sorted files back into one");
		ExternalSort.mergeSortedFiles(
				sortedPartialFiles,
				tempInputFile,
				lineComparator,
				Charset.defaultCharset(),
				isDistinct_pruneDuplicateLines,
				/*isAppend=*/false,
				isUseGzipForTempFiles
				);
		sLogger.info("Merged file: " + tempInputFile.getCanonicalPath() + "   " + tempInputFile.length() + " bytes");
	}


	private void deleteAllFilesOnExit(List<File> filesToDelete) {
		for(File f : filesToDelete) {
			f.deleteOnExit();
		}
	}



	/** Compare the lines by chrom, minBP, maxBP, ref, alt */
	protected int compareLines(String line1, String line2) {
		try {
			String[] colsLine1 = line1.split("\t");
			String[] colsLine2 = line2.split("\t");
			
			// If either is a header line, compare their header scores
			if( line1.startsWith("#")  ||  line2.startsWith("#") ) {
				//return Integer.compare(getHeaderScore(line1), getHeaderScore(line2));
				Integer iLine1 = new Integer(getHeaderScore(line1));
				Integer iLine2 = new Integer(getHeaderScore(line2));	
				return iLine1.compareTo(iLine2);
			}
			
			int jsonCol = getJsonCol(colsLine1);
			
			// If JSON column not found, then just compare by whole line
			if( jsonCol == -1 )
				return line1.compareToIgnoreCase(line2);
			
			String json1 = colsLine1[jsonCol];
			String json2 = colsLine2[jsonCol];
			
			// Else compare by chromosome score
			String chr1Lower = getChr(json1);
			String chr2Lower = getChr(json2);
			
			// Compare first by score
			int score1 = getChromScore(chr1Lower);
			int score2 = getChromScore(chr2Lower);
			// If score is different, return the difference
			if( score1 != score2 )
				return score1 - score2;
			
			// Else, the scores were the same, so string-compare by chromosome alone (if strings still different)
			int chrStrScore = chr1Lower.compareToIgnoreCase(chr2Lower);
			if( chrStrScore != 0 )
				return chrStrScore;
			
			// Else, compare by minBP
			Integer minBP1 = getMinBp(json1);
			Integer minBP2 = getMinBp(json2);
			int minBpScore = minBP1.compareTo(minBP2);
			if( minBpScore != 0 )
				return minBpScore;

			// Else, compare by maxBP
			// NOTE: This may cause a ref of "C" to come before "AAA"
			Integer maxBP1 = getMaxBp(json1);
			Integer maxBP2 = getMaxBp(json2);
			int maxBpScore = maxBP1.compareTo(maxBP2);
			if( maxBpScore != 0 )
				return maxBpScore;

			// Else compare by ref allele
			String ref1 = getRef(json1);
			String ref2 = getRef(json2);
			int refScore = ref1.compareTo(ref2);
			if( refScore != 0 )
				return refScore;
			
			// Else compare by alt alleles
			String alts1 = altsToString(json1);
			String alts2 = altsToString(json2);
			int altsScore = alts1.compareTo(alts2);
			if( altsScore != 0 )
				return altsScore;
			
			// Else just do a string compare on the entire line
			return line1.compareToIgnoreCase(line2);
		} catch(InvalidPathException e) {
			// There was probably a problem finding one of the JSON fields (chr,min,max,ref,alt), so just compare the lines whole
			return line1.compareTo(line2);
		} catch(Exception e) {
			System.err.println("Error comparing two lines:");
			System.err.println("  1) " + line1);
			System.err.println("  2) " + line2);
			return line1.compareTo(line2);
		}
	}
	
	// Get chrom from JSON (or dot if not found)
	private String getChr(String json) {
		try {
			return ((String)mJsonPathChrom.read(json)).toLowerCase();
		} catch(Exception e) {
			return ".";
		}
	}
	
	// Get minBP from JSON (or -1 if not found) 
	private Integer getMinBp(String json) {
		try {
			return (Integer)(mJsonPathMinBp.read(json));
		} catch(Exception e) {
			return -1;
		}
	}
	
	// Get maxBP from JSON (or -1 if not found) 
	private Integer getMaxBp(String json) {
		try {
			return (Integer)(mJsonPathMaxBp.read(json));
		} catch(Exception e) {
			return -1;
		}
	}
	
	// Get ref from JSON (or dot if not found) 
	private String getRef(String json) {
		try {
			return 	((String)mJsonPathRef.read(json)).toUpperCase();
		} catch(Exception e) {
			return ".";
		}
	}
	
	/** Convert the _altAlleles array to a comma-separated string, or dot if not found */
	private String altsToString(String json) {
		try {
			JSONArray altsArray = mJsonPathAlts.read(json);
			StringBuilder str = new StringBuilder();
			for(int i=0; i < altsArray.size(); i++) {
				if( i > 0 )
					str.append(",");
				str.append(altsArray.get(i));
			}
			return str.toString().toUpperCase();
		} catch(Exception e) {
			return ".";
		}
	}




	/** Returns either the typical or alternate column num (0-based) for a catalog depending on where the JSON is found.
	 *  Or returns -1 if JSON is not found in either expected column	 */
	private int getJsonCol(String[] ctgRow) {
		final int TYPICAL_JSON_COL = 3;
		final int ALT_JSON_COL = 0;
		if( ctgRow.length >= 4  &&  ctgRow[TYPICAL_JSON_COL].startsWith("{")  &&  ctgRow[TYPICAL_JSON_COL].endsWith("}") )
			return TYPICAL_JSON_COL;
		else if( ctgRow.length >= 1  &&  ctgRow[ALT_JSON_COL].startsWith("{")  &&  ctgRow[ALT_JSON_COL].endsWith("}") )
			return ALT_JSON_COL;
		else
			return -1;
	}




	protected int getHeaderScore(String line) {
		// ## metadata comes first
		if( line.startsWith("##") )
			return 0;
		// Then # column header line
		else if( line.startsWith("#") )
			return 1;
		else
			return 2;
	}
	
	
	// TODO: 
	// This could be one method of comparison (assign a score, with lowest score before highest)
	// 1) If both chromosomes are in the user-specified list, use that
	// 2) Else if both chromosomes are in the human list, then use that
	// 3) Else, try to figure out the order based on:
	//    a) ##
	//	  b) #
	//	  c) Integer values if chromosome is integer alone
	//    d) String comparison (NOT case sensitive) if alphanumeric only
	//    e) Integer values if first portion is integer
	//    f) String comparison (NOT case sensitive)
	// NOTE: For those that return the same score, they should then be string-compared
	//-----------------------------------
	// The order of human chromosomes is: (from David Rider):
	//  	regular chromosomes
	//		random chromosomes (in regular chromosome order and then alphabetical)
	//		Unknown chromosomes (in alphabetical)
	//		Alts (in regular chromosome order and then alphabetical)
	//			Alts are last because they aren’t used in the GRCh38 workflows (but others are) 
	//			so I followed the same convention for sorting GRCh37 chromosomes.
	//-----------------------------------
	protected int getChromScore(String chromIn) {
		// Strip "chr" off front of chromosome if given
		String chrom = GenomicObjectUtils.computechr(chromIn);
		
		String chromLower = chrom.toLowerCase();
		
		int score = 0;
		
		// Chromosomes in the user list start at 0 (lowest score)
		if( isInList(chrom, mUserChromList) ) {
			score = idxInList(chrom, mUserChromList);
		}
		// Chromosomes in human list start at 1000 (mid score)
		else if( isInList(chrom, mHumanChromList) ) {
			score = 1000 + idxInList(chrom, mHumanChromList);
		}
		// Chromosomes not in any list, but numeric only are next (high score), start at 2000
		else if( isInteger(chrom) ) {
			score = 2000 + Integer.parseInt(chrom);
		}
		// Chromosomes that are alphanumeric only start at 3000 (higher score), and that are not the special chromosome "UNKNOWN"
		else if( StringUtils.isAlphanumeric(chrom)  &&  ! chromLower.equals("unknown") ) {
			score = 3000;
		}
		// Chromosomes that contain "random" are next (score of 4000-6000), with those that do NOT start with integers getting a score of 5000)
		else if( chromLower.contains("random") ) {
			if( isStartsWithInteger(chrom) ) {
				score = 4000 + getIntegerPrefix(chrom);
			} else {
				score = 5000;
			}
		}
		// Chromosomes that contain "un" or "unknown" are next (score of 6000), (but that are not exactly the string "UNKNOWN", which is reserved for last place)
		else if( ! chromLower.equals("unknown")  &&  (chromLower.contains("unknown") || chromLower.startsWith("un_")) ) {
			score = 6000;
		}
		// Else, they are alt chrosomes and should be sorted first by those with integer prefixes (start at 7000), then alphabetically (score of 8000)
		else {
			if( isStartsWithInteger(chrom) ) {
				score = 7000 + getIntegerPrefix(chrom);
			} else if( ! chromLower.equals("unknown") ) {
				score = 8000;
			} else {  // IS "UNKNOWN"
				score = 9000;
			}
		}
			
		return score;
		
	}
	
	protected boolean isStartsWithInteger(String s) {
		Integer i = getIntegerPrefix(s);
		return i != null;
	}
	
	protected Integer getIntegerPrefix(String s) {
		StringBuilder str = new StringBuilder();
		for(int i=0; i < s.length(); i++) {
			if( Character.isDigit(s.charAt(i)) )
				str.append(s.charAt(i));
			else
				break; // exit for-loop
		}
		
		if( str.length() == 0 )
			return null;
		else
			return Integer.parseInt(str.toString());
	}

	protected boolean isInteger(String chrom) {
		try {
			Integer.parseInt(chrom);
			return true;
		} catch(Exception e) {
			return false;
		}
	}

	protected int idxInList(String chrom, String[] list) {
		for(int i=0; i < list.length; i++) {
			String chromInList = GenomicObjectUtils.computechr(list[i]);
			if( chrom.equalsIgnoreCase(chromInList) )
				return i;
		}
		// Not found
		return -1;
	}

	protected boolean isInList(String chrom, String[] list) {
		return  idxInList(chrom, list) != -1;
	}
	
	protected String[] loadUserChromList(String chromosomeListFile) throws IOException {
		if( chromosomeListFile != null ) {
			File chromListFile = new File(chromosomeListFile);
			if( chromListFile.exists() )
				mUserChromList = FileUtils.readLines(chromListFile).toArray(new String[0]);
		}
		return mUserChromList;
	}

	protected String[] loadHumanChromList(HumanBuildAssembly buildAssembly) throws IOException {
		// Default is GRCh37
		String fileRelativePath = "humanChromosomesSortedByName.GRCh37.txt";
		if( buildAssembly != null && buildAssembly.equals(HumanBuildAssembly.GRCh38) )
			fileRelativePath = "humanChromosomesSortedByName.GRCh38.txt";
		
		File file = new File(this.getClass().getClassLoader().getResource(fileRelativePath).getPath());
		mHumanChromList = FileUtils.readLines(file).toArray(new String[0]);
		return mHumanChromList;
	}

	
	protected Comparator<String> getLineComparator() {
		return new Comparator<String>() {
			public int compare(String line1, String line2) {
				return compareLines(line1, line2);
			}			
		};
	}
	
}
