package edu.mayo.bior.cli.cmd;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.jayway.jsonpath.JsonPath;
import edu.mayo.bior.catalog.latest.LatestCatalog;
import edu.mayo.bior.catalog.latest.LatestCatalogFinder;
import edu.mayo.cli.CommandPlugin;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import static edu.mayo.bior.cli.cmd.CommandUtil.DirectoryAttributes.*;
import static edu.mayo.bior.cli.cmd.CommandUtil.handleDirectory;
import static java.util.Arrays.asList;


/**
 * Searches for the latest version of a Catalog
 */
public class CatalogLatestCommand implements CommandPlugin {

    private static final Logger LOGGER = LoggerFactory.getLogger(CatalogLatestCommand.class);

    private static final String TAB = "\t";

    private static final String OPTION_ROOT_DIR = "r";
    private static final String OPTION_BUILD    = "B";
    private static final String OPTION_SOURCE   = "s";
    private static final String OPTION_CONTAINS = "c";
    private static final String OPTION_PATH     = "p";

    private static LatestCatalogFinder mockFinder;

    @Override
    public void init(Properties properties) throws Exception { }

    @Override
    public void execute(CommandLine cl, Options options) throws Exception {

        File rootDir  = handleDirectory(cl, OPTION_ROOT_DIR, asList(EXISTS, READABLE, EXECUTEABLE));
        LOGGER.info(String.format("Root dir: %s", rootDir.getAbsolutePath()));

        String build  = cl.getOptionValue(OPTION_BUILD);
        LOGGER.info(String.format("Build: %s", build));

        String source = cl.getOptionValue(OPTION_SOURCE);
        LOGGER.info(String.format("Source: %s", source));

        String contains = null;
        if (cl.hasOption(OPTION_CONTAINS)) {
            contains = cl.getOptionValue(OPTION_CONTAINS);
            LOGGER.info(String.format("Contains: %s", contains));
        }

        List<JsonPath> jsonPaths = new ArrayList<JsonPath>();
        if (cl.hasOption(OPTION_PATH)) {
            for (String path: cl.getOptionValues(OPTION_PATH)) {
                LOGGER.info(String.format("JSON Path: %s", path));
                jsonPaths.add(JsonPath.compile(path));
            }
        }

        LatestCatalogFinder finder = mockFinder != null ? mockFinder : new LatestCatalogFinder();

        LatestCatalog latestCatalog = finder.findLatestCatalog(rootDir, source, build, contains);

        if (jsonPaths.size() == 0) {
            printJsonToConsole(latestCatalog);
        } else {
            printExtractedFieldsToConsole(latestCatalog, jsonPaths);
        }
    }

    /**
     * Prints the catalog information to STDOUT in JSON format.
     * @param latestCatalog Contains catalog information to print as JSON
     */
    private void printJsonToConsole(LatestCatalog latestCatalog) {
        Gson gson = new GsonBuilder().setPrettyPrinting().create();
        gson.toJson(latestCatalog, System.out);
    }

    /**
     * Prints the extracted catalog information to STDOUT in TSV format.
     * @param latestCatalog Contains catalog metadata
     * @param jsonPaths JSON paths that determines what metadata fields to extract.  If a JSON path returns nothing
     */
    private void printExtractedFieldsToConsole(LatestCatalog latestCatalog, List<JsonPath> jsonPaths) {
        Gson gson = new GsonBuilder().create();
        String json = gson.toJson(latestCatalog);

        StringBuilder output = new StringBuilder();
        for (JsonPath jsonPath: jsonPaths) {
            String value = jsonPath.read(json).toString();
            output.append(value);
            output.append(TAB);
        }
        output.deleteCharAt(output.length() - 1); // chomp trailing tab
        System.out.println(output);
    }

    /**
     * Used for unit testing with mock business logic class
     */
    static void setLatestCatalogFinder(LatestCatalogFinder latestCatalogFinder) {
        mockFinder = latestCatalogFinder;
    }
}
