package edu.mayo.bior.catalog.markdown.transformer.comparison;

import edu.mayo.bior.catalog.markdown.transformer.MarkdownTransformer;
import edu.mayo.bior.catalog.markdown.transformer.TransformException;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.*;

import static edu.mayo.bior.catalog.markdown.MarkdownUtils.chompTrailingNewlines;
import static edu.mayo.bior.catalog.markdown.MarkdownUtils.escapeChrs;
import static edu.mayo.bior.catalog.markdown.transformer.comparison.ComparisonStatsImpl.NOT_AVAILABLE_MESG;
import static java.lang.String.format;
import static org.apache.commons.io.IOUtils.closeQuietly;

/**
 * Builds a markdown table about catalog content changes at the column level.
 */
public class CatalogContentChangesImpl implements MarkdownTransformer {

    static final String TEMPLATE_KEYWORD = "COMPARISON_STATISTICS_CONTENT_CHANGES_TABLE";

    private CatalogDiffStatistics stats;

    CatalogContentChangesImpl(CatalogDiffStatistics stats) {
        this.stats = stats;
    }

    @Override
    public String transform(String markdown) throws TransformException {
        if (stats == null) {
            return markdown.replace(TEMPLATE_KEYWORD, NOT_AVAILABLE_MESG);
        } else {

            return markdown.replace(TEMPLATE_KEYWORD, chompTrailingNewlines(buildTable()));
        }
    }

    private String buildTable() {
        StringWriter sWtr = new StringWriter();
        PrintWriter pWtr = new PrintWriter(sWtr);

        // index by column name
        Map<String, CatalogDiffStatistics.ColumnStats> columnStatsMap = indexByColumnName(stats.getColumnStats());

        // column names sorted alphabetically
        List<String> columnNames = new ArrayList<String>();
        columnNames.addAll(columnStatsMap.keySet());
        Collections.sort(columnNames);

        pWtr.println("ColumnName | Removed | Added | Changed | Datatype | Unchanged");
        pWtr.println("--- | --- | --- | --- | --- | ---");
        for (String columnName: columnNames) {
            CatalogDiffStatistics.ColumnStats columnStats = columnStatsMap.get(columnName);
            final float TOTAL =
                    columnStats.getRemoved() +
                            columnStats.getAdded() +
                            columnStats.getValueChanged() +
                            columnStats.getDatatypeChanged() +
                            columnStats.getValueUnchanged();

            pWtr.println(
                    format("%s | %d (%.3f%%) | %d (%.3f%%) | %d (%.3f%%) | %d (%.3f%%) | %d (%.3f%%)",
                            escapeChrs(columnName),
                            columnStats.getRemoved(),
                            columnStats.getRemoved() / TOTAL * 100,
                            columnStats.getAdded(),
                            columnStats.getAdded() / TOTAL * 100,
                            columnStats.getValueChanged(),
                            columnStats.getValueChanged() / TOTAL * 100,
                            columnStats.getDatatypeChanged(),
                            columnStats.getDatatypeChanged() / TOTAL * 100,
                            columnStats.getValueUnchanged(),
                            columnStats.getValueUnchanged() / TOTAL * 100
                    )
            );
        }

        final float TOTAL =
                stats.getSummaryStats().getCatalogColumnRemoved() +
                        stats.getSummaryStats().getCatalogColumnAdded() +
                        stats.getSummaryStats().getCatalogColumnValueChanged() +
                        stats.getSummaryStats().getCatalogColumnDataTypeChange() +
                        stats.getSummaryStats().getCatalogColumnValueUnchanged();
        pWtr.println(
                format("%s | %d (%.3f%%) | %d (%.3f%%) | %d (%.3f%%) | %d (%.3f%%) | %d (%.3f%%)",
                        "TOTAL",
                        stats.getSummaryStats().getCatalogColumnRemoved(),
                        stats.getSummaryStats().getCatalogColumnRemoved() / TOTAL * 100,
                        stats.getSummaryStats().getCatalogColumnAdded(),
                        stats.getSummaryStats().getCatalogColumnAdded() / TOTAL * 100,
                        stats.getSummaryStats().getCatalogColumnValueChanged(),
                        stats.getSummaryStats().getCatalogColumnValueChanged() / TOTAL * 100,
                        stats.getSummaryStats().getCatalogColumnDataTypeChange(),
                        stats.getSummaryStats().getCatalogColumnDataTypeChange() / TOTAL * 100,
                        stats.getSummaryStats().getCatalogColumnValueUnchanged(),
                        stats.getSummaryStats().getCatalogColumnValueUnchanged() / TOTAL * 100
                )
        );

        closeQuietly(pWtr);
        closeQuietly(sWtr);
        return sWtr.toString();
    }

    /**
     * Helper method to index {@link CatalogDiffStatistics.ColumnStats} objects by column name.
     * @param arr Array of objects to be indexed by column name.
     * @return {@link Map} where the key is the column name, value is the {@link  CatalogDiffStatistics.ColumnStats}
     */
    private Map<String, CatalogDiffStatistics.ColumnStats> indexByColumnName(CatalogDiffStatistics.ColumnStats[] arr) {
        Map<String, CatalogDiffStatistics.ColumnStats> map = new HashMap<String, CatalogDiffStatistics.ColumnStats>();
        for (CatalogDiffStatistics.ColumnStats columnStats: arr) {
            map.put(columnStats.getName(), columnStats);
        }
        return map;
    }
}