package edu.mayo.bior.catalog.list;

import edu.mayo.bior.catalog.CatalogFormatException;
import org.apache.commons.lang.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;


/**
 * Contains information about a catalog, including data that comes from its datasource.properties file
 */
public class CatalogInfo
{

   private static final String TAB = "\t";
   public static final String DOT = ".";

   /**
    * The status of the catalog - whether it should be used, or whether it has been deprecated in favor of newer or corrected one
    */
   public enum CatalogStatus
   {
      Active, Deprecated
   }

   /**
    * Catalog status.  Ex: Active
    */
   public CatalogStatus status;

   /**
    * Catalog path.    Ex: /research/bsi/data/catalogs/bior/v1/dbSNP/137/00-All_GRCh37.tsv.bgz
    */
   public String path;

   /**
    * Date last updated Ex:  2015-11-16 09:39:54
    */
   public String lastUpdated;

   /**
    * Short unique name for the catalog (at this particular version and build)  Ex: dbSNP_142_GRCh37p13
    */
   public String shortName;

   /**
    * Indexes for particular keys in the JSON portion of the catalog.  Ex: GENE, ID, RSID
    */
   public List<String> indexes;

   /**
    * The data source "common name".   Ex: dbSNP
    */
   public String dataSource;

   /**
    * The data source version.  Ex: 142
    */
   public String version;

   /**
    * The build (with any point releases).  Ex: GRCh37.p13
    */
   public String build;

   /**
    * The base build (no point releases).   Ex: GRCh37
    */
   public String buildBase;

   /**
    * The dataset type.  Ex: Variants
    */
   public String dataset;

   public CatalogInfo()
   {

   }

   public CatalogInfo(String tabSeparatedLine) throws CatalogFormatException
   {
      if (tabSeparatedLine == null)
      {
         throw new RuntimeException("Must supply line to CatalogInfo constructor");
      }
      String[] fields = tabSeparatedLine.split(TAB);
      if (fields.length != 10)
      {
         String numFieldsDescription = "fields";
         if (fields.length == 1)
         {
            numFieldsDescription = "field";
         }
         String msg = String.format("Expecting lines with 10 tab separated fields but got %d %s from '%s',",
            fields.length, numFieldsDescription, tabSeparatedLine);
         throw new CatalogFormatException(msg);
      }

      String statusStr = fields[0];
      if (statusStr.toLowerCase().equals(CatalogStatus.Active.toString().toLowerCase()))
      {
         status = CatalogStatus.Active;
      }
      else if (statusStr.toLowerCase().equals(CatalogStatus.Deprecated.toString().toLowerCase()))
      {
         status = CatalogStatus.Deprecated;
      }
      else
      {
         String msg = String.format("Expected '%s' or '%s' but got '%s' as first field (status) from '%s'",
            CatalogStatus.Active.toString(), CatalogStatus.Deprecated.toString(),
            statusStr, tabSeparatedLine);
         throw new CatalogFormatException(msg);
      }

      path = getStr(fields[1]);
      lastUpdated = getStr(fields[2]);
      shortName = getStr(fields[3]);
      indexes = indexesFromStr(fields[4]);
      dataSource = getStr(fields[5]);
      version = getStr(fields[6]);
      build = getStr(fields[7]);
      buildBase = getStr(fields[8]);
      dataset = getStr(fields[9]);
   }

   public String toString()
   {
      List<String> data = new ArrayList<String>();
      data.add(status.toString());
      data.add(dotIfNullOrEmpty(path));
      data.add(dotIfNullOrEmpty(lastUpdated));
      data.add(dotIfNullOrEmpty(shortName));
      data.add(getIndexesAsStr());
      data.add(dotIfNullOrEmpty(dataSource));
      data.add(dotIfNullOrEmpty(version));
      data.add(dotIfNullOrEmpty(build));
      data.add(dotIfNullOrEmpty(buildBase));
      data.add(dotIfNullOrEmpty(dataset));
      return StringUtils.join(data, TAB);
   }

   public boolean isActive()
   {
      return status == CatalogStatus.Active;
   }

   public boolean isDeprecated()
   {
      return status == CatalogStatus.Deprecated;
   }

   public String getPath()
   {
      return path;
   }

   public String getLastUpdated()
   {
      return lastUpdated;
   }

   public String getShortName()
   {
      return shortName;
   }

   public List<String> getIndexes()
   {
      return indexes;
   }

   public int getIndexesSize()
   {
      return indexes.size();
   }

   public String getDataSource()
   {
      return dataSource;
   }

   public String getVersion()
   {
      return version;
   }

   public String getBuild()
   {
      return build;
   }

   public String getBuildBase()
   {
      return buildBase;
   }

   public String getDataset()
   {
      return dataset;
   }

   private List<String> indexesFromStr(String s)
   {
      String indexStr = getStr(s);
      if (indexStr != null)
      {
         String[] fields = indexStr.split(",");
         return Arrays.asList(fields);
      }
      return new ArrayList<String>();
   }

   private String getIndexesAsStr()
   {
      if (indexes.size() == 0)
         return ".";

      StringBuilder str = new StringBuilder();
      for (int i = 0; i < indexes.size(); i++)
      {
         if (i > 0)
            str.append(",");
         str.append(indexes.get(i));
      }
      return str.toString();
   }

   private String getStr(String s)
   {
      if (s == null || s.trim().equals(DOT))
      {
         return null;
      }
      return s.trim();
   }

   /**
    * If the string is null or "", then return a dot (".") as a placeholder.  Else return the original string
    */
   private String dotIfNullOrEmpty(String s)
   {
      if (s == null || s.trim().length() == 0)
         return DOT;
      return s;
   }

}
