package edu.mayo.bior.catalog.index;

import edu.mayo.pipes.util.index.H2Connection;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;

/** Return stats about an index
 * @author Michael Meiners (m054457)
 */
public class IndexStats {
	public static void usage() {
		System.out.println("Usage:");
		System.out.println("  IndexStats <catalogIndexH2Database>  [--all]");
		System.out.println("Example:");
		System.out.println("  IndexStats  /local2/tmp/mycatalog.gene.idx.h2.db  --all");
		System.out.println("Returns stats about a bior catalog index");
		System.out.println("  - Number of rows");
		System.out.println("  - Number of unique keys");
		System.out.println("  - Table names and metadata");
		System.out.println("  - Indexes on columns");
		System.out.println("Note: Specifying the --all flag will print all data in the index table.  If provided, this should be the second argument.");
	}

	
	public static void main(String[] args) {
		if(args.length == 0  ||  "-h".equals(args[0])) {
			usage();
			return;
		}
		String databasePath = args[0];
		long numRowsToPrint = 100;
		if( args.length > 1 && "--all".equals(args[1]) )
			numRowsToPrint = -1;
		new IndexStats().getStats(databasePath, numRowsToPrint);
	}

	public void getStats(String databasePath, long numRowsToPrint) {
		Connection conn = null;
		Statement stmt = null;
		try {
			conn = new H2Connection(databasePath, /*isWritable=*/false).getConn();
			printTableMetadata(conn, numRowsToPrint);
		} catch(Exception e) {
			e.printStackTrace();
		} finally {
			try {
				if( conn != null )
					conn.close();
				if( stmt != null )
					stmt.close();
			} catch(Exception e2) {
				e2.printStackTrace();
			}
		}
	}
	




	private static void printTableMetadata(Connection conn, long numRowsToPrint) {
		Statement stmt1 = null;
		Statement stmt2 = null;
		ResultSet rs = null;
		ResultSet rs2 = null;
		try {
			// Note: Must have 2 Statements because we have two ResultSets open at the same time!
			stmt1 = conn.createStatement();
			stmt2 = conn.createStatement();
			rs = stmt1.executeQuery("SELECT TABLE_NAME FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_TYPE = 'TABLE'");
			while(rs.next()) {
				String tableName = rs.getString("TABLE_NAME");
				System.out.println("Table: " + tableName);
				
				// Row count
				rs2 = stmt2.executeQuery("SELECT COUNT(*) FROM " + tableName);
				rs2.next();
				System.out.println("  Row count: " + rs2.getInt(1));
				rs2.close();
				
				// Unique keys
				if( hasCol(tableName, "Key", stmt2) ) {
					rs2 = stmt2.executeQuery("SELECT COUNT(DISTINCT Key) FROM " + tableName);
					rs2.next();
					System.out.println("  Num unique keys (in 'Key' column): " + rs2.getInt(1));
					rs2.close();
				}
					
				// Column metadata
				rs2 = stmt2.executeQuery("SELECT * FROM " + tableName);
				ResultSetMetaData meta = rs2.getMetaData();
				System.out.println("  Columns:");
				for(int i=1; i <= meta.getColumnCount(); i++) {
					System.out.println("    " + meta.getColumnName(i) + "  (" + meta.getColumnTypeName(i) + ")");
				}
				rs2.close();
				
				// Indexes   (ex result: TRUE; KEYINDEX; KEY; FALSE; FALSE)
				rs2 = stmt2.executeQuery("select INDEX_NAME, NON_UNIQUE, COLUMN_NAME, PRIMARY_KEY, IS_GENERATED from INFORMATION_SCHEMA.INDEXES where TABLE_NAME='" + tableName + "'");
				System.out.println("  Indexes:");
				while(rs2.next()) {
					System.out.println("    " + rs2.getString("INDEX_NAME") 
							+ " on column " + rs2.getString("COLUMN_NAME") 
							+ ",  isPrimaryKey: " + rs2.getBoolean("PRIMARY_KEY")
							+ ",  isUnique: " + ! rs2.getBoolean("NON_UNIQUE")
							+ ",  isAutoGenerated: " + rs2.getBoolean("IS_GENERATED"));
				}
				rs2.close();
				
				printRows(stmt2, tableName, numRowsToPrint);
			}
		} catch(Exception e) {
			e.printStackTrace();
		} finally {
			try {
				if(rs != null)
					rs.close();
				if(rs2 != null) 
					rs2.close();
				if(stmt1 != null)
					stmt1.close();
				if(stmt2 != null) 
					stmt2.close();
			}catch(Exception e) {
				e.printStackTrace();
			}
		}
	}


	

/**
	 * Print X number of rows from the database
	 * @param stmt2  A generic statement to the database
	 * @param tableName  Table name to print the data from
	 * @param numRowsToPrint  Number of rows to print, or negative or zero to print all rows
	 * @return
	 * @throws SQLException
	 */
	private static void printRows(Statement stmt2, String tableName, long numRowsToPrint) throws SQLException {
		ResultSet rs2;
		ResultSetMetaData meta;

		//-----------------------------------
		// Print the first X rows
		//-----------------------------------
		String limitRowsStr = "";
		if( numRowsToPrint > 0 ) {
			System.out.println("\n  First " + numRowsToPrint + " rows:");
			limitRowsStr = " LIMIT " + numRowsToPrint;
		}
		rs2 = stmt2.executeQuery("SELECT * FROM " + tableName + limitRowsStr);
		meta = rs2.getMetaData();
		// The ROW is the row count column, and is NOT part of the metadata from the table
		System.out.print("ROW  ");
		for(int i=1; i <= meta.getColumnCount(); i++) {
			System.out.print(padRight(meta.getColumnName(i), meta.getColumnDisplaySize(i))  + (i < meta.getColumnCount() ? "  " : "\n") );
		}
		// Divider line below header
		System.out.print("---- ");
		for(int i=1; i <= meta.getColumnCount(); i++) {
			System.out.print(copyChar('-', meta.getColumnDisplaySize(i))  + (i < meta.getColumnCount() ? "  " : "\n") );
		}
		// All data lines
		long row = 1;
		while(rs2.next() ) {
			// Stop if we have printed the number of requested rows
			//if( numRowsToPrint > 0  &&  row > numRowsToPrint ) {
			//	break;
			//}
			System.out.print(padRight(row + "", 5));
			for(int i=1; i <= meta.getColumnCount(); i++) {
				System.out.print(padRight("" + rs2.getObject(i), meta.getColumnDisplaySize(i)) + (i < meta.getColumnCount() ? "  " : "\n") );
			}
			row++;
		}
		rs2.close();
	}
	
	private static String padRight(String s, int len) {
		StringBuilder str = new StringBuilder(s);
		while(str.length() < len) {
			str.append(" ");
		}
		return str.toString();
	}
	
	private static String copyChar(char c, int len) {
		StringBuilder str = new StringBuilder();
		while(str.length() < len) {
			str.append(c);
		}
		return str.toString();
	}


	private static boolean hasCol(String tableName, String colName, Statement stmt) throws SQLException {
		boolean hasCol = false;
		ResultSet rs = stmt.executeQuery("SELECT * FROM " + tableName);
		for(int i=1; i <= rs.getMetaData().getColumnCount(); i++) {
			if(colName.equalsIgnoreCase(rs.getMetaData().getColumnName(i)))
				hasCol = true;
		}
		rs.close();
		return hasCol;
	}
	

}
