package edu.mayo.bior.catalog;

public class CatalogTabixEntry {

	public CatalogTabixEntry(String catalogLine) throws CatalogFormatException
	{
		if (catalogLine == null)
		{
			throw new CatalogFormatException("Null line given to CatalogTabixEntry constructor");
		}
		setLine(catalogLine);
		String[] fields = catalogLine.split("\t");
		if (fields.length == 4)
		{
			is4FieldEntry = true;
			setJsonString(fields[3]);
			if (!looksLikeJson(getJsonString()))
			{
				throw new CatalogFormatException(
					String.format("A simple check of the 4th field (%s) from '%s' doesn't look like JSON",
					              getJsonString(), catalogLine));
			}
			setChromosome(fields[0]);
			mMinPosition = returnLongPosition(fields[1], 2, "min position");
			mMaxPosition = returnLongPosition(fields[2], 3, "max position");
		}
		else if (fields.length == 1)
		{
			is1FieldEntry = true;
			setJsonString(fields[0]);
			if (!looksLikeJson(getJsonString()))
			{
				throw new CatalogFormatException(
					String.format("A simple check of the 1st field (%s) from '%s' doesn't look like JSON",
						getJsonString(), catalogLine));
			}
		}
		else
		{
			throw new CatalogFormatException(String.format("'%s' must be 4 tab separated fields or 1 field. It has %d fields",
				                                            catalogLine, fields.length));
		}
	}

	public boolean is1Field()
	{
		return is1FieldEntry;
	}

	public boolean is4Field()
	{
		return is4FieldEntry;
	}

	public String getChromosome() {
		return mChr;
	}
	private void setChromosome(String chr) {
		this.mChr = chr;
	}
	public Long getMinPosition() {
		return mMinPosition;
	}
	public Long getMaxPosition() {
		return mMaxPosition;
	}
	public String getJsonString() {
		return mjsonString;
	}
	private void setJsonString(String jsonString) {
		this.mjsonString = jsonString;
	}
	public String getLine() {
		return mTabixLine;
	}
	private void setLine(String catalogTabixLine) {
		this.mTabixLine = catalogTabixLine;
	}

	private Long returnLongPosition(String stringPosition, int fieldNumber, String fieldDescription)
		throws CatalogFormatException
	{
		try
		{
			return new Long(stringPosition);
		}
		catch (NumberFormatException e)
		{
		   throw new CatalogFormatException(String.format("%s (%s) from field %d of '%s' is not an integer",
				                                            fieldDescription, stringPosition,
				                                            fieldNumber, getLine()));
		}
	}

	// TODO - update this documentation
	/** isPositional: determine whether the chromosome value is a valid positional chromosome value or not a positional value,
	 *                basically a place-holder so we can have a valid tabix file even when it is a non-positional catalog.
	 *                Now, problem is we are supposed to auto-detect whether we are positional catalog or not
	 *                and so there isn't a guarantee I'll have a valid chromosomes list for the catalog b/c it may not be
	 *                a human catalog. So, as a result, all we can do is check whether the chr value is one of the well-known
	 *                values that would make it a non-positional catalog.
	 * @return true if the chromosome value is not one of the well-known nonpositional chromosome values.
	 *         false if the chromosome value is one of the well-known nonpositional chromosome values.
	 */

	public boolean isPositional()
	{
		return isChromosomePositional() && isMinBPPositional() && isMaxBPPositional();
	}

	public boolean isValidNonPositional()
	{
	   return isUnknownPosition() || isPositionalInfoNull();
	}

	public boolean isPositionalInfoNull()
	{
		return getChromosome() == null && getMinPosition() == null && getMaxPosition() == null;
	}

	public boolean isUnknownPosition()
	{
		if (getChromosome() == null || getMinPosition() == null || getMaxPosition() == null)
		{
			return false;
		}
		return getChromosome().equals(UNKNOWN) && getMinPosition() == 0 && getMaxPosition() == 1;
	}

	public boolean isChromosomePositional()
	{
		if (getChromosome() == null)
		{
			return false;
		}
		if (getChromosome().toUpperCase().equals(UNKNOWN) ||
			 getChromosome().toUpperCase().equals(".")) // previous agreed so allow it for now. example using it: omim.
		{
			return false;
		}
		return true;
	}

	public boolean isMinBPPositional()
	{
		if (getMinPosition() == null)
		{
			return false;
		}
		return getMinPosition() > 0;
	}

	public boolean isMaxBPPositional()
	{
		if (getMaxPosition() == null )
		{
			return false;
		}
		return getMaxPosition() > 0;
	}

	private boolean looksLikeJson(String possibleJson)
	{
		return possibleJson.startsWith("{") && possibleJson.endsWith("}");
	}

	public static final String UNKNOWN = "UNKNOWN";
	private boolean is1FieldEntry = false;
	private boolean is4FieldEntry = false;

	private String mChr = null;
	private Long mMinPosition = null;
	private Long mMaxPosition = null;
	private String mjsonString = null;
	private String mTabixLine = null;
}
