package edu.mayo.bior.catalog;

import org.apache.commons.lang.StringUtils;

import java.io.File;

/**
 * Deals with information about a catalog and it's corresponding files
 */
public class CatalogFiles
{
   private File mCatalogFile;
   private String mPrefix;
   private File mDir;
   private File mColumnsFile;
   private File mDataSourceFile;
   private File mTabixIndexFile;
   private File mBlacklistFile;
   private File mBlacklistBiorwebFile;
   private boolean isDeprecated;


   /**
    * Calls this(new File(catalogPath))
    *
    * @param catalogPath path to the catalog file (including the suffix)
    */
   public CatalogFiles(String catalogPath) throws CatalogFormatException
   {
      this(new File(catalogPath));
   }

   /**
    * @param catalogFile file handle for the catalog file
    * @throws CatalogFormatException if the catalog file doesn't exist, is not a file, is not readable, doesn't end with
    *                                .tsv.bgz, can't find a prefix from the catalog name (e.g. if file is '.tsv.bgz'),
    *                                can't find the parent dir of the catalog, or if there's an issue with any of the
    *                                other catalog files like the tabix index, columns info file, data source file,
    *                                blacklist file or blacklist biorweb file
    */
   public CatalogFiles(File catalogFile) throws CatalogFormatException
   {
      mCatalogFile = catalogFile;
      if (!mCatalogFile.exists())
      {
         throw new CatalogFormatException(String.format("Catalog file '%s' does not exist", catalogFile.getPath()));
      }
      if (!mCatalogFile.isFile())
      {
         throw new CatalogFormatException(String.format("Catalog file '%s' is not a file", catalogFile.getPath()));
      }
      if (!mCatalogFile.canRead())
      {
         throw new CatalogFormatException(String.format("Can not read catalog file '%s'", catalogFile.getPath()));
      }
      setParentDir();
      setPrefix();
      mColumnsFile = setFileIfExists("columns", CatalogMetadataConstant.COLUMN_INFO_SUFFIX);
      mDataSourceFile = setFileIfExists("data source", CatalogMetadataConstant.DATASRC_PROPS_SUFFIX);
      mTabixIndexFile = setFileIfExists("tabix index", CatalogMetadataConstant.CATALOG_TABIX_INDEX_SUFFIX);
      mBlacklistFile = setFileIfExists("blacklist", CatalogMetadataConstant.BLACKLIST_SUFFIX);
      mBlacklistBiorwebFile = setFileIfExists("blacklist biorweb", CatalogMetadataConstant.BLACKLIST_BIORWEB_SUFFIX);
      isDeprecated = new File(mDir, getPrefix() + CatalogMetadataConstant.CATALOG_DEPRECATED_SUFFIX).exists();
   }

   /**
    * @return The file handle to the catalog bgzip file.
    */
   public File getCatalogFile()
   {
      return mCatalogFile;
   }

   /**
    * @return the file handle to the columns info if the file exists and it is readable. null is returned if the file
    *         doesn't exist
    */
   public File getColumnsFile()
   {
      return mColumnsFile;
   }

   /**
    * @return the file handle to the data source if the file exists and it is readable. null is returned if the file
    *         doesn't exist
    */
   public File getDataSourceFile()
   {
      return mDataSourceFile;
   }

   /**
    * @return the file handle to the tabix index if the file exists and it is readable. null is returned if the file
    *         doesn't exist
    */
   public File getTabixIndexFile()
   {
      return mTabixIndexFile;
   }

   /**
    * @return the file handle to the blacklist if the file exists and it is readable. null is returned if the file
    *         doesn't exist
    */
   public File getBlacklistFile()
   {
      return mBlacklistFile;
   }

   /**
    * @return the file handle to the blacklist biorweb link file if the file exists and it is readable. null is returned
    *         if the file doesn't exist
    */
   public File getBlacklistBiorwebFile()
   {
      return mBlacklistBiorwebFile;
   }

   /**
    * @return prefix for the catalog file (e.g. 'catalog' for catalog file name 'catalog.tsv.bgz')
    */
   public String getPrefix()
   {
      return mPrefix;
   }

   /**
    * @return directory where the catalog files live
    */
   public File getDir()
   {
      return mDir;
   }

   /**
    * @return TRUE if the Catalog is deprecated.  False otherwise.
    */
   public boolean isDeprecated() { return isDeprecated; }

   private void setPrefix() throws CatalogFormatException
   {
      String basename = mCatalogFile.getName();
      String catalogSuffix = CatalogMetadataConstant.CATALOG_FILE_SUFFIX;

      if (!basename.endsWith(catalogSuffix))
      {
         throw new CatalogFormatException(String.format("Catalog file '%s' does not end with %s", mCatalogFile, catalogSuffix));
      }

      int indexOfCatalogSuffix = basename.indexOf(catalogSuffix);
      if (indexOfCatalogSuffix != -1 && (indexOfCatalogSuffix + catalogSuffix.length() == basename.length()))
      {
         mPrefix = basename.substring(0, basename.indexOf(catalogSuffix));
         if (StringUtils.isEmpty(mPrefix))
         {
            throw new CatalogFormatException(String.format("Could not extract catalog prefix from catalog file '%s'",
                                                           mCatalogFile));
         }
      }
   }

   private void setParentDir() throws CatalogFormatException
   {
      mDir = CatalogFileUtils.getParent(mCatalogFile);
      if (!mDir.isDirectory())
      {
         throw new CatalogFormatException(String.format("Could not get directory for catalog file '%s'", mCatalogFile));
      }
   }

   private File setFileIfExists(String description, String suffix) throws CatalogFormatException
   {
      File file = new File(mDir, getPrefix() + suffix);
      if (!file.exists())
      {
         return null;
      }
      if (!file.isFile())
      {
         throw new CatalogFormatException(String.format("%s file '%s' exists but is not a file",
                                                        description, file.getPath()));
      }
      if (!file.canRead())
      {
         throw new CatalogFormatException(String.format("Can not read %s file '%s'", description, file.getPath()));
      }
      return file;
   }
}
