package edu.mayo.bior.catalog;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.samtools.util.BlockCompressedInputStream;

public class CatalogFileUtils {
	private static final Logger sLogger = LoggerFactory.getLogger(CatalogFileUtils.class);

	/**
	 * getBufferedReader: creates a BufferedReader for a file, ensuring it exists prior. It supports
	 * either a bgzipped (".bgz" file extension) file or a regular text file right now.
	 *
	 * @param filePath The text file or bgzip file path to read
	 * @return a BufferedReader that the caller needs to close before leaving the scope. If the file doesn't exist or
	 * there is a problem getting a reader, null is returned.
	 */
	public static BufferedReader getBufferedReader(String filePath)
	{

		File f = new File(filePath);
		if (!f.exists())
		{
			String msg = String.format("Couldn't get buffered reader for '%s' because it does not exist", f.getPath());
			sLogger.error(msg);
			return null;
		}
		if (!f.isFile())
		{
			String msg = String.format("Couldn't get buffered reader for '%s' because it is not a file", f.getPath());
			sLogger.error(msg);
			return null;
		}
		if (!f.canRead())
		{
			String msg = String.format("Couldn't get buffered reader for '%s' because it is not readable", f.getPath());
			sLogger.error(msg);
			return null;
		}

		BufferedReader r = null;
		try
		{
			if (f.getAbsoluteFile().getName().endsWith(".bgz"))
			{
				r = new BufferedReader(new InputStreamReader(new BlockCompressedInputStream(new FileInputStream(f))));
			}
			else
			{
				r = new BufferedReader(new FileReader(f));
			}
		}
		catch (IOException e)
		{
			String msg = String.format("Couldn't get buffered reader for '%s'. Msg: '%s'", f.getPath(), e.getMessage());
			sLogger.error(msg);
			return null;
		}
		return r;
	}
	
	public static boolean validSymLink(File f) {
		try {
			String resolvedFileNm = f.getCanonicalPath(); // resolves symlinks
			if (resolvedFileNm == null || resolvedFileNm.length() == 0) {
				return false;
			}
		} catch (IOException io) {
			//logger.write("Unable to follow/resolve symlink for biorweb sym link file: " + blackList_biorWebSymLink.getAbsolutePath());
			return false;
		} catch (SecurityException s) {
			//logger.write("Security exception trying to follow/resolve symlink for biorweb sym link file: " + blackList_biorWebSymLink.getAbsolutePath());
			return false;
		}
		
		return true;
	}

	public static File getParent(File f)
	{
		File parent = f.getParentFile();
		if (parent != null)
		{
			return parent;
		}
		return f.getAbsoluteFile().getParentFile();
	}

	/**
	 * Finds all files with the suffix .tsv.bgz located under the given rootDir.
	 * @param rootDir The root level directory to start finding files
	 * @param exhaustive If true, all sub-directories are exhaustively searched for .tsv.bgz files.
	 *                   If false, once a directory is found with a .tsv.bgz file, no sub-directories are searched.  This is
	 *                   helpful performance wise to avoid the 'build' sub-directory that is often found in a catalog dir.
	 * @return List of files found
	 */
	public static List<File> findAllTsvBgzFiles(File rootDir, boolean exhaustive) {
		List<File> hits = new ArrayList<File>();
		List<File> subdirs =  new ArrayList<File>();
		File[] nestedFiles = rootDir.listFiles();
		if (nestedFiles != null) {
			for (File f: nestedFiles) {
				if (f.isFile() && f.getName().toLowerCase().endsWith(".tsv.bgz")) {
					hits.add(f);
				} else if (f.isDirectory()) {
					subdirs.add(f);
				}
			}
		}

		// only recurse to subdirs if the exhaustive flag is TRUE or we found NO HITS at the current directory level
		// this is to boost performance since a proper catalog directory should contain a .tsv.bgz file
		// a catalog directory should not contain a nested catalog directory inside it, thus no need to go deeper
		if (exhaustive || (hits.size() == 0)) {
			for (File subdir: subdirs) {
				// recurse and look for nested .tsv.bgz files one level down
				hits.addAll(findAllTsvBgzFiles(subdir, exhaustive));
			}
		}
		return hits;
	}
}
