package edu.mayo.bior.buildcatalog;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import edu.mayo.bior.pipeline.createcatalog.TjsonToCatalog;
import edu.mayo.exec.Command;
import htsjdk.samtools.util.BlockCompressedInputStream;

import org.apache.commons.io.IOUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The MAKE_JSON step within the BuildCatalog command
 *
 * @author Michael Meiners (m054457)
 *         2016-04-25
 */
public class MakeJsonStep implements BuildCatalogStepInterface
{
   private static Logger sLogger = LoggerFactory.getLogger(MakeJsonStep.class);

   private BuildInfo mBuildInfo;
   private StepLogger mStepLogger;

   private int mCmdExitCode;
   private String mCmdStdout;
   private String mCmdStderr;

   private static final String STEP_DISPLAY_NAME = BuildStepKey.MAKE_JSON.getDisplayName();

   public MakeJsonStep(BuildInfo buildInfo, StepLogger stepLogger)
   {
      mBuildInfo = buildInfo;
      mStepLogger = stepLogger;
   }

   private void checkInput() throws BuildCatalogStepInputException
   {
      // Verify the make_json path exists
      File scriptFile = mBuildInfo.getMakeJsonScriptPathFile();
      if (!scriptFile.exists())
      {
         throw new BuildCatalogStepInputException(String.format("Error: could not find %s script '%s'.",
            STEP_DISPLAY_NAME, mBuildInfo.getMakeJsonScriptPath()));
      }
      if (!scriptFile.canExecute())
      {
         throw new BuildCatalogStepInputException(String.format("Error: %s script '%s' is not executable.",
            STEP_DISPLAY_NAME, mBuildInfo.getMakeJsonScriptPath()));
      }
   }

   public void execute() throws BuildCatalogStepInputException, BuildCatalogStepExecuteException
   {
      checkInput();

      Command cmd = null;
      try
      {
          // For more info on running commands see: BaseFunctionalTest.java
          String[] cmdArgs = createMakeJsonCommand(mBuildInfo);

          Map<String,String> envVars = getEnvironmentMap(mBuildInfo);

          // create and execute the command
          cmd = new Command(cmdArgs, envVars, true);

          String msg = String.format("Executing script '%s' with args: '%s'",
        		  			mBuildInfo.getMakeJsonScriptPath(),  mBuildInfo.getMakeJsonArgs());
          mStepLogger.logAndSummary(msg);
          sLogger.info(msg);
          cmd.execute();
      }
      catch (Exception e)
      {
         throw new BuildCatalogStepExecuteException(
        	String.format("Problem executing the script during the %s step", STEP_DISPLAY_NAME), e);
      }

      mCmdExitCode = cmd.getExitCode();
      mCmdStdout = cmd.getStdout();
      mCmdStderr = cmd.getStderr();

      String msg = String.format("%s step:%n", STEP_DISPLAY_NAME);
      mStepLogger.log(msg);
      sLogger.info(msg);

      String warningLineBreak = mCmdExitCode == 0  ?  ""  :  "!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n";
      msg = String.format("%sExit code: %d%n%s", warningLineBreak, mCmdExitCode, warningLineBreak);
      mStepLogger.log(msg);
      sLogger.info(msg);

      msg = String.format("STDOUT:%n%s%n", mCmdStdout);
      mStepLogger.log(msg);
      sLogger.info(msg);

      msg = String.format("STDERR:%n%s%n", mCmdStderr);
      mStepLogger.log(msg);
      sLogger.info(msg);

      checkOutput();
   }

   private Map<String, String> getEnvironmentMap(BuildInfo buildInfo) throws IOException {
       // pass along current environment
       Map<String, String> envVars = new HashMap<String, String>();
       
       // Add all system environment variables
       envVars.putAll(System.getenv());
       envVars.putAll(buildInfo.getMap());

       // Special case for $PATH variable - prepend any user-specified paths to the PATH variable
       String path = buildInfo.getVal(BuildInfoKey.PATH);
       if( path != null  &&  path.trim().length() > 0 ) {
    	   path = path + ":" + System.getenv().get("PATH");
    	   envVars.put(BuildInfoKey.PATH.name(), path);
       }
       
       return envVars;
   }

   private boolean isGiven(String val) {
	   return val != null  &&  val.trim().length() > 0;
   }

   private void checkOutput() throws BuildCatalogStepExecuteException
   {
      // Verify exit code was 0
      if (mCmdExitCode != 0)
      {

         String msg = String.format("Output of '%s' was non-zero: %d%nSTDOUT:%n%s%nSTDERR:%n%s%n",
            mBuildInfo.getMakeJsonScriptPath(), mCmdExitCode, mCmdStdout, mCmdStderr);
         throw new BuildCatalogStepExecuteException(msg);
      }

      // Print to stdout any warnings or errors
      if (mCmdStderr.length() > 0)
      {
         String msg = String.format("Warning: Saw this output to stderr but exit code was 0. Continuing.%n%s", mCmdStderr);
         System.out.println(msg);
      }

      // Verify that the output file exists
      File outputFile = new File(mBuildInfo.getMakeJsonOutputFilePath());
      if (!outputFile.exists())
      {
         throw new BuildCatalogStepExecuteException(String.format("Error: output file path does not exist: '%s'",
            outputFile.getPath()));
      }

      if (outputFile.length() == 0)
      {
         throw new BuildCatalogStepExecuteException(String.format("Error: output file was empty: '%s'",
            outputFile.getPath()));
      }

      verifyLastColIsJson(outputFile);
   }


   private void verifyLastColIsJson(File outputFile) throws BuildCatalogStepExecuteException
   {
      // Verify that the output is a text file with JSON in the last column for the first 100 lines
      // (or less if there are not 100 lines in the file)
      BufferedReader fin = null;
      try
      {
         fin = TjsonToCatalog.getBufferedReader(outputFile);
         String line;
         long lineNum = 1;
         while ((line = fin.readLine()) != null && lineNum <= 100)
         {
            // Skip comment lines
            if (line.startsWith("#"))
            {
               continue;
            }
            String[] cols = line.split("\t");
            String lastCol = cols[cols.length - 1];
            boolean isJson = lastCol.startsWith("{") && lastCol.endsWith("}");
            if (!isJson)
            {
               throw new BuildCatalogStepExecuteException(
            		   String.format("Error: each line in the %s output must end with a JSON object", STEP_DISPLAY_NAME));
            }
            lineNum++;
         }
      }
      catch (IOException e)
      {
         String errMsg = String.format("Issue checking if the output file '%s' has JSON in the last column",
            outputFile.getPath());
         throw new BuildCatalogStepExecuteException(errMsg, e);
      }
      finally
      {
         IOUtils.closeQuietly(fin);
      }
   }


private String[] createMakeJsonCommand(BuildInfo buildInfo) throws IOException
   {
      return new String[]
         {
            "/bin/sh",
            "-c",
            buildInfo.getMakeJsonScriptPath() + "  " + buildInfo.getMakeJsonArgs()
         };
   }


   /**
    * Get the exit code after executing the script
    * FOR TESTING ONLY
    */
   int getScriptExitCode()
   {
      return mCmdExitCode;
   }


   /**
    * Get the STDERR output after executing the script
    * FOR TESTING ONLY
    */
   String getScriptStderr()
   {
      return mCmdStderr;
   }

   /**
    * Get the STDOUT output after executing the script
    * FOR TESTING ONLY
    */
   String getScriptStdout()
   {
      return mCmdStdout;
   }

}
