package edu.mayo.bior.cli.func;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Map;
import java.util.zip.GZIPInputStream;

import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class CreateCatalogCommandITCase extends BaseFunctionalTest {

	private static final String DIR 			= "src/test/resources/createCatalog/";
	private static final String INPUT_FILE  	= DIR + "input.json";
	//private static final String OUTPUT_FILE		= DIR + "outputCatalog.tsv.bgz";
	//private static final String OUTPUT_TABIX	= OUTPUT_FILE + ".tbi";
	private static final String EXPECTED_OUTPUT_FILE	= DIR + "outputCatalog.expected.tsv.bgz";
	
	@Rule
	public TemporaryFolder tempFolder = new TemporaryFolder();
	private File tempDir;
	private File outputFile;
	private File outputTabixFile;
	
    private final String STDIN = "";

	@BeforeClass
	public static void beforeAll() throws FileNotFoundException {
		setBiorToolkitCmdsRequired(true);
	}
    
	@Before
	@After
	public void removeTempOutputFiles() throws IOException {
		tempDir = tempFolder.newFolder();
		outputFile 		= new File(tempDir, "outputCatalog.tsv.bgz");
		outputTabixFile = new File(tempDir, "outputCatalog.tsv.bgz.tbi");
		//removeFile(OUTPUT_FILE);
		//removeFile(OUTPUT_TABIX);
	}
	
	private void removeFile(String filePath) throws IOException {
		File output = new File(filePath).getCanonicalFile();
		if(output.exists())
			output.delete();
	}

	@Test
	public void testShortArgs() throws IOException, InterruptedException {
        System.out.println("CreateCatalogCommandITCase.testShortArgs");
        
	    CommandOutput out = executeScript("bior_create_catalog", STDIN, "-i", INPUT_FILE, "-o", outputFile.getCanonicalPath());
	    assertOutputs(out);
	}
	
	@Test
	public void testLongArgs() throws IOException, InterruptedException {
        System.out.println("CreateCatalogCommandITCase.testLongArgs");
	    CommandOutput out = executeScript("bior_create_catalog", STDIN, "--input", INPUT_FILE, "--output", outputFile.getCanonicalPath(), "--column", "1");
	    assertOutputs(out);
	}
	
	/** NOTE: This should be covered in CreateCatalogCommandTest.java
	@Test
	// NOTE: We were getting an error prior to the fix for large <DEL>s that said:
	//       "Error: length of the refAllele does not equal (max-min+1):"
	//       But this should build ok
	public void testLargeDels() throws IOException, InterruptedException {
        System.out.println("CreateCatalogCommandITCase.testLargeDels");
		File inputFile = new File(DIR, "largeDels.json");
	    CommandOutput out = executeScript("bior_create_catalog", STDIN, 
	    		"--input",  inputFile.getCanonicalPath(),
	    		"--output", outputFile.getCanonicalPath()
	    		);
	    assertOutputs(out);
	}
	*/


	private void assertOutputs(CommandOutput out) throws FileNotFoundException, IOException {
	    //printPathVar();
	    warnIfNoTabix(out);
        assertEquals(out.stderr, 0, out.exit);
        assertEquals("", out.stderr);
        assertTrue(outputFile.exists());
        assertTrue(outputTabixFile.exists());
        // Assert the output looks as expected
        String expected = readGzipFile(EXPECTED_OUTPUT_FILE);
        String actual   = readGzipFile(outputFile.getCanonicalPath());
        assertEquals(expected, actual);
	}



	private void warnIfNoTabix(CommandOutput out) {
	    if( out.stderr.contains("I require tabix") || out.stderr.contains("I require bgzip") ) {
	    	System.out.println("Please add the tabix directory to your path to pick up both tabix and bgzip, ");
	    	System.out.println("  or for the case of tests within Eclipse which may not pick up the $PATH variable,");
	    	System.out.println("  please create a symbolic link to either:");
	    	System.out.println("    " + BaseFunctionalTest.TABIX_PATH1 + "   OR:");
	    	System.out.println("    " + BaseFunctionalTest.TABIX_PATH2 + "\n");
	    	System.out.println("  For example, say your tabix executable is installed on your system at (but not in the path):");

	    	String tabixDirInstall = "/Users/m054457/DATA/BioR_2011/JavaProjects/Code/BiorTabix/tabix-0.2.6/";
	    	int idxTabixLastDir   = BaseFunctionalTest.TABIX_PATH2.lastIndexOf("/") + 1;
	    	String tabixParentDir = BaseFunctionalTest.TABIX_PATH2.substring(0, idxTabixLastDir);
	    	String tabixLastDir   = BaseFunctionalTest.TABIX_PATH2.substring(idxTabixLastDir);
	    	
	    	System.out.println("    " + tabixDirInstall + "\n");
	    	System.out.println("  You could then create the required directories and create a symlink to the tabix directory:");
	    	System.out.println("    mkdir -p " + tabixParentDir);
	    	System.out.println("    cd " + tabixParentDir);
	    	System.out.println("    ln -s  " + tabixDirInstall + "  " + tabixLastDir + "\n");
	    	System.out.println("You can download tabix/bgzip from:");
	    	System.out.println("  https://github.com/samtools/htslib \n\n");
	    }
	}

	private void printPathVar() {
		Map<String, String> env = System.getenv();
        for (String envName : env.keySet()) {
            System.out.format("%s=%s%n",
                              envName,
                              env.get(envName));
        }
	}
	
	private String readGzipFile(String filePath) throws FileNotFoundException, IOException {
	    byte[] buf = new byte[64*1024];
	    GZIPInputStream gzipInputStream = new GZIPInputStream(new FileInputStream(filePath));
	    int len = -1;
	    StringBuilder str = new StringBuilder();
	    while( (len = gzipInputStream.read(buf)) != -1 ) {
	    	str.append(new String(buf, 0, len));
	    }
	    gzipInputStream.close();
	    return str.toString();
	}

	
	private void pathJunk() throws IOException, InterruptedException {
		Map<String, String> env = System.getenv();
		String path1 = env.get("PATH");
		
		String pathProp = System.getProperty("PATH");
		final String tabixDir  = "/projects/bsi/bictools/apps/alignment/tabix/0.2.5/";
		final String tabixDir2 = "/projectsLocal/bsi/bictools/apps/alignment/tabix/0.2.5/";
		

		path1 = tabixDir + ":" + tabixDir2 + ":" + path1;
		
		System.setProperty("PATH", path1);
		
		env = System.getenv();
		String path2 = env.get("PATH");
		
		System.out.println();
		
		
        String path = System.getProperty("java.library.path");
        String clspath = System.getProperty("java.class.path");
        
        ProcessBuilder pb = new ProcessBuilder(new String[] { "sh", "-c", "echo $PATH" });
        Process p = pb.start();
        java.util.Scanner s1 = new java.util.Scanner(p.getInputStream()).useDelimiter("\\A");
        String stdout = s1.hasNext() ? s1.next() : "";
        java.util.Scanner s2 = new java.util.Scanner(p.getErrorStream()).useDelimiter("\\A");
        String stderr = s2.hasNext() ? s2.next() : "";
        
        CommandOutput o = executeScript("/bin/echo", "$PATH");
        CommandOutput out1 = executeScript("export", "PATH=$PATH:" + tabixDir + ":" + tabixDir2);
	}
}
