package edu.mayo.bior.cli.cmd;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.cli.func.BaseFunctionalTest;
import edu.mayo.bior.cli.func.CommandOutput;

/** Test the hidden command for updating a columns.tsv file from an authoritative/patch one */
public class UpdateColumnsTsvCommandTest extends BaseFunctionalTest {
	@Rule
	public TemporaryFolder mTempFolder = new TemporaryFolder();
	
	// The Columns.tsv to be modified
	private File mTargetColumnsTsv;
	// The authoritative columns.tsv to pull fields from
	// This could also consist of just 2 or 3 columns:  ColumnName, HumanReadableName, Description
	private File mPatchColumnsTsv;

	private final String EOL = "\n";

	
	@Before
	public void beforeEach() throws IOException {
		mTempFolder.create();
		mTargetColumnsTsv = mTempFolder.newFile("target.columns.tsv");
		mPatchColumnsTsv  = mTempFolder.newFile("patch.columns.tsv");
	}
	
    @Test
    public void testNormal() throws IOException {
    	FileUtils.write(mTargetColumnsTsv,
    			"##Some header" + EOL +
    			concat("#ColumnName", "Type",   "Count", "Description",        "HumanReadableName") + EOL +
    			concat("CHROM",       "String", "1",     "Chromosome desc",    "Chromosome") + EOL +
    			concat("ID",          "Integer","1",     "Identifier",         "rsId") + EOL +
    			concat("ALT",         "String", ".",     "\"Alt\" alleles",    "AltsssAlleles") + EOL +
    			concat("REF",         "Bools",  "X",     "",                   "") + EOL +
    			// This is not updated except for the double-quotes because it doesn't have a counterpart in the patch file
    			concat("MAF",         "Float",  "1",     "\"Minor\" allele freq","MinorAlleleFreq") + EOL +
    			// The Description and HumanReadableName should be updated (there is no matching line in the patch file)
    			concat("JAT",         "Boolean","0",     "",                   "")
    			);
    	
    	FileUtils.write(mPatchColumnsTsv,
    			"##Some header 2" + EOL +
    			concat("#ColumnName", "Type",   "Count", "Description",               "HumanReadableName") + EOL +
    			concat("CHROM",       "Integer","1",     "Chromosome as an integer",  "Chromosome (VCF Field)") + EOL +
    			concat("ALT",         "String", ".",     "Alternate alleles",         "AltAlleles") + EOL +
    			concat("REF",         "String", "1",     "Ref allele",                "REF")
    			);

 
    	final String EXPECTED_MSG =
    			// NOTE: The double-quotes are stripped out by the loader before we can deal with them:
    			"Warning: the Description in columns.tsv contains a double-quote which would mess up any downstream VCF validators.  Converting these to single-quotes.  Offending Line: [ALT	String	.	\"Alt\" alleles	AltsssAlleles].  File: " + mTargetColumnsTsv.getCanonicalPath() + EOL +
    			"Warning: the Description in columns.tsv contains a double-quote which would mess up any downstream VCF validators.  Converting these to single-quotes.  Offending Line: [MAF	Float	1	\"Minor\" allele freq	MinorAlleleFreq].  File: " + mTargetColumnsTsv.getCanonicalPath() + EOL +
    			EOL +
    			"--------------------------------------------" + EOL +
    			"Field: CHROM" + EOL +
    			"  HumanReadableName updated: [Chromosome] ==> [Chromosome (VCF Field)]" + EOL +
    			"  Description updated:       [Chromosome desc] ==> [Chromosome as an integer]" + EOL +
    			"  Warning: Type is different between target [String] and patch [Integer]" + EOL +
    			// NOTE: Skip "ID" field since no changes
    			//    ....
    			EOL +
    			"--------------------------------------------" + EOL +
    			"Field: ALT" + EOL +
    			"  HumanReadableName updated: [AltsssAlleles] ==> [AltAlleles]" + EOL +
    			"  Description updated:       ['Alt' alleles] ==> [Alternate alleles]" + EOL +
    			// NOTE: This warning about description not containing double-quotes is pre-empted by the loader which replaces double-quotes with single quotes
    			//"  Warning: Description cannot contain double-quotes.  These have been converted to single quotes." + EOL +
    			EOL +
    			"--------------------------------------------" + EOL +
    			"Field: REF" + EOL +
    			// NOTE: The Description and HumanReadableName are updated from the patch file BEFORE the check for empty columns
    			"  HumanReadableName updated: [] ==> [REF]" + EOL +
    			"  Description updated:       [] ==> [Ref allele]" + EOL +
    			"  Warning: Type is different between target [Bools] and patch [String]" + EOL +
    			"  Warning: Count is different between target [X] and patch [1]" + EOL +
    			"  ERROR: Type [Bools] is not valid.  Possible types include: JSON,  String,  Float,  Integer,  Boolean" + EOL +
    			"  ERROR: Count [X] is not valid.  It must be one of the values: '.', 'A', 'G', 'R', or a non-negative integer (0,1,2...)" + EOL +
    			"  Warning: HumanReadableName is same as ColumnName [REF].  Please update the HumanReadableName" + EOL +
    			// NOTE: This warning about description not containing double-quotes is pre-empted by the loader which replaces double-quotes with single quotes
    			//"--------------------------------------------" + EOL +
    			//"Field: MAF" + EOL +
    			//"  Warning: Description cannot contain double-quotes.  These have been converted to single quotes." + EOL
    			EOL +
    			"--------------------------------------------" + EOL +
    			"Field: JAT" + EOL +
    			"  Warning: Description is missing.  Setting this to dot." + EOL +
    			"  Warning: HumanReadableName is missing.   Setting this to to the same as the ColumnName." + EOL +
    			"  Warning: HumanReadableName is same as ColumnName [JAT].  Please update the HumanReadableName" + EOL
    			;

    	CommandOutput output = runCmdApp(new UpdateColumnsTsvCommand(),  "_bior_update_columns_tsv", "-f", mTargetColumnsTsv.getCanonicalPath(), "-p", mPatchColumnsTsv.getCanonicalPath());
    	
    	assertEquals(output.stderr, EXPECTED_MSG, output.stderr);
		assertEquals("", output.stdout);
        assertEquals(output.exit,   0);
        
        // Verify the modified columns.tsv:
       	final String EXPECTED_COLUMNS_OUT =
    			"##Some header" + EOL +
    			concat("#ColumnName", "Type",   "Count", "Description",               "HumanReadableName") + EOL +
    			// Update Description and HumanReadableName
    			concat("CHROM",       "String", "1",     "Chromosome as an integer",  "Chromosome (VCF Field)")	+ EOL +
    			// No changes at all and no warnings
    			concat("ID",          "Integer","1",     "Identifier",                "rsId") + EOL +
    			// Update Description and HumanReadableName; dot in count should NOT throw error;
    			concat("ALT",         "String", ".",     "Alternate alleles",         "AltAlleles") + EOL +
    			// No change to Description and HumanReadableName, warning about invalid type and count; warning about HumanReadableName being same as ColumnName
    			concat("REF",         "Bools", "X",      "Ref allele",                "REF") + EOL +
    			// Double-quotes in Description converted to single-quotes
    			concat("MAF",         "Float",  "1",     "'Minor' allele freq",       "MinorAlleleFreq") + EOL +
    			// Empty Description converted to dot;  empty HumanReadableName set to same as ColumnName
    			concat("JAT",         "Boolean","0",     ".",                         "JAT") + EOL;

        assertEquals(EXPECTED_COLUMNS_OUT, FileUtils.readFileToString(mTargetColumnsTsv));
    }
    
    @Test
    public void testOnly_noPatchFile() throws IOException {
    	String colsTsv = concat("#ColumnName", "Type",   "Count", "Description",        "HumanReadableName") + EOL +
    					 concat("REF",         "Bools",  "X",     "",                   "");

    	FileUtils.write(mTargetColumnsTsv, colsTsv);
    	
    	final String EXPECTED_MSG =
    			EOL +
    			"--------------------------------------------" + EOL +
    			"Field: REF" + EOL +
    			"  Warning: Description is missing.  Setting this to dot." + EOL +
    			"  Warning: HumanReadableName is missing.   Setting this to to the same as the ColumnName." + EOL +
    			"  ERROR: Type [Bools] is not valid.  Possible types include: JSON,  String,  Float,  Integer,  Boolean" + EOL +
    			"  ERROR: Count [X] is not valid.  It must be one of the values: '.', 'A', 'G', 'R', or a non-negative integer (0,1,2...)" + EOL +
    			"  Warning: HumanReadableName is same as ColumnName [REF].  Please update the HumanReadableName" + EOL +
				EOL +				
				"NOTE: The test-only flag was supplied, so changes will NOT go into the target columns.tsv" + EOL +
				EOL
				;
    	  
    	
    	CommandOutput output = runCmdApp(new UpdateColumnsTsvCommand(),  "_bior_update_columns_tsv", "-f", mTargetColumnsTsv.getCanonicalPath(), "-t");
    	
    	assertEquals(output.stderr, EXPECTED_MSG, output.stderr);
		assertEquals("", output.stdout);
        assertEquals(output.exit,   0);
        
        // Verify the columns.tsv is unchanged
        assertEquals(colsTsv, FileUtils.readFileToString(mTargetColumnsTsv));
    }
    
    @Test
    public void testOnly_withPatchFile() throws IOException {
    	String colsTsv = concat("#ColumnName", "Type",   "Count", "Description",        "HumanReadableName") + EOL +
				 		 concat("REF",         "Bools",  "X",     "",                   "");
		
		FileUtils.write(mTargetColumnsTsv, colsTsv);
		
		FileUtils.write(mPatchColumnsTsv, 
				concat("#ColumnName", "Type",   "Count", "Description",        "HumanReadableName") + EOL +
		 		concat("REF",         "String", "1",     "Reference Allele",   "RefAllele")
				);
		
		// Make both the target and patch files read-only as they should not need to be writeable in test mode
		mTargetColumnsTsv.setReadOnly();
		mPatchColumnsTsv.setReadOnly();

		CommandOutput output = runCmdApp(new UpdateColumnsTsvCommand(),  "_bior_update_columns_tsv",
				"-f", mTargetColumnsTsv.getCanonicalPath(),
				"-p", mPatchColumnsTsv.getCanonicalPath(),
				"-t");

		final String EXPECTED_MSG =
				EOL +
				"--------------------------------------------" + EOL +
				"Field: REF" + EOL +
				"  HumanReadableName updated: [] ==> [RefAllele]" + EOL +
				"  Description updated:       [] ==> [Reference Allele]" + EOL +
				"  Warning: Type is different between target [Bools] and patch [String]" + EOL +
				"  Warning: Count is different between target [X] and patch [1]" + EOL +
				"  ERROR: Type [Bools] is not valid.  Possible types include: JSON,  String,  Float,  Integer,  Boolean" + EOL +
				"  ERROR: Count [X] is not valid.  It must be one of the values: '.', 'A', 'G', 'R', or a non-negative integer (0,1,2...)" + EOL +
				EOL +				
				"NOTE: The test-only flag was supplied, so changes will NOT go into the target columns.tsv" + EOL +
				EOL
				;

		assertEquals(output.stderr, EXPECTED_MSG, output.stderr);
		assertEquals("", output.stdout);
		assertEquals(output.exit,   0);
		
		// Verify the columns.tsv is unchanged, since even though we passed a patch file, we are only in test mode
		assertEquals(colsTsv, FileUtils.readFileToString(mTargetColumnsTsv));
    }
    
    @Test
    /** Update from patch file that has only ColumnName and HumanReadableName */
    public void updateFromTwoColPatchFile() throws IOException {
    	String colsTsv = concat("#ColumnName", "Type",   "Count", "Description",  "HumanReadableName") + EOL +
		 		 		 concat("CHROM",       "String", "1",     "chr",          "CHROM");

		FileUtils.write(mTargetColumnsTsv, colsTsv);
		
		final String EXPECTED_MSG =
				EOL +
				"--------------------------------------------" + EOL +
				"Field: CHROM" + EOL +
				"  HumanReadableName updated: [CHROM] ==> [Chromosome]" + EOL
				;
		
		FileUtils.write(mPatchColumnsTsv, 
				concat("#ColumnName", "HumanReadableName") + EOL +
				concat("CHROM",       "Chromosome")
				);
		
		CommandOutput output = runCmdApp(new UpdateColumnsTsvCommand(),  "_bior_update_columns_tsv",
				"-f", mTargetColumnsTsv.getCanonicalPath(),
				"-p", mPatchColumnsTsv.getCanonicalPath()
				);
		
		assertEquals(output.stderr, EXPECTED_MSG, output.stderr);
		assertEquals("", output.stdout);
		assertEquals(output.exit,   0);
		
		// Verify the columns.tsv output has changed, with HumanReadableName updated
		final String EXPECTED_COLS_TSV = colsTsv.replace("chr\tCHROM", "chr\tChromosome\n");
		assertEquals(EXPECTED_COLS_TSV, FileUtils.readFileToString(mTargetColumnsTsv));
    }


    @Test
    /** Update from patch file that has only ColumnName, HumanReadableName, Description - out of order from a regular columns.tsv file */
    public void updateFromThreeColPatchFile() throws IOException {
    	String colsTsv = concat("#ColumnName", "Type",   "Count", "Description",  "HumanReadableName") + EOL +
		 		 		 concat("CHROM",       "String", "1",     "chr",          "CHROM");

		FileUtils.write(mTargetColumnsTsv, colsTsv);
		
		final String EXPECTED_MSG =
				EOL +
				"--------------------------------------------" + EOL +
				"Field: CHROM" + EOL +
				"  HumanReadableName updated: [CHROM] ==> [Chromosome]" + EOL +
				"  Description updated:       [chr] ==> [A Human Chromosome]" + EOL
				;
		
		// NOTE: The order of columns is different from normal (here HumanReadableName is before Description)
		FileUtils.write(mPatchColumnsTsv, 
				concat("#ColumnName", "HumanReadableName", "Description") + EOL +
				concat("CHROM",       "Chromosome",        "A Human Chromosome")
				);
		
		CommandOutput output = runCmdApp(new UpdateColumnsTsvCommand(),  "_bior_update_columns_tsv",
				"-f", mTargetColumnsTsv.getCanonicalPath(),
				"-p", mPatchColumnsTsv.getCanonicalPath()
				);
		
		assertEquals(output.stderr, EXPECTED_MSG, output.stderr);
		assertEquals("", output.stdout);
		assertEquals(output.exit,   0);
		
		// Verify the columns.tsv output has changed, with HumanReadableName updated
		final String EXPECTED_COLS_TSV = colsTsv.replace("chr\tCHROM", "A Human Chromosome\tChromosome\n");
		assertEquals(EXPECTED_COLS_TSV, FileUtils.readFileToString(mTargetColumnsTsv));
    }
}
