package edu.mayo.bior.cli.cmd;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.bior.cli.func.BaseFunctionalTest;
import edu.mayo.bior.cli.func.CommandOutput;


// For more tests, see the Pipes project:  ModifyJsonPipeTest.java
public class ModifyTjsonCommandTest extends BaseFunctionalTest {

	@Rule
	public TemporaryFolder mTempFolder = new TemporaryFolder();
	private File mConfigFile; 
	private final String EOL = "\n";
	
	@Before
	public void beforeEach() throws IOException {
		mTempFolder.create();
		mConfigFile = mTempFolder.newFile();
	}
	
	
    @Test
    public void test() throws IOException {
    	String stdin = 
    			concat("1", "100", "100", "{'isInDbSNP':0,'POS':'100','MAF':'0.34','Extra':'Blah','isInt':0}") + EOL + 
    	    	concat("2", "200", "200", "{'isInDbSNP':1,'POS':'200','MAF':'0','isInt':1}") + EOL +
    			concat("3", "300", "300", "{'POS':'NA','MAF':'-inf','isInt':null}") + EOL;

    	FileUtils.write(mConfigFile, 
    			concat("#KEY",      "Operation", "New_type", "New_count", "Delim", "Comment",        "Find", "Set_to", "Find", "Set_to") + EOL +
    			// Boolean is explicitly converted to true/false first (from int values)
    			concat("isInDbSNP", "Modify",    "Boolean",  "0",         ".",     "int to boolean", "0",    "false",  "1",    "true") + EOL +
    			// Boolean is converted by int-to-boolean modification  (note: null should be removed)
    			concat("isInt",     "Modify",    "Boolean",  "0",         ".",     "int to boolean") + EOL +
    			concat("POS",       "Modify",    "Integer",  "1",         ".",     "str to int",     "NA",   "(REMOVE)") + EOL +
    			concat("MAF",       "Modify",    "Float",    "1",         ".",     "str to float",   "-inf", "-999999") + EOL +
    			concat("Extra",     "Remove") + EOL +
    			concat("_type",     "Add",       "String",   "1",         ".",     "add static type","(ANY)","variant")
    			);

    	
    	CommandOutput output = runCmdApp(stdin, new ModifyTjsonCommand(), "bior_modify_tjson", "-f", mConfigFile.getCanonicalPath(), "-c", "4");
        
		String expected = swapQuotes( 
				concat("1", "100", "100", "{'isInDbSNP':false,'POS':100,'MAF':0.34,'isInt':false,'_type':'variant'}") + EOL +
				concat("2", "200", "200", "{'isInDbSNP':true,'POS':200,'MAF':0,'isInt':true,'_type':'variant'}") + EOL +
				concat("3", "300", "300", "{'MAF':-999999,'_type':'variant'}") + EOL
				);
    	
		// Warnings are now going to the logfile instead of stderr
		assertEquals(output.stderr, "", output.stderr);
		assertEquals(expected, output.stdout);
        assertEquals(output.exit,   0);
    }
    

    @Test
    public void testWithHeaders() throws IOException {
    	String stdin = 
    			       "##Simple test with headers" + EOL +
    			concat("#CHROM", "MIN", "MAX", "JSON") + EOL +
    			concat("1", "100", "100", "{'isInDbSNP':0,'POS':'100','MAF':'0.34','Extra':'Blah'}") + EOL + 
    	    	concat("2", "200", "200", "{'isInDbSNP':1,'POS':'200','MAF':'0'}" + EOL  );

    	FileUtils.write(mConfigFile, 
    			concat("#KEY",      "Operation", "New_type", "New_count", "Delim", "Comment",        "Find", "Set_to", "Find", "Set_to") + EOL +
    			concat("isInDbSNP", "Modify",    "Boolean",  "0",         ".",     "int to boolean", "0",    "false",  "1",    "true") + EOL +
    			concat("POS",       "Modify",    "Integer",  "1",         ".",     "str to int",     "NA",   "(REMOVE)") + EOL
    			);
    	
    	CommandOutput output = runCmdApp(stdin, new ModifyTjsonCommand(), "bior_modify_tjson", "-f", mConfigFile.getCanonicalPath(), "-c", "4");
        
		String expected = swapQuotes( 
			    "##Simple test with headers" + EOL +
			    concat("#CHROM", "MIN", "MAX", "JSON") + EOL +
				concat("1", "100", "100", "{'isInDbSNP':false,'POS':100,'MAF':'0.34','Extra':'Blah'}") + EOL +
				concat("2", "200", "200", "{'isInDbSNP':true,'POS':200,'MAF':'0'}") + EOL
				);
    	
		// Warnings are now going to the logfile instead of stderr
		assertEquals(output.stderr, "", output.stderr);
		assertEquals(expected, output.stdout);
        assertEquals(output.exit,   0);
    }

    @Test
    public void testLargeFloats_floatToString() throws IOException {
    	String stdin = 
    			"{'num':12345.12345678901}\n"	// Double
    		  + "{'num':1234567.12345}\n"		// Double
    		  + "{'num':123456789.12345}\n"		// BigDecimal
    		  + "{'num':0.1234567890123456}\n"	// Double
    		  + "{'num':1.1234567890123456}\n"	// BigDecimal
    		  + "{'num':12345678901234567890.12345678901234567899}\n"	// BigDecimal
    		  + "{'nums':[12345.12345678901, 1234567.12345, 123456789.12345, 0.1234567890123456, 1.1234567890123456, 12345678901234567890.12345678901234567899]}";  // Array with mix

    	FileUtils.write(mConfigFile, 
    			concat("#KEY",      "Operation", "New_type", "New_count", "Delim", "Comment") + EOL +
    			// Boolean is explicitly converted to true/false first (from int values)
    			concat("num",       "Modify",    "String",   "1",         ".",     "Float to String") + EOL +
    			concat("nums",      "Modify",    "String",   ".",         ",",     "Floats to Strings Array") + EOL
    			);
    	
    	CommandOutput output = runCmdApp(stdin, new ModifyTjsonCommand(), "bior_modify_tjson",
    			"-f", mConfigFile.getCanonicalPath(), "-c", "1");
        
		String EXPECTED = swapQuotes( 
	   			    "{'num':'12345.12345678901'}\n"		// Double
	    		  + "{'num':'1234567.12345'}\n"			// Double
	    		  + "{'num':'123456789.12345'}\n"		// BigDecimal
	    		  + "{'num':'0.1234567890123456'}\n"	// Double
	    		  + "{'num':'1.1234567890123456'}\n"	// BigDecimal
	    		  + "{'num':'12345678901234567890.12345678901234567899'}\n"	// BigDecimal
	    		  + "{'nums':['12345.12345678901','1234567.12345','123456789.12345','0.1234567890123456','1.1234567890123456','12345678901234567890.12345678901234567899']}\n"  // Array with mix
				);
    	
		// Warnings are now going to the logfile instead of stderr
		assertEquals(output.stderr, "", output.stderr);
		assertEquals(EXPECTED, output.stdout);
        assertEquals(output.exit,   0);
    }
    
    @Test
    public void testLargeFloats_stringToFloat() throws IOException {
    	String stdin = 
    			"{'num':'12345.12345678901'}\n"	// Double
    		  + "{'num':'1234567.12345'}\n"		// Double
    		  + "{'num':'123456789.12345'}\n"		// BigDecimal
    		  + "{'num':'0.1234567890123456'}\n"	// Double
    		  + "{'num':'1.1234567890123456'}\n"	// BigDecimal
    		  + "{'num':'12345678901234567890.12345678901234567899'}\n"	// BigDecimal
    		  + "{'nums':['12345.12345678901','1234567.12345','123456789.12345','0.1234567890123456','1.1234567890123456','12345678901234567890.12345678901234567899']}";  // Array with mix

    	FileUtils.write(mConfigFile, 
    			concat("#KEY",      "Operation", "New_type", "New_count", "Delim", "Comment") + EOL +
    			// Boolean is explicitly converted to true/false first (from int values)
    			concat("num",       "Modify",    "Float",    "1",         ".",     "String to Float") + EOL +
    			concat("nums",      "Modify",    "Float",    ".",         ",",     "Strings to Floats Array") + EOL
    			);
    	
    	CommandOutput output = runCmdApp(stdin, new ModifyTjsonCommand(), "bior_modify_tjson",
    			"-f", mConfigFile.getCanonicalPath(), "-c", "1");
        
		String EXPECTED = swapQuotes( 
	   			    "{'num':12345.12345678901}\n"		// Double
	    		  + "{'num':1234567.12345}\n"			// Double
	    		  + "{'num':123456789.12345}\n"		// BigDecimal
	    		  + "{'num':0.1234567890123456}\n"	// Double
	    		  + "{'num':1.1234567890123456}\n"	// BigDecimal
	    		  + "{'num':12345678901234567890.12345678901234567899}\n"	// BigDecimal
	    		  + "{'nums':[12345.12345678901,1234567.12345,123456789.12345,0.1234567890123456,1.1234567890123456,12345678901234567890.12345678901234567899]}\n"  // Array with mix
				);
    	
		// Warnings are now going to the logfile instead of stderr
		assertEquals(output.stderr, "", output.stderr);
		assertEquals(EXPECTED, output.stdout);
        assertEquals(output.exit,   0);   }
   
}
