package edu.mayo.bior.catalog.verification;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;

import org.junit.Before;
import org.junit.Test;

public class SeqLookupTest {
	
	private SeqLookup seqLookup;
	
	@Before
	public void beforeEach() throws IOException {
		File refAssemblyCtg = new File("src/test/resources/testData/fastaToCtg/refAssembly.50PerLine.tsv.bgz");
		//HumanReferenceInfo refInfo = new HumanReferenceInfo(null,  null);
		seqLookup = new SeqLookup(refAssemblyCtg.getCanonicalPath());
	}

	@Test
	public void testSeqSingleBP() throws VerifierInputException, IOException {
		// Start of Chrom 1
		assertEquals("T", seqLookup.getRefSeq("1", 1, 1));
		// Mid Chrom 1
		assertEquals("G", seqLookup.getRefSeq("1", 100, 100));
		// End Chrom 1
		assertEquals("C", seqLookup.getRefSeq("1", 191, 191));
		// Start Chrom 2
		assertEquals("G", seqLookup.getRefSeq("2", 1, 1));
		// Mid Chrom 2
		assertEquals("A", seqLookup.getRefSeq("2", 200, 200));
		// End Chrom 2
		assertEquals("T", seqLookup.getRefSeq("2", 400, 400));
		
	}
	
	@Test
	// Multi-base pair sequences and those spanning multiple lines
	public void testSeqMultiBP() throws NumberFormatException, IOException {
		// Start Chrom 1
		assertEquals("TNNNNNNNNN", 		seqLookup.getRefSeq("1", 1, 10));
		// Mid Chrom 1 - line wrap
		assertEquals("CCTGAC", 			seqLookup.getRefSeq("1", 97, 102));
		// End Chrom 1
		assertEquals("ACCCTAAC", 		seqLookup.getRefSeq("1", 184, 191));
		// Start Chrom 2
		assertEquals("GNNNNNNNNN",		seqLookup.getRefSeq("2", 1, 10));
		// Mid Chrom 2 - Wrap 5 lines:
		assertEquals("ACACCCACACACACCACACCCACACACCACACCCACACCCACACACCACA"
				+    "CCCACACCACACCCACACACCACACACCACACCCACACCCACACACACCA"
				+    "CACCCACACACCACACCCACACACACCCTAACCCTAACCCCTAACCCCTA"
				+    "ACCCTAACCCTACCCGAACCCTAACCCTAACCCTAACCCCTAACCCTAAC"
				+    "CCCTAAC",  		seqLookup.getRefSeq("2", 101, 307));
		// End Chrom 2
		assertEquals("TGACCCT",			seqLookup.getRefSeq("2", 394, 400));
	}


	@Test
	public void testSeqOutsideRange() throws NumberFormatException, IOException {
		final String MSG_SUFFIX = " could not be retrieved from ref assembly catalog.  This may mean you have the wrong ref catalog assembly for the data you are verifying.";
		
		// 1 bp beyond chrom 1
		assertError("Range [1:192-192]" + MSG_SUFFIX,
				    "1", 192, 192);
		// 1 bp beyond chrom 1 - multi bp
		assertError("Range [1:192-199]" + MSG_SUFFIX,
					"1", 192, 199);
		// 1 bp beyond chrom 1 - multi bp and multiline
		assertError("Range [1:192-500]" + MSG_SUFFIX,
					"1", 192, 500);
		// 1 bp beyond chrom 2
		assertError("Range [1:401-401]" + MSG_SUFFIX,
					"1", 401, 401);
	}

	private void assertError(String expectedErrorMsg, String chrom, long min, long max) {
		try {
			seqLookup.getRefSeq(chrom, min, max);
			fail("FAIL: Expected an exception");
		} catch(Exception e) {
			assertEquals(expectedErrorMsg, e.getMessage());
		}
	}
}
