package edu.mayo.bior.catalog.verification;

import edu.mayo.bior.catalog.CatalogFormatException;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import edu.mayo.bior.catalog.GoldenAttribute;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Categories;
import org.junit.rules.ExpectedException;

public class CatalogEntryGoldenJsonTest
{

   @Rule
   public ExpectedException expectedException = ExpectedException.none();

   private static final String JSON1 = sqToDq("{'_landmark':'13','_minBP':123,'_maxBP':123,'_strand':'+','_refAllele':'A','_altAlleles':['G','T']}");
   private static final String JSON2 = sqToDq("{'_landmark':'13','_minBP':123,'_maxBP':234,'_strand':'+','_refAllele':'A','_altAlleles':['G','T']}");
   private static final String JSON3 = sqToDq("{'_landmark':'13','_minBP':123,'_maxBP':123,'_strand':'+','_refAllele':'T','_altAlleles':['G','C']}");
   private static final String JSON4_SQ = "{'_landmark':'14','_minBP':123,'_maxBP':123,'_strand':'+','_refAllele':'A','_altAlleles':['G','T']}";
   private static final String JSON4 = sqToDq(JSON4_SQ);

   private static final String JSON5 = sqToDq("{'_landmark':'13','_minBP':123,'_maxBP':123,'_strand':'+'}");
   private static final String JSON6 = sqToDq("{'_landmark':'13','_minBP':123,'_maxBP':234,'_strand':'+'}");
   private static final String JSON7 = sqToDq("{'_landmark':'14','_minBP':123,'_maxBP':123,'_strand':'+'}");


   @Test
   public void testBasic() throws CatalogFormatException
   {
      CatalogEntryGoldenJson j1 = new CatalogEntryGoldenJson(JSON2);
      assertEquals("13", j1.getChr());
      assertEquals(123, j1.getMinBP().intValue());
      assertEquals(234, j1.getMaxBP().intValue());
      assertEquals("+", j1.getStrand());
      assertEquals("A", j1.getRefAllele());
      assertEquals(2, j1.getAltAlleles().size());
      assertTrue(j1.getAltAlleles().contains("G"));
      assertTrue(j1.getAltAlleles().contains("T"));
      assertFalse(j1.getAltAlleles().contains("A"));
   }

   @Test
   public void testVariantEquality() throws CatalogFormatException
   {
      CatalogEntryGoldenJson j1 = new CatalogEntryGoldenJson(JSON1);

      CatalogEntryGoldenJson j2 = new CatalogEntryGoldenJson(JSON1);
      assertTrue(j1.equals(j2));

      CatalogEntryGoldenJson j3 = new CatalogEntryGoldenJson(JSON2);
      assertFalse(j1.equals(j3));

      CatalogEntryGoldenJson j4 = new CatalogEntryGoldenJson(JSON3);
      assertFalse(j1.equals(j4));

      CatalogEntryGoldenJson j5 = new CatalogEntryGoldenJson(JSON4);
      assertFalse(j1.equals(j5));

      CatalogEntryGoldenJson j6 = new CatalogEntryGoldenJson(JSON4);
      assertFalse(j1.equals(j6));
      assertTrue(j5.equals(j6));

      String json4DifferentAlts = JSON4_SQ.replace("'G','T'", "'C','G'");
      CatalogEntryGoldenJson j7 = new CatalogEntryGoldenJson(sqToDq(json4DifferentAlts));
      assertFalse(j5.equals(j7));
   }

   @Test
   public void testEmptyEqual() throws CatalogFormatException
   {
      String emptyJson = "{}";
      CatalogEntryGoldenJson j1 = new CatalogEntryGoldenJson(emptyJson);
      CatalogEntryGoldenJson j2 = new CatalogEntryGoldenJson(emptyJson);
      assertTrue(j1.equals(j2));
   }

   @Test
   public void testNonvariantEquality() throws CatalogFormatException
   {

		CatalogEntryGoldenJson j1 = new CatalogEntryGoldenJson(JSON5);

		CatalogEntryGoldenJson j2 = new CatalogEntryGoldenJson(JSON5);
      assertTrue(j1.equals(j2));

		CatalogEntryGoldenJson j3 = new CatalogEntryGoldenJson(JSON6);
      assertFalse(j1.equals(j3));

		CatalogEntryGoldenJson j4 = new CatalogEntryGoldenJson(JSON7);
      assertFalse(j1.equals(j4));
   }

   @Test
   public void testBadLandmarkInteger() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage(GoldenAttribute._landmark.toString());
      expectedException.expectMessage("not a JSON string");
      new CatalogEntryGoldenJson(sqToDq("{'_landmark':12}"));
   }

   @Test
   public void testBadMinBpString() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage(GoldenAttribute._minBP.toString());
      expectedException.expectMessage("not a JSON number");
      new CatalogEntryGoldenJson(sqToDq("{'_minBP':'12'}"));
   }

   @Test
   public void testBadMinBpFloat() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage(GoldenAttribute._minBP.toString());
      expectedException.expectMessage("not an integer");
      new CatalogEntryGoldenJson(sqToDq("{'_minBP':12.2}"));
   }

   @Test
   public void testBadMaxBpString() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage(GoldenAttribute._maxBP.toString());
      expectedException.expectMessage("not a JSON number");
      new CatalogEntryGoldenJson(sqToDq("{'_maxBP':'12'}"));
   }

   @Test
   public void testBadMaxBpFloat() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage(GoldenAttribute._maxBP.toString());
      expectedException.expectMessage("not an integer");
      new CatalogEntryGoldenJson(sqToDq("{'_maxBP':12.2}"));
   }

   @Test
   public void testBadRefInteger() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage(GoldenAttribute._refAllele.toString());
      expectedException.expectMessage("not a JSON string");
      new CatalogEntryGoldenJson(sqToDq("{'_refAllele':12}"));
   }

   @Test
   public void testBadAltInteger() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage(GoldenAttribute._altAlleles.toString());
      expectedException.expectMessage("as a JSON Array");
      new CatalogEntryGoldenJson(sqToDq("{'_altAlleles':12}"));
   }

   @Test
   public void testBadAltString() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage(GoldenAttribute._altAlleles.toString());
      expectedException.expectMessage("as a JSON Array");
      new CatalogEntryGoldenJson(sqToDq("{'_altAlleles':'A,C'}"));
   }

   @Test
   public void testBadAltArrayInteger() throws CatalogFormatException
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage(GoldenAttribute._altAlleles.toString());
      expectedException.expectMessage("not a JSON string");
      expectedException.expectMessage("Item from array");
      new CatalogEntryGoldenJson(sqToDq("{'_altAlleles':[1,2]}"));
   }

   private static String sqToDq(String str)
   {
      return str.replaceAll("'", "\"");
   }
}
