package edu.mayo.bior.catalog.latest;

import edu.mayo.bior.catalog.CatalogFiles;
import edu.mayo.bior.catalog.CatalogFormatException;
import edu.mayo.bior.catalog.DataSourceKey;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

import java.io.File;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.Collections;

import static edu.mayo.bior.catalog.CatalogMetadataConstant.CATALOG_DEPRECATED_SUFFIX;
import static edu.mayo.bior.util.TestUtil.buildEmptyCatalog;
import static edu.mayo.bior.util.TestUtil.mkdir;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

public class LatestCatalogFinderTest {

    @Rule
    public TemporaryFolder tempFolder = new TemporaryFolder();

    @Rule
    public ExpectedException expectedEx = ExpectedException.none();

    private final String FOO = "FOO";
    private final String BAR = "BAR";
    private final String GRCh37 = "GRCh37";
    private final String GRCh38 = "GRCh38";

    @Test
    public void findLatestCatalogHappyPath() throws Exception {
        File rootDir = tempFolder.newFolder("catalog_root_dir");

        // FOO catalog has 3 versions, v3 being the latest
        File fooDir = mkdir(rootDir, FOO);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V1_GRCh37"), FOO, "v1", GRCh37, "v1 desc", "v1 dataset", "2017-01-11", tempFolder);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V1_GRCh38"), FOO, "v1", GRCh38, "v1 desc", "v1 dataset", "2017-01-11", tempFolder);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V2_GRCh37"), FOO, "v2", GRCh37, "v2 desc", "v2 dataset", "2018-01-11", tempFolder);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V2_GRCh38"), FOO, "v2", GRCh38, "v2 desc", "v2 dataset", "2018-01-11", tempFolder);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V3_GRCh37"), FOO, "v3", GRCh37, "v3 desc", "v3 dataset", "2019-01-11", tempFolder);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V3_GRCh38"), FOO, "v3", GRCh38, "v3 desc", "v3 dataset", "2019-01-11", tempFolder);

        LatestCatalogFinder finder = new LatestCatalogFinder();
        LatestCatalog latest = finder.findLatestCatalog(rootDir, FOO, GRCh37);

        assertEquals(FOO, latest.getSource());
        assertEquals("v3", latest.getVersion());
        assertEquals(GRCh37, latest.getBuild());
        assertEquals("v3 desc", latest.getDescription());
        assertEquals("v3 dataset", latest.getDataset());
        assertEquals("2019-01-11", latest.getDataSourceReleaseDate());
        assertTrue(new File(latest.getCatalogPath()).exists());
    }

    @Test
    public void findLatestCatalogOptionalContains() throws Exception {
        File rootDir = tempFolder.newFolder("catalog_root_dir");

        // FOO catalog with same build, same source, same version - different absolute path
        File fooDir = mkdir(rootDir, FOO);
        buildEmptyCatalog(mkdir(fooDir,"FOO.disease.V1_GRCh37"), FOO, "v1", GRCh37, "v1 desc", "v1 dataset", "2017-01-11", tempFolder);
        buildEmptyCatalog(mkdir(fooDir,"FOO.disease_no_spaces.V1_GRCh37"), FOO, "v1 no_spaces", GRCh37, "v1 desc no_spaces", "v1 dataset no_spaces", "2017-01-11", tempFolder);

        LatestCatalogFinder finder = new LatestCatalogFinder();
        LatestCatalog latest = finder.findLatestCatalog(rootDir, FOO, GRCh37, "disease_no_spaces");

        assertEquals(FOO, latest.getSource());
        assertEquals("v1 no_spaces", latest.getVersion());
        assertEquals(GRCh37, latest.getBuild());
        assertEquals("v1 desc no_spaces", latest.getDescription());
        assertEquals("v1 dataset no_spaces", latest.getDataset());
        assertEquals("2017-01-11", latest.getDataSourceReleaseDate());
        assertTrue(latest.getCatalogPath().contains("disease_no_spaces"));
    }

    @Test
    public void findLatestCatalogIgnoreMissingDatasourceProps() throws Exception {
        File rootDir = tempFolder.newFolder("catalog_root_dir");

        // FOO catalog has 3 versions, v3 being the latest
        File fooDir = mkdir(rootDir, FOO);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V1_GRCh37"), FOO, "v1", GRCh37, "v1 desc", "v1 dataset", "2017-01-11", tempFolder);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V2_GRCh37"), FOO, "v2", GRCh37, "v2 desc", "v2 dataset", "2018-01-11", tempFolder);
        File v3CatalogFile = buildEmptyCatalog(mkdir(fooDir,"FOO_V3_GRCh37"), FOO, "v3", GRCh37, "v3 desc", "v3 dataset", "2019-01-11", tempFolder);

        // intentionally delete datasource.properties from v3
        assertTrue(new CatalogFiles(v3CatalogFile).getDataSourceFile().delete());

        LatestCatalogFinder finder = new LatestCatalogFinder();
        LatestCatalog latest = finder.findLatestCatalog(rootDir, FOO, GRCh37);

        assertEquals(FOO, latest.getSource());
        assertEquals("v2", latest.getVersion());
        assertEquals(GRCh37, latest.getBuild());
        assertEquals("v2 desc", latest.getDescription());
        assertEquals("v2 dataset", latest.getDataset());
        assertEquals("2018-01-11", latest.getDataSourceReleaseDate());
        assertTrue(new File(latest.getCatalogPath()).exists());
    }

    @Test
    public void findLatestCatalogWithDeprecated() throws Exception {
        File rootDir = tempFolder.newFolder("catalog_root_dir");

        // FOO catalog has 2 versions AAA and BBB, BBB has a more recent release date, and is deprecated
        File fooDir = mkdir(rootDir, FOO);
        File aaaDir = mkdir(fooDir, "FOO_AAA_GRCh37.v1");
        File bbbDir = mkdir(fooDir, "FOO_BBB_GRCh37.v1");
        buildEmptyCatalog(aaaDir, FOO, "AAA", GRCh37, "AAA desc", "AAA dataset", "2017-01-11", tempFolder);
        File bbbCatalogFile = buildEmptyCatalog(bbbDir, FOO, "BBB", GRCh37, "BBB desc", "BBB dataset", "2018-01-11", tempFolder);

        // deprecate BBB
        CatalogFiles catalogFiles = new CatalogFiles(bbbCatalogFile);
        File deprecateFile = new File(bbbDir, catalogFiles.getPrefix() + CATALOG_DEPRECATED_SUFFIX);
        assertTrue(deprecateFile.createNewFile());

        LatestCatalogFinder finder = new LatestCatalogFinder();
        LatestCatalog latest = finder.findLatestCatalog(rootDir, FOO, GRCh37);

        assertEquals(FOO, latest.getSource());
        assertEquals("AAA", latest.getVersion());
        assertEquals(GRCh37, latest.getBuild());
        assertEquals("AAA desc", latest.getDescription());
        assertEquals("AAA dataset", latest.getDataset());
        assertEquals("2017-01-11", latest.getDataSourceReleaseDate());
        assertTrue(new File(latest.getCatalogPath()).exists());
    }

    @Test
    public void findLatestCatalogBuildStartsWith() throws Exception {
        File rootDir = tempFolder.newFolder("catalog_root_dir");

        File fooDir = mkdir(rootDir, FOO);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V1_GRCh37.p13"), FOO, "v1", "GRCh37.p13", "v1 desc", "v1 dataset", "2017-01-11", tempFolder);

        LatestCatalogFinder finder = new LatestCatalogFinder();
        LatestCatalog latest = finder.findLatestCatalog(rootDir, FOO, GRCh37);

        assertEquals(FOO, latest.getSource());
        assertEquals("v1", latest.getVersion());
        assertEquals("GRCh37.p13", latest.getBuild());
        assertEquals("v1 desc", latest.getDescription());
        assertEquals("v1 dataset", latest.getDataset());
        assertEquals("2017-01-11", latest.getDataSourceReleaseDate());
        assertTrue(new File(latest.getCatalogPath()).exists());
    }

    @Test
    public void findLatestCatalogNoHitsBadSource() throws Exception {
        File rootDir = tempFolder.newFolder("catalog_root_dir");

        File fooDir = mkdir(rootDir, FOO);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V1_GRCh37"), FOO, "v1", GRCh37, "v1 desc", "v1 dataset", "2017-01-11", tempFolder);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V1_GRCh38"), FOO, "v1", GRCh38, "v1 desc", "v1 dataset", "2017-01-11", tempFolder);

        File barDir = mkdir(rootDir, BAR);
        buildEmptyCatalog(mkdir(barDir,"BAR_V1_GRCh37"), BAR, "v1", GRCh37, "v1 desc", "v1 dataset", "2017-01-11", tempFolder);
        buildEmptyCatalog(mkdir(barDir,"BAR_V1_GRCh38"), BAR, "v1", GRCh38, "v1 desc", "v1 dataset", "2017-01-11", tempFolder);

        LatestCatalogFinder finder = new LatestCatalogFinder();

        expectedEx.expect(LatestCatalogNotFoundException.class);
        StringWriter sWtr = new StringWriter();
        PrintWriter  pWtr = new PrintWriter(sWtr);
        pWtr.println("Found 0 catalogs with source=UNKNOWN_SOURCE.  Try one of the following available values:");
        pWtr.println(BAR);
        pWtr.println(FOO);
        expectedEx.expectMessage(sWtr.toString());

        finder.findLatestCatalog(rootDir,"UNKNOWN_SOURCE", GRCh37);
    }

    @Test
    public void findLatestCatalogNoHitsBadBuild() throws Exception {
        File rootDir = tempFolder.newFolder("catalog_root_dir");

        File fooDir = mkdir(rootDir, FOO);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V1_GRCh37"), FOO, "v1", GRCh37, "v1 desc", "v1 dataset", "2017-01-11", tempFolder);
        buildEmptyCatalog(mkdir(fooDir,"FOO_V1_GRCh38"), FOO, "v1", GRCh38, "v1 desc", "v1 dataset", "2017-01-11", tempFolder);

        LatestCatalogFinder finder = new LatestCatalogFinder();

        expectedEx.expect(LatestCatalogNotFoundException.class);
        StringWriter sWtr = new StringWriter();
        PrintWriter  pWtr = new PrintWriter(sWtr);
        pWtr.println("Found 0 catalogs with build=UNKNOWN_BUILD.  Try one of the following available values:");
        pWtr.println(GRCh37);
        pWtr.println(GRCh38);
        expectedEx.expectMessage(sWtr.toString());

        finder.findLatestCatalog(rootDir,FOO, "UNKNOWN_BUILD");
    }

    @Test
    public void findLatestCatalogNoCatalogs() throws Exception {
        File rootDir = tempFolder.newFolder("catalog_root_dir");

        LatestCatalogFinder finder = new LatestCatalogFinder();

        expectedEx.expect(LatestCatalogNotFoundException.class);
        expectedEx.expectMessage(String.format("0 catalogs found under the root directory %s", rootDir.getAbsolutePath()));

        finder.findLatestCatalog(rootDir, FOO, GRCh37);
    }

    @Test
    public void findLatestMissingAllReleaseDates() throws Exception {
        File rootDir = tempFolder.newFolder("catalog_root_dir");

        String blankReleaseDate = "";

        File fooDir = mkdir(rootDir, FOO);
        File barDir = mkdir(rootDir, BAR);
        File fooCatalogFile = buildEmptyCatalog(mkdir(fooDir,"FOO_V1_GRCh37"), FOO, "v1", GRCh37, "v1 desc", "v1 dataset", blankReleaseDate, tempFolder);
        File barCatalogFile = buildEmptyCatalog(mkdir(barDir,"BAR_V1_GRCh37"), BAR, "v1", GRCh37, "v1 desc", "v1 dataset", blankReleaseDate, tempFolder);

        LatestCatalogFinder finder = new LatestCatalogFinder();

        expectedEx.expect(CatalogFormatException.class);
        StringWriter sWtr = new StringWriter();
        PrintWriter  pWtr = new PrintWriter(sWtr);
        pWtr.println("Unable to find the latest catalog because the following catalog datasource.properties are missing dataSourceReleaseDate:");
        pWtr.println(new CatalogFiles((fooCatalogFile)).getDataSourceFile().getAbsolutePath());
        pWtr.println(new CatalogFiles((barCatalogFile)).getDataSourceFile().getAbsolutePath());
        expectedEx.expectMessage(sWtr.toString());

        finder.findLatest(Arrays.asList(fooCatalogFile, barCatalogFile));
    }

    @Test
    public void findLatestMissingPartialReleaseDates() throws Exception {
        File rootDir = tempFolder.newFolder("catalog_root_dir");

        String blankReleaseDate = "";

        File fooDir = mkdir(rootDir, FOO);
        File barDir = mkdir(rootDir, BAR);
        File fooCatalogFile = buildEmptyCatalog(mkdir(fooDir,"FOO_V1_GRCh37"), FOO, "v1", GRCh37, "v1 desc", "v1 dataset", "2017-01-11", tempFolder);
        File barCatalogFile = buildEmptyCatalog(mkdir(barDir,"BAR_V1_GRCh37"), BAR, "v1", GRCh37, "v1 desc", "v1 dataset", blankReleaseDate, tempFolder);

        LatestCatalogFinder finder = new LatestCatalogFinder();

        LatestCatalog latest = finder.findLatest(Arrays.asList(fooCatalogFile, barCatalogFile));
        assertEquals(FOO, latest.getSource());
        assertEquals("v1", latest.getVersion());
        assertEquals(GRCh37, latest.getBuild());
        assertEquals("v1 desc", latest.getDescription());
        assertEquals("v1 dataset", latest.getDataset());
        assertEquals("2017-01-11", latest.getDataSourceReleaseDate());
        assertTrue(new File(latest.getCatalogPath()).exists());
    }

    @Test
    public void findLatestMultipleSameReleaseDates() throws Exception {
        File rootDir = tempFolder.newFolder("catalog_root_dir");

        String sameReleaseDate = "2017-01-11";

        File fooDir = mkdir(rootDir, FOO);
        File barDir = mkdir(rootDir, BAR);
        File fooCatalogFile = buildEmptyCatalog(mkdir(fooDir,"FOO_V1_GRCh37"), FOO, "v1", GRCh37, "v1 desc", "v1 dataset", sameReleaseDate, tempFolder);
        File barCatalogFile = buildEmptyCatalog(mkdir(barDir,"BAR_V1_GRCh37"), BAR, "v1", GRCh37, "v1 desc", "v1 dataset", sameReleaseDate, tempFolder);

        LatestCatalogFinder finder = new LatestCatalogFinder();

        expectedEx.expect(LatestCatalogNotFoundException.class);
        StringWriter sWtr = new StringWriter();
        PrintWriter  pWtr = new PrintWriter(sWtr);
        pWtr.println(String.format("Unable to find the latest catalog because multiple catalog datasource.properties had dataSourceReleaseDate=%s", sameReleaseDate));
        pWtr.println(new CatalogFiles((barCatalogFile)).getDataSourceFile().getAbsolutePath());
        pWtr.println(new CatalogFiles((fooCatalogFile)).getDataSourceFile().getAbsolutePath());
        expectedEx.expectMessage(sWtr.toString());

        finder.findLatest(Arrays.asList(fooCatalogFile, barCatalogFile));
    }

    @Test
    public void findLatestInvalidReleaseDate() throws Exception {
        File rootDir = tempFolder.newFolder("catalog_root_dir");

        String invalidReleaseDate = "NOT_VALID";

        File fooDir = mkdir(rootDir, FOO);
        File fooCatalogFile = buildEmptyCatalog(mkdir(fooDir,"FOO_V1_GRCh37"), FOO, "v1", GRCh37, "v1 desc", "v1 dataset", invalidReleaseDate, tempFolder);

        LatestCatalogFinder finder = new LatestCatalogFinder();

        expectedEx.expect(CatalogFormatException.class);
        String s = String.format("%s value '%s' is incorrectly formatted (yyyy-MM-dd) in file %s",
                DataSourceKey.DataSourceReleaseDate.name(),
                invalidReleaseDate,
                new CatalogFiles((fooCatalogFile)).getDataSourceFile().getAbsolutePath());
        expectedEx.expectMessage(s);

        finder.findLatest(Collections.singletonList(fooCatalogFile));
    }
}
