package edu.mayo.bior.catalog.index;

import static org.junit.Assert.assertEquals;

import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import org.apache.commons.io.FileUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import edu.mayo.pipes.util.index.H2Connection;


public class IndexDatabaseCreatorTest {

	@Rule
	public TemporaryFolder mTempFolder = new TemporaryFolder();
	
	protected File mTempDir  = null;
	
	protected File 	 mGenesCatalog = null;
	protected final String GENES_CATALOG__ORIGINAL   = "src/test/resources/genes.tsv.bgz";
	protected final String GENES_CATALOG__GENE_INDEX = "genes.gene.idx.h2.db";
	protected final String GENES_CATALOG__GENE_KEY   = "gene";
	protected final String GENES_CATALOG__HGNC_INDEX = "genes.HGNC.idx.h2.db";
	protected final String GENES_CATALOG__HGNC_KEY   = "HGNC";
	
	protected File   mJsonArrayCatalog = null;
	protected final String JSON_ARRAY_CATALOG__GENES_INDEX = "catalogWithJsonArray.Genes.idx.h2.db";
	protected final String JSON_ARRAY_CATALOG__GENES_KEY   = "Genes";

	@Before
	public void createTempAndIndexFolders() throws IOException {
		mTempDir  = mTempFolder.newFolder();
		File indexDir = new File(mTempDir, "index");
		indexDir.mkdirs();
		
		// NOTE: Even though these indexes already exist in the "src/test/resources" directory, we want to 
		//   create them in the temp directory so that they do not force an update each time we check in code
		
		// Copy the genes catalog to the temp folder
		FileUtils.copyFileToDirectory(new File(GENES_CATALOG__ORIGINAL), mTempDir);
		mGenesCatalog = new File(mTempDir, new File(GENES_CATALOG__ORIGINAL).getName());
		

		// Copy the catalogWithJsonArray catalog to the temp folder
		FileUtils.copyFileToDirectory(new File("src/test/resources/catalogWithJsonArray.tsv.bgz"), mTempDir);
		mJsonArrayCatalog = new File(mTempDir, "catalogWithJsonArray.tsv.bgz");
	}
	
	
	/** Create an index from the genes file based on gene name/symbol
	 * @throws ClassNotFoundException 
	 * @throws IOException 
	 * @throws SQLException */
	@Test
	public void createGeneNameIndex() throws SQLException, IOException, ClassNotFoundException {
		File indexDbFile = createIndex(
				mGenesCatalog, 
				GENES_CATALOG__GENE_INDEX, 
				GENES_CATALOG__GENE_KEY );
		
		long rowCount = getRowCount(indexDbFile.getCanonicalPath(), "Indexer");
		assertEquals(37301, rowCount); 
		System.out.println("\n----------------------\n");
	}

	
	
	/** Create an index on Genes JSON object, which is actually an ARRAY
	 * @throws ClassNotFoundException 
	 * @throws IOException 
	 * @throws SQLException */
	@Test
	public void createGeneNameIndex_jsonArray() throws SQLException, IOException, ClassNotFoundException {
		File indexDbFile = createIndex(
				mJsonArrayCatalog,
				JSON_ARRAY_CATALOG__GENES_INDEX,
				JSON_ARRAY_CATALOG__GENES_KEY );

		String actual = IndexDatabaseCreator.getTableAsString(indexDbFile);
		String expected = "	KEY	FILEPOS\n"
				+		  "1)	GeneA	20\n"
				+ 		  "2)	GeneB	20\n"
				+		  "3)	GeneC	20\n"
				+ 		  "4)	GeneD	78\n"
				+ 		  "5)	GeneE	78\n"
				+ 		  "6)	GeneF	78\n";
		assertEquals(expected, actual);
		
		long rowCount = getRowCount(indexDbFile.getCanonicalPath(), "Indexer");
		assertEquals(6, rowCount); 
		
		System.out.println("\n----------------------\n");
	}
	
	
	/** Create an index from the genes file based on HGNC id
	 * @throws ClassNotFoundException 
	 * @throws IOException 
	 * @throws SQLException */
	@Test
	public void createGenesIndex() throws SQLException, IOException, ClassNotFoundException {
		File indexDbFile = createIndex(
				mGenesCatalog,
				GENES_CATALOG__HGNC_INDEX,
				GENES_CATALOG__HGNC_KEY );
		
		long rowCount = getRowCount(indexDbFile.getCanonicalPath(), "Indexer");
		assertEquals(30628, rowCount); 
		System.out.println("\n----------------------\n");

	}

	//============================================================================================
	// Helper methods
	//============================================================================================
	
	private File createIndex( File catalog, String newIndexDbFilename, String key) throws IOException, SQLException, ClassNotFoundException {
		// Create an index based on the key name
		File newIndexDbFile = new File(mTempDir, "index/" + newIndexDbFilename);
		System.out.println("Creating index at: " + newIndexDbFile.getCanonicalPath());
		IndexDatabaseCreator indexH2 = new IndexDatabaseCreator();
		final int KEY_COL = 4;
		indexH2.buildIndexH2(catalog.getCanonicalPath(), KEY_COL, key, newIndexDbFile.getCanonicalPath(), true);
		return newIndexDbFile;
	}
	
	private long getRowCount(String h2DbPath, String tableName) throws SQLException, IOException {
		H2Connection h2Conn = new H2Connection(h2DbPath, /*isWritable=*/false);
		Connection dbConn = h2Conn.getConn();
		Statement stmt = dbConn.createStatement();
		ResultSet rs = stmt.executeQuery("SELECT COUNT(*) FROM " + tableName);
		rs.next();
		long numRows = rs.getInt(1);
		rs.close();
		stmt.close();
		dbConn.close();
		return numRows;
	}


}
