package edu.mayo.bior.catalog;

import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import java.util.ArrayList;
import java.util.List;

import static junit.framework.Assert.assertNull;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

/**
 * Test CatalogTabixEntry class
 */
public class CatalogTabixEntryTest
{
   public static final String UNKNOWN = "UNKNOWN";
   private static final String ZERO = "0";
   private static final String ONE = "1";
   private static final String TWO = "2";
   private static final String THREE = "3";
   private static final String EMPTY_JSON = "{}";
   private List<String> fields = new ArrayList<String>();

   @Rule
   public ExpectedException expectedEx = ExpectedException.none();

   @Before
   public void setUp()
   {
      fields.add(ONE);
      fields.add(TWO);
      fields.add(THREE);
      fields.add(EMPTY_JSON);
   }
   
   @Test
   public void testBasic() throws Exception
   {
      CatalogTabixEntry entry = new CatalogTabixEntry(StringUtils.join(fields, "\t"));
      assertEquals(ONE, entry.getChromosome());
      assertEquals(new Long(TWO), entry.getMinPosition());
      assertEquals(new Long(THREE), entry.getMaxPosition());
      assertEquals(EMPTY_JSON, entry.getJsonString());
      assertTrue(entry.is4Field());
      assertFalse(entry.is1Field());
      assertTrue(entry.isChromosomePositional());
      assertTrue(entry.isMinBPPositional());
      assertTrue(entry.isMaxBPPositional());
      assertTrue(entry.isPositional());
      assertFalse(entry.isPositionalInfoNull());
      assertFalse(entry.isUnknownPosition());

      entry = new CatalogTabixEntry(EMPTY_JSON);
      assertEquals(EMPTY_JSON, entry.getJsonString());
      assertNull(entry.getChromosome());
      assertNull(entry.getMinPosition());
      assertNull(entry.getMaxPosition());
      assertTrue(entry.is1Field());
      assertFalse(entry.is4Field());
      assertFalse(entry.isChromosomePositional());
      assertFalse(entry.isMinBPPositional());
      assertFalse(entry.isMaxBPPositional());
      assertFalse(entry.isPositional());
      assertTrue(entry.isPositionalInfoNull());
      assertFalse(entry.isUnknownPosition());

      fields.set(0, UNKNOWN);
      fields.set(1, ZERO);
      fields.set(2, ONE);
      fields.set(3, EMPTY_JSON);
      entry = new CatalogTabixEntry(StringUtils.join(fields, "\t"));
      assertEquals(UNKNOWN, entry.getChromosome());
      assertEquals(new Long(ZERO), entry.getMinPosition());
      assertEquals(new Long(ONE), entry.getMaxPosition());
      assertEquals(EMPTY_JSON, entry.getJsonString());
      assertTrue(entry.is4Field());
      assertFalse(entry.is1Field());
      assertFalse(entry.isChromosomePositional());
      assertFalse(entry.isMinBPPositional());
      assertTrue(entry.isMaxBPPositional());
      assertFalse(entry.isPositional());
      assertFalse(entry.isPositionalInfoNull());
      assertTrue(entry.isUnknownPosition());
   }

   @Test
   public void testBadNumFields2() throws CatalogFormatException
   {
      fields.remove(1);
      fields.remove(2);

      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("It has 2 fields");

      new CatalogTabixEntry(StringUtils.join(fields, "\t"));
   }

   @Test
   public void testBadNumFields5() throws CatalogFormatException
   {
      fields.add(ONE);

      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("It has 5 fields");

      new CatalogTabixEntry(StringUtils.join(fields, "\t"));
   }

   @Test
   public void testBadJsonField1() throws CatalogFormatException
   {
      fields.clear();
      fields.add("{");

      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("1st field");
      expectedEx.expectMessage("doesn't look like JSON");

      new CatalogTabixEntry(StringUtils.join(fields, "\t"));
   }

   @Test
   public void testBadJsonField4() throws CatalogFormatException
   {
      fields.remove(3);
      fields.add("}");

      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("4th field");
      expectedEx.expectMessage("doesn't look like JSON");

      new CatalogTabixEntry(StringUtils.join(fields, "\t"));
   }

   @Test
   public void testNullLine() throws CatalogFormatException
   {
      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("Null line given");

      new CatalogTabixEntry(null);
   }

   @Test
   public void testBadMin() throws CatalogFormatException
   {
      fields.set(1, "NotAnInteger");

      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("min position");
      expectedEx.expectMessage("not an integer");
      expectedEx.expectMessage("field 2");

      new CatalogTabixEntry(StringUtils.join(fields, "\t"));
   }
   @Test
   public void testBadMax() throws CatalogFormatException
   {
      fields.set(2, "");

      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("max position");
      expectedEx.expectMessage("not an integer");
      expectedEx.expectMessage("field 3");

      new CatalogTabixEntry(StringUtils.join(fields, "\t"));
   }
}
