package edu.mayo.bior.catalog;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

import java.io.File;
import java.io.IOException;

import static edu.mayo.bior.catalog.CatalogMetadataConstant.CATALOG_DEPRECATED_SUFFIX;
import static edu.mayo.bior.util.TestUtil.buildEmptyCatalog;
import static edu.mayo.bior.util.TestUtil.mkdir;
import static org.junit.Assert.*;

/**
 * Test CatalogFiles class
 */
public class CatalogFilesTest
{
   @Rule
   public ExpectedException expectedEx = ExpectedException.none();

   @Rule
   public TemporaryFolder temporaryFolder = new TemporaryFolder();

   @Test
   public void testBasic() throws CatalogFormatException, IOException
   {
      String prefix = "catalog";
      File catalogFile = temporaryFolder.newFile(prefix + CatalogMetadataConstant.CATALOG_FILE_SUFFIX);
      CatalogFiles catalog = new CatalogFiles(catalogFile.getPath());
      assertEquals(prefix, catalog.getPrefix());
      assertEquals(catalog.getDir(), temporaryFolder.getRoot());
      assertNull(catalog.getColumnsFile());
      assertNull(catalog.getDataSourceFile());
      assertNull(catalog.getBlacklistFile());
      assertNull(catalog.getTabixIndexFile());
      assertNull(catalog.getBlacklistBiorwebFile());
   }

   @Test
   public void testBasicRelativePath() throws CatalogFormatException, IOException
   {
      String prefix = "catalognamethatcouldntpossiblyexist";
      File catalogFile = new File(prefix + CatalogMetadataConstant.CATALOG_FILE_SUFFIX);
      if (!catalogFile.createNewFile())
      {
         fail(String.format("Couldn't create a file that starts with '%s'", prefix));
      }
      CatalogFiles catalog = new CatalogFiles(catalogFile);
      assertEquals(new File(System.getProperty("user.dir")), catalog.getDir());
      if (!catalogFile.delete())
      {
         fail(String.format("Couldn't delete a file that starts with '%s'", prefix));
      }
   }

   @Test
   public void testCatalogNotExist() throws CatalogFormatException
   {
      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("does not exist");
      new CatalogFiles(temporaryFolder.getRoot().getPath() + File.separator + "notexists");
   }

   @Test
   public void testCatalogNotFile() throws CatalogFormatException
   {
      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("is not a file");
      new CatalogFiles(temporaryFolder.getRoot().getPath());
   }

   @Test
   public void testCatalogNotReadable() throws CatalogFormatException, IOException
   {
      String prefix = "catalog";
      File catalogFile = temporaryFolder.newFile(prefix + CatalogMetadataConstant.CATALOG_FILE_SUFFIX);
      if (!catalogFile.setReadable(false))
      {
         fail(String.format("Couldn't set %s to readable to test it", catalogFile.getPath()));
      }
      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("Can not read");
      new CatalogFiles(catalogFile);
   }

   @Test
   public void testCatalogBogusSuffix() throws CatalogFormatException, IOException
   {
      String prefix = "catalog";
      File catalogFile = temporaryFolder.newFile(prefix + ".tsv");
      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("does not end");
      new CatalogFiles(catalogFile);

   }

   @Test
   public void testNoPrefix() throws CatalogFormatException, IOException
   {
      String prefix = "";
      File catalogFile = temporaryFolder.newFile(prefix + CatalogMetadataConstant.CATALOG_FILE_SUFFIX);
      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("not extract catalog prefix");
      new CatalogFiles(catalogFile);
   }

   @Test
   public void testColumnsNotFile() throws CatalogFormatException, IOException
   {
      String prefix = "catalog";
      File catalogFile = temporaryFolder.newFile(prefix + CatalogMetadataConstant.CATALOG_FILE_SUFFIX);
      temporaryFolder.newFolder(prefix + CatalogMetadataConstant.COLUMN_INFO_SUFFIX);
      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("columns");
      expectedEx.expectMessage("not a file");
      new CatalogFiles(catalogFile.getPath());
   }

   @Test
   public void testColumnsNotReadable() throws CatalogFormatException, IOException
   {
      String prefix = "catalog";
      File catalogFile = temporaryFolder.newFile(prefix + CatalogMetadataConstant.CATALOG_FILE_SUFFIX);
      File columnsFile = temporaryFolder.newFile(prefix + CatalogMetadataConstant.COLUMN_INFO_SUFFIX);
      if (!columnsFile.setReadable(false))
      {
         fail(String.format("Couldn't set %s to readable to test it", columnsFile.getPath()));
      }
      expectedEx.expect(CatalogFormatException.class);
      expectedEx.expectMessage("columns");
      expectedEx.expectMessage("Can not read");
      new CatalogFiles(catalogFile.getPath());
   }

   @Test
   public void testColumnsFile() throws CatalogFormatException, IOException
   {
      String prefix = "catalog";
      File catalogFile = temporaryFolder.newFile(prefix + CatalogMetadataConstant.CATALOG_FILE_SUFFIX);
      File columnsFile = temporaryFolder.newFile(prefix + CatalogMetadataConstant.COLUMN_INFO_SUFFIX);
      CatalogFiles catalog = new CatalogFiles(catalogFile.getPath());
      assertEquals(columnsFile, catalog.getColumnsFile());
   }

   @Test
   public void testDeprecatedCatalog() throws Exception {
      File rootDir = temporaryFolder.newFolder();

      File bbbDir = mkdir(rootDir, "FOO_BBB");
      File bbbCatalogFile = buildEmptyCatalog(bbbDir, "FOO", "BBB", "GRCh37", "BBB desc", "BBB dataset", "2018-01-11", temporaryFolder);

      // deprecate BBB
      CatalogFiles catalogFiles = new CatalogFiles(bbbCatalogFile);
      File deprecateFile = new File(bbbDir, catalogFiles.getPrefix() + CATALOG_DEPRECATED_SUFFIX);
      assertTrue(deprecateFile.createNewFile());

      assertTrue(new CatalogFiles(bbbCatalogFile).isDeprecated());
   }
}
