package edu.mayo.bior.catalog;

import edu.mayo.bior.catalog.latest.LatestCatalogFinder;
import org.apache.commons.io.FileUtils;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

import java.io.File;
import java.io.IOException;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import static edu.mayo.bior.util.TestUtil.buildEmptyCatalog;
import static edu.mayo.bior.util.TestUtil.mkdir;
import static junit.framework.Assert.*;

/**
 * Test CatalogDataSource
 */
public class CatalogDataSourceTest
{
   @Rule
   public ExpectedException expectedException = ExpectedException.none();

   @Rule
   public TemporaryFolder temporaryFolder = new TemporaryFolder();

   @Test
   public void testDataSourceFormatMissing() throws Exception
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("Must supply Format property");
      File file = new File("src/test/resources/testData/format_missing.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNull(source);
   }

   @Test
   public void testDataSourceFormatEmptyString() throws Exception
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("Format value '' from");
      expectedException.expectMessage("not in supported set");
      File file = new File("src/test/resources/testData/format_empty.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNull(source);
   }

   @Test
   public void testDataSourceFormatBadValue() throws Exception
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("Format value '2.0' from");
      expectedException.expectMessage("not in supported set");
      File file = new File("src/test/resources/testData/format_wrong.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNull(source);
   }

   @Test
   public void testDataSource110() throws Exception
   {
      File file = new File("src/test/resources/testData/format_good_1.1.0.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNotNull(source);
      assertTrue(source.is110Format());
      assertFalse(source.is111Format());
      assertEquals("shortname", source.getShortUniqueName());
      assertEquals("our description", source.getDescription());
      assertEquals("my imagination", source.getSource());
      assertEquals("", source.getDataset());
      assertEquals("bozo", source.getVersion());
      assertEquals("none", source.getBuild());
   }

   @Test
   public void testDataSource111() throws Exception
   {
      File file = new File("src/test/resources/testData/format_good_1.1.1.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNotNull(source);
      assertFalse(source.is110Format());
      assertTrue(source.is111Format());
      assertEquals("shortname", source.getShortUniqueName());
      assertEquals("our description", source.getDescription());
      assertEquals("my imagination", source.getSource());
      assertEquals("vars", source.getDataset());
      assertEquals("bozo", source.getVersion());
      assertEquals("none", source.getBuild());
   }

   @Test
   public void testDataSource111ExtraProperties() throws Exception
   {
      File file = new File("src/test/resources/testData/format_good_1.1.1_extra_fields_including_source_release_date.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNotNull(source);
      assertFalse(source.is110Format());
      assertTrue(source.is111Format());
      assertEquals("shortname", source.getShortUniqueName());
      assertEquals("our description", source.getDescription());
      assertEquals("my imagination", source.getSource());
      assertEquals("vars", source.getDataset());
      assertEquals("bozo", source.getVersion());
      assertEquals("none", source.getBuild());
      assertEquals("2018-06-01", source.getDataSourceReleaseDate());
   }

   @Test
   public void testDataSource112() throws Exception
   {
      File file = new File("src/test/resources/testData/format_good_1.1.2.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNotNull(source);
      assertFalse(source.is110Format());
      assertFalse(source.is111Format());
      assertTrue(source.is112Format());
      assertEquals("shortname", source.getShortUniqueName());
      assertEquals("our description", source.getDescription());
      assertEquals("my imagination", source.getSource());
      assertEquals("vars", source.getDataset());
      assertEquals("bozo", source.getVersion());
      assertEquals("none", source.getBuild());
      assertEquals("2018-06-01", source.getDataSourceReleaseDate());
   }

   @Test
   public void testDataSource112ReleaseDateEmpty() throws Exception
   {
      File file = new File("src/test/resources/testData/empty_source_release_date_1.1.2.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNotNull(source);
      assertFalse(source.is110Format());
      assertFalse(source.is111Format());
      assertTrue(source.is112Format());
      assertEquals("shortname", source.getShortUniqueName());
      assertEquals("our description", source.getDescription());
      assertEquals("my imagination", source.getSource());
      assertEquals("vars", source.getDataset());
      assertEquals("bozo", source.getVersion());
      assertEquals("none", source.getBuild());
      assertEquals("", source.getDataSourceReleaseDate());
   }

   @Test
   public void testDataSourceMissingProps110() throws Exception
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("Required datasource properties");
      expectedException.expectMessage("not supplied in");
      expectedException.expectMessage("properties 'ShortUniqueName,Source'");
      File file = new File("src/test/resources/testData/missing_props_1.1.0.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNull(source);
   }

   @Test
   public void testDataSourceMissingProps111() throws Exception
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("Required datasource properties");
      expectedException.expectMessage("not supplied in");
      expectedException.expectMessage("properties 'Dataset,Source'");
      File file = new File("src/test/resources/testData/missing_props_1.1.1.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNull(source);
   }

   @Test
   public void testDataSourceMissingProps112() throws Exception
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("Required datasource properties");
      expectedException.expectMessage("not supplied in");
      expectedException.expectMessage("properties 'DataSourceReleaseDate,Dataset,Source'");
      File file = new File("src/test/resources/testData/missing_props_1.1.2.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNull(source);
   }

   @Test
   public void testNullFile() throws Exception
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("Supplied null");
      new CatalogDataSource(null);
   }

   @Test
   public void testNonExistentFile() throws Exception
   {
      String filename = "file_that_couldnt_possibly_exist";
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("does not exist");
      expectedException.expectMessage(filename);
      new CatalogDataSource(new File(filename));
   }

   @Test
   public void testNonReadableFile() throws Exception
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("is not readable");
      File file = temporaryFolder.newFile();
      if (!file.setReadable(false))
      {
         throw new RuntimeException("Problem setting file to readable");
      }
      new CatalogDataSource(file);
   }

   @Test
   public void testDirectory() throws Exception
   {
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("is not a file");
      File dir = temporaryFolder.newFolder();
      new CatalogDataSource(dir);

   }

   @Test
   public void testGrch38() throws Exception
   {
      File file = new File("src/test/resources/testData/grch38.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertTrue(source.isHumanCatalog());
      assertEquals(HumanBuildAssembly.GRCh38, source.getHumanBuildAssembly());
   }

   @Test
   public void testGrch37() throws Exception
   {
      File file = new File("src/test/resources/testData/grch37.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertTrue(source.isHumanCatalog());
      assertEquals(HumanBuildAssembly.GRCh37, source.getHumanBuildAssembly());
   }

   @Test
   public void testNoBuild() throws Exception
   {
      File file = new File("src/test/resources/testData/nobuild.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertFalse(source.isHumanCatalog());
      assertEquals(HumanBuildAssembly.NA, source.getHumanBuildAssembly());
   }

   @Test
   public void testPropertiesWithBlankValues() throws Exception
   {
      File file = new File("src/test/resources/testData/grch37.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      List<String> propsWithBlank = source.getPropertiesWithBlankValue();
      assertEquals(5, propsWithBlank.size());
      assertTrue(propsWithBlank.contains("Dataset"));
      assertTrue(propsWithBlank.contains("Description"));
      assertTrue(propsWithBlank.contains("ShortUniqueName"));
      assertTrue(propsWithBlank.contains("Source"));
      assertTrue(propsWithBlank.contains("Version"));
   }

   @Test
   public void testPropertiesLowercase() throws Exception
   {
      File file = new File("src/test/resources/testData/format_good_1.1.1_lowercase.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNotNull(source);
      assertFalse(source.is110Format());
      assertTrue(source.is111Format());
      assertEquals("shortname", source.getShortUniqueName());
      assertEquals("our description", source.getDescription());
      assertEquals("my imagination", source.getSource());
      assertEquals("vars", source.getDataset());
      assertEquals("bozo", source.getVersion());
      assertEquals("none", source.getBuild());
   }

   @Test
   public void testDescriptionSeenMultipleTimes() throws Exception
   {
      File file = new File("src/test/resources/testData/format_good_1.1.1_repeated_Description.datasource.properties");
      CatalogDataSource source = new CatalogDataSource(file);
      assertNotNull(source);
      assertFalse(source.is110Format());
      assertTrue(source.is111Format());
      // Make sure it gets the second value of Description from the file
      assertEquals("second description", source.getDescription());

      assertEquals("shortname", source.getShortUniqueName());
      assertEquals("my imagination", source.getSource());
      assertEquals("vars", source.getDataset());
      assertEquals("bozo", source.getVersion());
      assertEquals("none", source.getBuild());
   }

   @Test
   public void testUpperLowerCaseDescription() throws Exception
   {
      File file = new File("src/test/resources/testData/format_good_1.1.1_lower_upper_description.datasource.properties");
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("had multiple instances of key 'Description'");
      new CatalogDataSource(file);
   }

   @Test
   public void testUpperLowerCaseDescriptionFormat() throws Exception
   {
      File file = new File("src/test/resources/testData/format_good_1.1.1_lower_upper_description_format.datasource.properties");
      expectedException.expect(CatalogFormatException.class);
      expectedException.expectMessage("had multiple instances of keys 'Description,Format'");
      new CatalogDataSource(file);
   }
   
   
   @Test
   // Test double-quotes within values are flagged as errors, but the data is still loaded and the double-quotes converted to single-quotes */
   public void testValuesWithDoubleQuotes_convertToSingleQuotes() throws Exception
   {
	   // Copy the datasource.properties to temp dir
	   File from = new File("src/test/resources/testData/format_good_1.1.1.datasource.properties");
	   temporaryFolder.create();
	   File dataSrcProps = temporaryFolder.newFile("format_good_1.1.1.datasource.properties");
	   FileUtils.copyFile(from, dataSrcProps);
	   
	   // Change the "Description" line to contain double-quotes
	   String contents = FileUtils.readFileToString(dataSrcProps);
	   contents = contents.replace("Description=our description",  "Description=Our \"description\" for our \"catalog\"");
	   FileUtils.write(dataSrcProps, contents);
	   
	   CatalogDataSource source = new CatalogDataSource(dataSrcProps);
	   
	   assertEquals("Our 'description' for our 'catalog'", source.getDescription());
   }

   @Test
   public void testGetDataSourceReleaseDateAsDateObject() throws Exception {

      String dataSourceReleaseDate = "2018-06-20";

      File fooCatalogFile = buildEmptyCatalog(temporaryFolder.newFolder(), "FOO", "v1", "GRCh37", "v1 desc", "v1 dataset", dataSourceReleaseDate, temporaryFolder);

      CatalogDataSource dataSrc = new CatalogDataSource(new CatalogFiles(fooCatalogFile).getDataSourceFile());

      Calendar cal = Calendar.getInstance();
      cal.setTime(dataSrc.getDataSourceReleaseDateAsDateObject());

      assertEquals(2018, cal.get(Calendar.YEAR));
      assertEquals(Calendar.JUNE, cal.get(Calendar.MONTH));
      assertEquals(20, cal.get(Calendar.DAY_OF_MONTH));
   }

   @Test
   public void testGetDataSourceReleaseDateAsDateObjectError() throws Exception {

      String invalidReleaseDate = "06-20-2018";

      File fooCatalogFile = buildEmptyCatalog(temporaryFolder.newFolder(), "FOO", "v1", "GRCh37", "v1 desc", "v1 dataset", invalidReleaseDate, temporaryFolder);

      CatalogDataSource dataSrc = new CatalogDataSource(new CatalogFiles(fooCatalogFile).getDataSourceFile());

      expectedException.expect(CatalogFormatException.class);
      String s = String.format("%s value '%s' is incorrectly formatted (yyyy-MM-dd) in file %s",
              DataSourceKey.DataSourceReleaseDate.name(),
              invalidReleaseDate,
              new CatalogFiles((fooCatalogFile)).getDataSourceFile().getAbsolutePath());
      expectedException.expectMessage(s);

      dataSrc.getDataSourceReleaseDateAsDateObject();
   }
}
