package edu.mayo.bior.buildcatalog;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.base.Joiner;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.junit.Rule;
import org.junit.rules.TemporaryFolder;

import static org.junit.Assert.*;

import org.junit.rules.ExpectedException;

public class BuildInfoTest
{
   private Map<String, String> buildInfoMap;

   @Rule
   public ExpectedException expectedEx = ExpectedException.none();

   @Rule
   public TemporaryFolder tempFolder = new TemporaryFolder();

   private static final String DEFAULT_MAKE_JSON_SCRIPT_PATH = "src/test/resources/buildCatalog/makeJsonStep.sh";
   private static final String DEFAULT_MAKE_JSON_ARGS = "--inputFile inputFile  --column 4";
   private static final String DEFAULT_DATA_SOURCE_BUILD = "GRCh37";
   private static final String DEFAULT_DATA_SOURCE = "x";
   private static final String DEFAULT_CATALOG_PREFIX = "xy";
   private static final String DEFAULT_DATA_SOURCE_VERSION = "2b";
   private static final String DEFAULT_DATA_SOURCE_RELEASE_DATE = "2018-07-13";
   private static final String EMPTY = "";



   @Before
   public void setUp() throws IOException
   {
      buildInfoMap = new HashMap<String, String>();
      File compileScript = new File(DEFAULT_MAKE_JSON_SCRIPT_PATH);
      buildInfoMap.put(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name(), compileScript.getCanonicalPath());
      buildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), DEFAULT_MAKE_JSON_ARGS);
      buildInfoMap.put(BuildInfoKey.DATA_SOURCE_BUILD.name(), DEFAULT_DATA_SOURCE_BUILD);
      buildInfoMap.put(BuildInfoKey.DATA_SOURCE.name(), DEFAULT_DATA_SOURCE);
      buildInfoMap.put(BuildInfoKey.CATALOG_PREFIX.name(), DEFAULT_CATALOG_PREFIX);
      buildInfoMap.put(BuildInfoKey.DATA_SOURCE_VERSION.name(), DEFAULT_DATA_SOURCE_VERSION);
      buildInfoMap.put(BuildInfoKey.DATA_SOURCE_RELEASE_DATE.name(), DEFAULT_DATA_SOURCE_RELEASE_DATE);
   }

   @Test
   public void testRequiredKeys() throws IOException, BuildCatalogStepInputException
   {
      File tempDir = tempFolder.newFolder();
      File buildInfoFile = BuildCatalogTestUtils.substituteInITCaseBuildInfo(tempDir, tempDir);
      new BuildInfo(buildInfoFile.getPath());
   }

   @Test
   public void testRequiredKeysErrMessage() throws IOException, BuildCatalogStepInputException
   {
      String expectedErrList = Joiner.on(", ").join(BuildInfoKey.getRequiredKeys());
      String buildInfoFile = "src/test/resources/buildCatalog/build_info_no_keys.txt";

      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage(expectedErrList);
      expectedEx.expectMessage("keys: ");
      expectedEx.expectMessage("file '" + buildInfoFile + "'");

      new BuildInfo(buildInfoFile);
   }

   @Test
   public void testRequiredKeysErrMessageWithBuildInfoObj() throws BuildCatalogStepInputException
   {
      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("is missing");
      expectedEx.expectMessage(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name());
      expectedEx.expectMessage("data supplied to create BuildInfo object");

      buildInfoMap.remove(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testRequiredNonEmptyValuesKeysErrMessage() throws IOException, BuildCatalogStepInputException
   {
      String expectedErrList = Joiner.on(", ").join(BuildInfoKey.getNonEmptyValueRequiredKeys());
      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("keys: ");
      expectedEx.expectMessage(expectedErrList);
      new BuildInfo("src/test/resources/buildCatalog/build_info_no_values.txt");
   }

   @Test
   public void testMissingOneNonEmptyValuesKeysErrMessage() throws IOException, BuildCatalogStepInputException
   {
      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("key: ");
      expectedEx.expectMessage(BuildInfoKey.CATALOG_PREFIX.name());
      new BuildInfo("src/test/resources/buildCatalog/build_info_missing_one_required_value.txt");
   }
   
   @Test
   /** Test a BuildInfo key that is not one of the recognized BuildInfoKey enum values */
   public void testUnrecognizedKey() throws IOException, BuildCatalogStepInputException {
	   // The makeJson script must exist and must be executable
	   File makeJsonScript = tempFolder.newFile("makeJson.sh");
	   makeJsonScript.setExecutable(true);
	   
	   File buildInfoFile = tempFolder.newFile("build_info.txt");
	   // Add all required ones, plus an extra one that is not in the enum
	   String buildInfoContents = BuildInfoKey.CATALOG_PREFIX 		+ "=ctgPrefix\n"
			   					+ BuildInfoKey.DATA_SOURCE    		+ "=dbSNP\n"
			   					+ BuildInfoKey.DATA_SOURCE_BUILD	+ "=GRCh37\n"
			   					+ BuildInfoKey.DATA_SOURCE_VERSION	+ "=142\n"
			   					+ BuildInfoKey.DATA_SOURCE_RELEASE_DATE	+ "=2018-07-11\n"
			   					+ BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH + "=myCtg.json\n"
			   					+ BuildInfoKey.MAKE_JSON_SCRIPT_PATH + "=" + makeJsonScript.getCanonicalPath() + "\n"
			   					+ BuildInfoKey.MAKE_JSON_ARGS 		+ "=--flag\n"
			   					+ BuildInfoKey.TARGET_DIR 			+ "=" + tempFolder.newFolder().getCanonicalPath() + "\n"
			   					+ "SomeUnrecognizedKey=val\n";
	   FileUtils.write(buildInfoFile, buildInfoContents);
	   
	   // Redirect System.err to a pipe
	   PrintStream oldSysErr = System.err;
	   ByteArrayOutputStream bOut = new ByteArrayOutputStream();
	   PrintStream tempSysErr = new PrintStream( bOut );
	   System.setErr(tempSysErr);
	   BuildInfo buildInfo = new BuildInfo(buildInfoFile.getCanonicalPath());
	   System.setErr(oldSysErr);
	   assertEquals("Warning: Unrecognized key in build_info.txt file: SomeUnrecognizedKey\n", bOut.toString());
   }
   

   @Test
   public void testBuildInfoValues() throws BuildCatalogStepInputException
   {
      BuildInfo buildInfo = new BuildInfo(buildInfoMap);
      assertTrue(buildInfo.getMakeJsonScriptPath().contains(DEFAULT_MAKE_JSON_SCRIPT_PATH));
      assertTrue(buildInfo.getMakeJsonScriptPath().contains(DEFAULT_MAKE_JSON_SCRIPT_PATH));
      assertEquals(buildInfo.getMakeJsonArgs(), DEFAULT_MAKE_JSON_ARGS);
      assertEquals(buildInfo.getDataSourceBuild(), DEFAULT_DATA_SOURCE_BUILD);
      assertEquals(buildInfo.getDataSourceName(), DEFAULT_DATA_SOURCE);
      assertEquals(buildInfo.getCatalogPrefix(), DEFAULT_CATALOG_PREFIX);
      assertEquals(buildInfo.getDataSourceVersion(), DEFAULT_DATA_SOURCE_VERSION);
      assertEquals(buildInfo.getDataSourceReleaseDate(), DEFAULT_DATA_SOURCE_RELEASE_DATE);
      assertNull(buildInfo.getPreviousCatalogPath());

      // These are the values it is ok to set to ""
      buildInfoMap.put(BuildInfoKey.MAKE_JSON_ARGS.name(), EMPTY);
      buildInfoMap.put(BuildInfoKey.DATA_SOURCE_BUILD.name(), EMPTY);
      buildInfo = new BuildInfo(buildInfoMap);
      assertEquals(EMPTY, buildInfo.getMakeJsonArgs());
      assertEquals(EMPTY, buildInfo.getDataSourceBuild());
   }

   @Test
   public void testMakeJsonScriptIsNotExecutable() throws BuildCatalogStepInputException, IOException
   {
      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("is not executable");

      File compileScript = new File("src/test/resources/buildCatalog/build_info_no_keys.txt");
      buildInfoMap.put(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name(), compileScript.getCanonicalPath());

      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testMakeJsonScriptDoesNotExist() throws BuildCatalogStepInputException, IOException
   {
      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("could not find");

      File compileScript = new File("/doesNotExist/script");
      buildInfoMap.put(BuildInfoKey.MAKE_JSON_SCRIPT_PATH.name(), compileScript.getCanonicalPath());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testTempDirEmpty() throws BuildCatalogStepInputException
   {
      buildInfoMap.put(BuildInfoKey.TEMP_DIR.name(), "   ");
      BuildInfo buildInfo = new BuildInfo(buildInfoMap);
      assertTrue(StringUtils.isNotBlank(buildInfo.getBaseTempDirectory()));

      buildInfoMap.remove(BuildInfoKey.TEMP_DIR.name());
      buildInfo = new BuildInfo(buildInfoMap);
      assertTrue(StringUtils.isNotBlank(buildInfo.getBaseTempDirectory()));
      assertTrue(buildInfo.getBaseTempDirectory().contains(System.getProperty("user.name")));
   }

   @Test
   public void testTempDirNotADir() throws BuildCatalogStepInputException, IOException
   {
      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("is not a directory");

      File tempDir = tempFolder.newFile();
      buildInfoMap.put(BuildInfoKey.TEMP_DIR.name(), tempDir.getPath());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testTempDirNotWritable() throws BuildCatalogStepInputException, IOException
   {
      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("not create temp dir");
      expectedEx.expectMessage("likely due to permissions");

      File parentDir = tempFolder.newFolder();
      File tempDir = new File(parentDir, "tmp");
      if (!parentDir.setWritable(false, false))
      {
         fail(String.format("Couldn't set '%s' to non-writeable to test condition", parentDir.getPath()));
      }
      buildInfoMap.put(BuildInfoKey.TEMP_DIR.name(), tempDir.getPath());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testSuppliedTempDir() throws BuildCatalogStepInputException, IOException
   {
      File tempDir = tempFolder.newFolder();
      String tempDirPath = tempDir.getCanonicalPath();
      buildInfoMap.put(BuildInfoKey.TEMP_DIR.name(), tempDirPath);
      BuildInfo buildInfo = new BuildInfo(buildInfoMap);
      // NOTE: A temp path like:
      //   /private/var/folders/gl/wdqzn5ws0jz5qtzy41x7h0d9ygjpzy/T/junit6963729479722839714/junit5653438778713754865
      // Will have another temp dir added to the end of it, similar to:
      //   /private/var/folders/gl/wdqzn5ws0jz5qtzy41x7h0d9ygjpzy/T/junit6963729479722839714/junit5653438778713754865/tmp_5016739545553248870_bior_build_catalog
      assertTrue(buildInfo.getBaseTempDirectory().startsWith(tempDirPath + "/tmp_")  &&  buildInfo.getBaseTempDirectory().endsWith("_bior_build_catalog"));
   }

   @Test
   public void testJsonOutputPath() throws BuildCatalogStepInputException, IOException
   {
      File tempFile = tempFolder.newFile();
      buildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), tempFile.getPath());
      BuildInfo buildInfo = new BuildInfo(buildInfoMap);
      String jsonOutputPath = buildInfo.getMakeJsonOutputFilePath();
      // NOTE: tempFile.getPath() and tempFile.getCanonicalPath() may return different results here.  Ex:
      //   tempFile.getPath(): 			         /var/folders/gl/wdqzn5ws0jz5qtzy41x7h0d9ygjpzy/T/junit7923949773784570448/junit1560564631861427491.tmp
      //   tempFile.getCanonicalPath():  /private/var/folders/gl/wdqzn5ws0jz5qtzy41x7h0d9ygjpzy/T/junit5324692481829392827/junit7155111497631017516.tmp
      String tempFilePath = tempFile.getCanonicalPath();
      assertEquals(jsonOutputPath, tempFilePath);
   }

   
   @Test
   /** This should use the default output path if the JSON output path is not specified */
   public void testJsonOutputPathNotSpecified() throws BuildCatalogStepInputException, IOException
   {
      buildInfoMap.remove(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name());
      BuildInfo buildInfo = new BuildInfo(buildInfoMap);
      String javaTemp = System.getProperty("java.io.tmpdir");
      String jsonOutputPath = buildInfo.getMakeJsonOutputFilePath();
      boolean isStartsWithTemp =
    		  jsonOutputPath.startsWith("/local2/tmp/")  ||
    		  jsonOutputPath.startsWith("/local1/tmp/")  ||
    		  jsonOutputPath.startsWith(javaTemp) ||
              jsonOutputPath.startsWith("/private" + javaTemp);
      boolean isEndsWithDefaultOutputFile = jsonOutputPath.endsWith(BuildInfo.DEFAULT_MAKE_JSON_OUTPUT_FILENAME);
      assertTrue(isStartsWithTemp);
      assertTrue(isEndsWithDefaultOutputFile);
   }

   @Test
   public void testWhenJsonOutputPathIsDir() throws BuildCatalogStepInputException, IOException
   {
      File tempDir = tempFolder.newFolder();

      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.toString());
      expectedEx.expectMessage("is a directory, not a file");
      expectedEx.expectMessage(tempDir.getPath());

      buildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), tempDir.getPath());
      new BuildInfo(buildInfoMap);
   }


   @Test
   public void testWhenJsonOutputPathParentNotWriteable() throws BuildCatalogStepInputException, IOException
   {
      File tempDir = tempFolder.newFolder();
      File jsonOut = new File(tempDir, "json");

      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.toString());
      expectedEx.expectMessage("is not writeable");
      expectedEx.expectMessage(tempDir.getPath());

      if (!tempDir.setWritable(false, false))
      {
         fail(String.format("Couldn't set '%s' to non-writeable to test condition", tempDir.getPath()));
      }

      buildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), jsonOut.getPath());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testWhenJsonOutputPathParentNotReadable() throws BuildCatalogStepInputException, IOException
   {
      File tempDir = tempFolder.newFolder();
      File jsonOut = new File(tempDir, "json");

      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.toString());
      expectedEx.expectMessage("is not readable");
      expectedEx.expectMessage(tempDir.getPath());

      if (!tempDir.setReadable(false, false))
      {
         fail(String.format("Couldn't set '%s' to non-readable to test condition", tempDir.getPath()));
      }
      buildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), jsonOut.getPath());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testWhenJsonOutputPathParentNotExecutable() throws BuildCatalogStepInputException, IOException
   {
      File tempDir = tempFolder.newFolder();
      File jsonOut = new File(tempDir, "json");

      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.toString());
      expectedEx.expectMessage("is not executable");
      expectedEx.expectMessage(tempDir.getPath());

      if (!tempDir.setExecutable(false, false))
      {
         fail(String.format("Couldn't set '%s' to non-executable to test condition", tempDir.getPath()));
      }

      buildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), jsonOut.getPath());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testJsonOutputPathCreation() throws BuildCatalogStepInputException, IOException
   {
      File tempDir = tempFolder.newFolder();
      String tempDirPath = tempDir.getCanonicalPath();
      buildInfoMap.put(BuildInfoKey.TEMP_DIR.name(), tempDirPath);
      BuildInfo buildInfo = new BuildInfo(buildInfoMap);
      String jsonOutputPath = buildInfo.getMakeJsonOutputFilePath();
      assertTrue(jsonOutputPath.startsWith(tempDirPath));
      assertTrue(jsonOutputPath.endsWith(BuildInfo.DEFAULT_MAKE_JSON_OUTPUT_FILENAME));
   }

   @Test
   public void testJsonOutputPathCreationWithNoTempDir() throws BuildCatalogStepInputException, IOException
   {
      File tempDir = tempFolder.newFolder();
      buildInfoMap.put(BuildInfoKey.TEMP_DIR.name(), tempDir.getPath());
      BuildInfo buildInfo = new BuildInfo(buildInfoMap);
      String jsonOutputPath = buildInfo.getMakeJsonOutputFilePath();
      String workingTempDirPath = buildInfo.getWorkingTempDirectory().getCanonicalPath();
      assertTrue(jsonOutputPath.startsWith(workingTempDirPath));
      assertTrue(jsonOutputPath.endsWith(BuildInfo.DEFAULT_MAKE_JSON_OUTPUT_FILENAME));
   }

   @Test
   public void testJsonOutputPathSimpleFileName() throws BuildCatalogStepInputException
   {
      buildInfoMap.put(BuildInfoKey.MAKE_JSON_OUTPUT_FILE_PATH.name(), "json");
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testTargetDir() throws BuildCatalogStepInputException, IOException
   {
      File tempDir = tempFolder.newFolder();
      buildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), tempDir.getPath());
      assertEquals(new BuildInfo(buildInfoMap).getTargetDirectory(), tempDir.getCanonicalPath());
   }

   @Test
   public void testTargetDirEmpty() throws BuildCatalogStepInputException, IOException
   {
	  // The path to the TARGET_DIR is resolved and canonicalized in BuildInfo, so "." will be the current Java project's directory
	  String currentDir = new File(".").getCanonicalPath();
	   
	  // First, make sure that the TARGET_DIR is not in the map
	  buildInfoMap.remove(BuildInfoKey.TARGET_DIR.name());
      assertEquals(new BuildInfo(buildInfoMap).getTargetDirectory(), currentDir);
      
      buildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), "");
      assertEquals(new BuildInfo(buildInfoMap).getTargetDirectory(), currentDir);
      
      buildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), "      ");
      assertEquals(new BuildInfo(buildInfoMap).getTargetDirectory(), currentDir);
      
      buildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), null);
      assertEquals(new BuildInfo(buildInfoMap).getTargetDirectory(), currentDir);
   }

   @Test
   public void testWhenTargetDirIsFile() throws BuildCatalogStepInputException, IOException
   {
      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("is not a directory");

      File tempFile = tempFolder.newFile();
      buildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), tempFile.getPath());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testWhenTargetDirDoesntExist() throws BuildCatalogStepInputException, IOException
   {
      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("Could not find");

      buildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), "/doesNotExist/tempdir1234");
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testWhenTargetDirIsNotWritable() throws BuildCatalogStepInputException, IOException
   {
      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("is not writeable");

      File tempDir = tempFolder.newFolder();
      if (!tempDir.setWritable(false, false))
      {
         fail(String.format("Couldn't set '%s' to non-writeable to test condition", tempDir.getPath()));
      }
      buildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), tempDir.getPath());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testWhenTargetDirIsNotReadable() throws BuildCatalogStepInputException, IOException
   {
      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("is not readable");

      File tempDir = tempFolder.newFolder();
      if (!tempDir.setReadable(false, false))
      {
         fail(String.format("Couldn't set '%s' to non-readable to test condition", tempDir.getPath()));
      }

      buildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), tempDir.getPath());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testWhenTargetDirIsNotExecutable() throws BuildCatalogStepInputException, IOException
   {
      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("is not executable");

      File tempDir = tempFolder.newFolder();
      if (!tempDir.setExecutable(false, false))
      {
         fail(String.format("Couldn't set '%s' to non-executable to test condition", tempDir.getPath()));
      }

      buildInfoMap.put(BuildInfoKey.TARGET_DIR.name(), tempDir.getPath());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testPreviousCatalog() throws BuildCatalogStepInputException, IOException
   {
      // When it's set to nothing in a variety of ways
      new BuildInfo(buildInfoMap);
      buildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), null);
      new BuildInfo(buildInfoMap);
      buildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), "");
      new BuildInfo(buildInfoMap);
      buildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), "    ");
      new BuildInfo(buildInfoMap);

      // Make a fake catalog file
      File tempDir = tempFolder.newFolder();
      File previousCatalogFile = new File(tempDir, "catalog.tsv.bgz");
      buildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), previousCatalogFile.getPath());
      if (!previousCatalogFile.createNewFile())
      {
         fail(String.format("Couldn't create new file %s to test BuildInfo constructor", previousCatalogFile.getPath()));
      }
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testPreviousCatalogDoesntExist() throws BuildCatalogStepInputException, IOException
   {
      File tempDir = tempFolder.newFolder();
      File previousCatalogFile = new File(tempDir, "catalog.tsv.bgz");

      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("Could not find");
      expectedEx.expectMessage(BuildInfoKey.PREVIOUS_CATALOG_PATH.toString());
      expectedEx.expectMessage(previousCatalogFile.getPath());

      buildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), previousCatalogFile.getPath());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testPreviousCatalogBadSuffix() throws BuildCatalogStepInputException, IOException
   {
      File tempDir = tempFolder.newFolder();
      File previousCatalogFile = new File(tempDir, "catalog");

      expectedEx.expect(BuildCatalogStepInputException.class);
      expectedEx.expectMessage("does not have expected catalog suffix");
      expectedEx.expectMessage(BuildInfoKey.PREVIOUS_CATALOG_PATH.toString());
      expectedEx.expectMessage(previousCatalogFile.getPath());

      if (!previousCatalogFile.createNewFile())
      {
         fail(String.format("Couldn't create new file %s to test BuildInfo constructor", previousCatalogFile.getPath()));
      }
      buildInfoMap.put(BuildInfoKey.PREVIOUS_CATALOG_PATH.name(), previousCatalogFile.getPath());
      new BuildInfo(buildInfoMap);
   }

   @Test
   public void testIndexes() throws BuildCatalogStepInputException
   {
      BuildInfo buildInfo = new BuildInfo(buildInfoMap);
      List<String> indexes = buildInfo.getIndexes();
      assertEquals(0, indexes.size());

      buildInfoMap.put(BuildInfoKey.INDEXES.name(), "  idx1,   idx2  ");
      buildInfo = new BuildInfo(buildInfoMap);
      indexes = buildInfo.getIndexes();
      assertEquals(2, indexes.size());
      assertEquals(indexes.get(0), "idx1");
      assertEquals(indexes.get(1), "idx2");

      buildInfoMap.put(BuildInfoKey.INDEXES.name(), " ,  ");
      buildInfo = new BuildInfo(buildInfoMap);
      indexes = buildInfo.getIndexes();
      assertEquals(0, indexes.size());
   }
}
