/**
 * bior_pipeline
 *
 * <p>@author Gregory Dougherty</p>
 * Copyright Mayo Clinic, 2014
 *
 */
package edu.mayo.bior.util;

/**
 * Class to clean text that might be surrounded by , or interspersed with, unwanted text
 *
 * <p>@author Gregory Dougherty</p>
 */
public class TextCleaner extends FieldProcessorBase
{
	private String	start;
	private String	middle;
	private String	replace;
	private String	end;
	private int		startLen;
	private int		middleLen;
	private int		replaceLen;
	private int		endLen;
	
	
	/**
	 * Create a TextCleaner that looks for the given {@code start} and {@code end} text, and 
	 * removes any that appear.  If have {@code start} but not {@code end}, or the reverse, 
	 * deletes the one found, and returns the rest
	 * 
	 * @param start	Text to remove from the beginning of the passed in string
	 * @param end	Text to remove from the end of the passed in string
	 */
	public TextCleaner (String start, String end)
	{
		this (start, null, null, end);
	}
	
	
	/**
	 * Create a TextCleaner that looks for the given {@code start} and {@code end} text, and 
	 * removes any that appear.  Also looks for any instances of {@code middle} and replaces them 
	 * with {@code replace}.  If have Looks for anything it can replace, does not require 
	 * "all or nothing".  Deletes {@code start} and {@code end} text before deleting any 
	 * {@code middle} matches
	 * 
	 * @param start		Text to remove from the beginning of the passed in string
	 * @param middle	Text to look for in the string, can be found multiple times
	 * @param replace	Text to replace {@code middle} with. Just delete if null / empty
	 * @param end		Text to remove from the end of the passed in string
	 */
	public TextCleaner (String start, String middle, String replace, String end)
	{
		this.start = start;
		this.middle = middle;
		this.replace = replace;
		this.end = end;
		
		if (start == null)
			startLen = 0;
		else
			startLen = start.length ();
		
		if (middle == null)
			middleLen = 0;
		else
			middleLen = middle.length ();
		
		if (replace == null)
			replaceLen = 0;
		else
			replaceLen = replace.length ();
		
		if (end == null)
			endLen = 0;
		else
			endLen = end.length ();
	}
	
	
	/* (non-Javadoc)
	 * @see edu.mayo.bior.util.FieldProcessor#process(java.lang.String)
	 */
	public String process (String value)
	{
		if (super.isBlank (value))
			return gCurBlankValue;
		
		boolean	hasStart = (startLen > 0) ? value.startsWith (start) : false;
		boolean	hasEnd = (endLen > 0) ? value.endsWith (end) : false;
		
		if (middleLen > 0)
			return processReplace (value, hasStart, hasEnd);
		
		return processNoReplace (value, hasStart, hasEnd);
	}
	
	
	/**
	 * Process when just have start / end
	 * 
	 * @param value		String to process
	 * @param hasStart	True if have the start text to remove
	 * @param hasEnd	True if have the end text to remove
	 * @return	Processed String
	 */
	private String processNoReplace (String value, boolean hasStart, boolean hasEnd)
	{
		if (hasStart)
		{
			if (hasEnd)
				return value.substring (startLen, value.length () - endLen);
			
			return value.substring (startLen);
		}
		
		if (hasEnd)
			return value.substring (0, value.length () - endLen);
		
		return value;
	}
	
	
	/**
	 * Process when have replace text as well as start / end
	 * 
	 * @param value		String to process
	 * @param hasStart	True if have the start text to remove
	 * @param hasEnd	True if have the end text to remove
	 * @return	Processed String
	 */
	private String processReplace (String value, boolean hasStart, boolean hasEnd)
	{
		int	curStart = hasStart ? startLen : 0;
		int	foundPos = value.indexOf (middle, curStart);
		int	curEnd = hasEnd ? value.length () -  endLen: value.length ();
		
		if ((foundPos < 0) || ((foundPos + middleLen) > curEnd))
			return value.substring (curStart, curEnd);
		
		int				curSize = curEnd - curStart;
		StringBuilder	hold = new StringBuilder (curSize);
		
		do
		{
			hold.append (value.substring (curStart, foundPos));
			if (replaceLen > 0)
				hold.append (replace);
			
			curStart = foundPos + middleLen;
			foundPos = value.indexOf (middle, curStart);
		}
		while ((foundPos >= 0) && ((foundPos + middleLen) <= curEnd));
		
		hold.append (value.substring (curStart, curEnd));	// Copy over everything left
		return hold.toString ();
	}
	
}
