/**
 * bior_pipeline
 *
 * <p>@author Gregory Dougherty</p>
 * Copyright Mayo Clinic, 2014
 *
 */
package edu.mayo.bior.util;

import static edu.mayo.bior.pipeline.SplitFile.kIncludeBlank;
import static edu.mayo.bior.pipeline.SplitFile.kReturnAll;
import edu.mayo.bior.pipeline.SplitFile;
import edu.mayo.pipes.iterators.Compressor;
import java.io.*;
import java.util.*;
import java.util.Map.Entry;


/**
 * Static Class for extracting information from Catalog files and their associated helper files
 *
 * <p>@author Gregory Dougherty</p>
 */
public class CatalogUtils
{
	private static File	catalogFile;
	
	protected static final String	kDatasource = ".datasource.properties";
	protected static final int		kDSLen = kDatasource.length ();
	protected static final String	kColumns = ".columns.tsv";
	protected static final int		kColLen = kColumns.length ();
	protected static final String	kData = ".tsv.bgz";
	private static final String	kDeprecated = "DEPRECATED";
	private static final String	kCatFileEnd = ".tsv.bgz";
	private static final int	kCatFileLen = kCatFileEnd.length ();
	private static final int	kEnabledCol = 0;
	private static final int	kPathCol = kEnabledCol + 1;
	private static final int	kNumCols = kPathCol + 1;
	
	private static final String[]		kGoldenTests = {"_altAlleles", "_refAllele", "_landmark", "_minBP", "_maxBP", "_type", "_id", "_strand"};
	private static final Set<String>	kTestGolden = populateSet (new HashSet<String> (), kGoldenTests);
	private static final int	kChromCol = 0;
	private static final int	kStartCol = kChromCol + 1;
	private static final int	kEndCol = kStartCol + 1;
	private static final int	kJSONCol = kEndCol + 1;
	private static final int	kNumDataCols = kJSONCol + 1;
	/** Can only run Overlap against this catalog file */
	public static final String	kOverlap = "bior_overlap";
	/** Can run SameVariant or Overlap against this catalog file */
	public static final String	kSameVariant = "bior_same_variant";
	
	/** Only return keyword pairs if the human name is different from the keyword name */
	public static final boolean	kOnlyUnique = true;
	/** Return all found keyword pairs */
	public static final boolean	kReturnAllNames = false;
	
	protected static final boolean	kHonorBlacklist = false;
	private static final boolean	kIgnoreBlacklist = true;
	
	/** Column of 2d array holding the human names */
	public static final int	kHumanNameCol = 0;
	/** Column of 2d array holding the machine names */
	public static final int	kMachineNameCol = kHumanNameCol + 1;
	/** Min number of columns in the 2d array */
	public static final int	kNumNameCols = kMachineNameCol + 1;
	
	
	/**
	 * Get the catalog build from the ".datasource.properties" file for the catalog
	 * 
	 * @param catalogPath		Path to the catalog file.  If null or empty will return null
	 * @return	String with the pared down build of the catalog, or null if there was a problem
	 */
	public static final Properties getCatalogProperties (String catalogPath)
	{
		if (isEmpty (catalogPath))
			return null;
		
		if (!catalogPath.endsWith (kCatFileEnd))
			return null;
		
		String	basePath = catalogPath.substring (0, catalogPath.length () - kCatFileLen);
		File	dataSourceFile = new File (basePath + kDatasource);
		
		if (!dataSourceFile.exists ())
			return null;
		
		try
		{
			Properties	catProperties = new Properties ();
			InputStream	input = new FileInputStream (dataSourceFile);
			
			if (!loadProperties (input, catProperties))
				return null;
			
			return catProperties;
		}
		catch (FileNotFoundException oops)
		{
			return null;
		}
	}
	
	
	/**
	 * Get the catalog build from the ".datasource.properties" file for the catalog
	 * 
	 * @param catalogPath		Path to the catalog file.  If null or empty will return null
	 * @return	String with the pared down build of the catalog, or null if there was a problem
	 */
	public static final String getCatalogBuild (String catalogPath)
	{
		Properties	catProperties = getCatalogProperties (catalogPath);
		
		if (catProperties == null)
			return null;
		
		return getBuild (catProperties);
	}
	
	
	/**
	 * Get the base build, stripping off any .p... or " from liftover..."
	 * 
	 * @param catProperties	{@linkplain Properties} to get data from, if null will throw 
	 * {@linkplain NullPointerException}
	 * @return	String, never null, only empty if Properties has no data for us: {@code Build}
	 */
	static final String getBuild (Properties catProperties)
	{
		return getBuild (catProperties.getProperty ("Build"));
	}
	
	
	/**
	 * Get the unique across BioR Catalogs name for the catalog
	 * 
	 * @param catProperties	{@linkplain Properties} to get data from, if null will throw 
	 * {@linkplain NullPointerException}
	 * @return	String, never null, only empty if Properties has no data for us: {@code ShortUniqueName}
	 */
	static final String getUniqueName (Properties catProperties)
	{
		return catProperties.getProperty ("ShortUniqueName");
	}
	
	
	/**
	 * Get the not unique across BioR Catalogs human friendly name for the catalog
	 * 
	 * @param catProperties	{@linkplain Properties} to get data from, if null will throw 
	 * {@linkplain NullPointerException}
	 * @param catNames		{@linkplain Map} of names to check if no {@code HumanName} in {@code catProperties}
	 * @param uniqueName	Name to look for in {@code catNames} if no {@code HumanName} in {@code catProperties}
	 * @return	String, never null, only empty if Properties has no data for us: {@code HumanName}
	 */
	static final String getHumanName (Properties catProperties, Map<String, String> catNames, String uniqueName)
	{
		String	humanName = catProperties.getProperty ("HumanName");
		
		if (humanName == null)
			humanName = catNames.get (uniqueName);
		
		return humanName;
	}
	
	
	/**
	 * Get the Category Path for the catalog
	 * 
	 * @param catProperties	{@linkplain Properties} to get data from, if null will throw 
	 * {@linkplain NullPointerException}
	 * @param categories	{@linkplain Map} of categories to check if no {@code Category} in {@code catProperties}
	 * @param uniqueName	Name to look for in {@code categories} if no {@code Category} in {@code catProperties}
	 * @return	String, never null, only empty if Properties has no data for us: {@code Category}
	 */
	private static String getCategory (Properties catProperties, Map<String, String> categories, String uniqueName)
	{
		String	category = catProperties.getProperty ("Category");
		
		if (category == null)
			category = categories.get (uniqueName);
		
		return category;
	}
	
	
	/**
	 * Get the source of the data in the catalog
	 * 
	 * @param catProperties	{@linkplain Properties} to get data from, if null will throw 
	 * {@linkplain NullPointerException}
	 * @return	String, never null, only empty if Properties has no data for us: {@code Source}
	 */
	static final String getSource (Properties catProperties)
	{
		return catProperties.getProperty ("Source");
	}
	
	
	/**
	 * Get the description of the data in the catalog
	 * 
	 * @param catProperties	{@linkplain Properties} to get data from, if null will throw 
	 * {@linkplain NullPointerException}
	 * @return	String, never null, only empty if Properties has no data for us: {@code Description}
	 */
	static final String getDescription (Properties catProperties)
	{
		return catProperties.getProperty ("Description");
	}
	
	
	/**
	 * Get the version of the data in the catalog
	 * 
	 * @param catProperties	{@linkplain Properties} to get data from, if null will throw 
	 * {@linkplain NullPointerException}
	 * @return	String, never null, only empty if Properties has no data for us: {@code Version}
	 */
	static final String getVersion (Properties catProperties)
	{
		return catProperties.getProperty ("Version");
	}
	
	
	/**
	 * Get all the Categories, and return them in a {@linkplain Map} from build to {@linkplain List} 
	 * of {@linkplain Category} for that build
	 * 
	 * @param catalogsFile		File that holds all the catalog files of interest.  Must not be null
	 * @param categoriesFile	File that holds default categories for catalog files that don't have that information.  Can be null
	 * @param namesFile			File that holds default human names for catalog files that don't have that information.  Can be null
	 * @return	Map from Build name to Lists of {@linkplain Category} for that build 
	 * @throws IOException	File not found, file can't be read
	 */
	public static Map<String, List<Category>> getTheCategories (String catalogsFile, String categoriesFile, String namesFile) throws IOException
	{
//		System.out.println ("getTheCategories called with ");
//		System.out.println (catalogsFile);
//		System.out.println (categoriesFile);
//		System.out.println (namesFile);
//		FilenameFilter				filter = getCatalogFilter ();
//		Comparator<File>			fileSorter = getCatalogComparator ();
		Map<String, List<Category>> results = new HashMap<String, List<Category>> ();
		Map<String, String>			catPaths = getStringToStringMap (categoriesFile);
		Map<String, String>			catNames = getStringToStringMap (namesFile);
		BufferedReader				dataReader = Compressor.readFile (new File (catalogsFile));
		String						line;
		
//		dump ("catPaths", catPaths, System.out);
//		dump ("catNames", catNames, System.out);
		while ((line = dataReader.readLine ()) != null)
		{
			if (line.startsWith ("#"))
				continue;	// Skip comments
			
			String[]	cols = SplitFile.mySplit (line, "\t", kReturnAll);
			
			if ((cols.length < kNumCols) || cols[kEnabledCol].equalsIgnoreCase (kDeprecated))
			{
//				System.out.println ("Bad line:");
//				System.out.println (line);
				continue;
			}
			
			String	path = cols[kPathCol];
//			System.err.print ("Looking at ");
//			System.err.println (path);
			if (!path.endsWith (kCatFileEnd))
			{
//				System.out.println ("Bad path:");
//				System.out.println (line);
				continue;
			}
			
			path = path.substring (0, path.length () - kCatFileLen);
			
			File	columnsFile = new File (path + kColumns);
			File	dataSourceFile = new File (path + kDatasource);
			
			if (!columnsFile.exists () || !dataSourceFile.exists ())
			{
				System.err.println ("Missing files:");
				System.err.println (line);
				continue;
			}
			
//			System.err.print ("Getting categories ");
//			System.err.println (dataSourceFile.getAbsolutePath ());
			parseCategories (columnsFile, dataSourceFile, catPaths, catNames, results);
//			File	catFile = new File (path);
//			
//			if (catFile.exists ())
//				parseDirectory (catFile.getParentFile (), filter, fileSorter, results, false);
		}
		
//		dump ("results", results, System.out);
		return results;
	}
	
	
	/**
	 * Get all the keywords and their human readable names for a specified catalog, if its 
	 * ".columns.tsv" file exists
	 * 
	 * @param catalogPath		Path to the catalog file.  If null or empty will return null
	 * @param onlyUniqueHuman	Only return those keywords that have human names different from the 
	 * machine names
	 * @return	Array of String arrays, where the first array holds the  human readable names and 
	 * the second array holds the matching machine readable keywords, or null if there was a problem
	 */
	public static String[][] getCatalogKeywords (String catalogPath, boolean onlyUniqueHuman)
	{
		File	columnsFile = getColumnsFile (catalogPath);
		
		if (columnsFile == null)
			return null;
		
		return getKeywords (columnsFile, onlyUniqueHuman, kIgnoreBlacklist);
	}
	
	
	/**
	 * Get all the Categories, and return them in a list
	 * 
	 * @param parsePath	Where to start looking for .datasource.properties and .columns.tsv files.  Must not be null
	 * @return	Map from Build name to Lists of {@linkplain Category} for that build 
	 */
	static Map<String, List<Category>> getAllCategories (String parsePath, String categoriesFile, String namesFile)
	{
		Map<String, List<Category>> results = new HashMap<String, List<Category>> ();
		Map<String, String>			catPaths = null;
		Map<String, String>			catNames = null;
		File				root = new File (parsePath);
		FilenameFilter		filter = getCatalogFilter ();
		Comparator<File>	fileSorter = getCatalogComparator ();
		BufferedWriter		dataWriter = null;
		
		try
		{
			catPaths = getStringToStringMap (categoriesFile);
			catNames = getStringToStringMap (namesFile);
			catalogFile = new File (root, "catalog_file");
			
			Compressor	comp = new Compressor (null, catalogFile, true);
			
			dataWriter = comp.getWriter ();
			dataWriter.newLine ();
		}
		catch (IOException oops)
		{
			catalogFile = new File ("catalog_file");
		}
		finally
		{
			closeWriter (dataWriter);
		}
		
		parseDirectory (root, filter, fileSorter, catPaths, catNames, results);
		
		return results;
	}
	
	/** Return true if the catalog is deprecated (the catalog directory contains a file that ends with "DEPRECATED.txt
	 *  	ALL.wgs.sites.vcf.DEPRECATED.TXT
			ALL.wgs.sites.vcf.tsv.bgz  
	 * @param catalogPath Path to the catalog
	 * @return true if deprecated file exists:  [CatalogName].DEPRECATED.txt */
	public static boolean isDeprecated(String catalogPath) {
		if( catalogPath == null  ||  catalogPath.length() == 0 ) {
			return false;
		}
		File deprecatedFile = new File(catalogPath.replace(".tsv.bgz", ".DEPRECATED.txt"));
		return deprecatedFile.exists();
	}

	
	public static void warnIfDeprecated(String catalogPath) {
		if( isDeprecated(catalogPath) ) {
			System.err.println("Warning: The catalog you are using has been deprecated (" + catalogPath + ").  Please switch to a newer version.");
		}
	}
	
	/**
	 * Parse a directory, getting any data files and reading them, and then recursing down through any sub directories
	 * 
	 * @param theDir		Directory to parse
	 * @param filter		Filter to use to get the files we're interested in
	 * @param fileSorter	File sorter that sorts files before directories, and all in alphabetical order
	 * @param results		Map from Build name to Lists to add results to / update 
	 */
	private static void parseDirectory (File theDir, FilenameFilter filter, Comparator<File> fileSorter, Map<String, String> catPaths, 
										Map<String, String> catNames, Map<String, List<Category>> results)
	{
		File[]	items = theDir.listFiles (filter);
		File	columnsFile = null;
		
		if (items == null)
			return;
		
		Arrays.sort (items, fileSorter);
		for (File theItem : items)
		{
			if (theItem.isDirectory ())
			{
				parseDirectory (theItem, filter, fileSorter, catPaths, catNames, results);
				continue;
			}
			
			String	name = theItem.getName ();
			
			if (name.endsWith (kColumns))
				columnsFile = theItem;
			else if (columnsFile != null)	// Have a data source file, need a column file, too
			{
				File	dataSourceFile = theItem;
				String	baseName = dataSourceFile.getName ();
				
				baseName = baseName.substring (0, baseName.length () - kDSLen);	// Get shared part of name
				if (!columnsFile.getName ().startsWith (baseName))
				{
					columnsFile = null;
					continue;	// No Matching names.  bail
				}
				
				parseCategories (columnsFile, dataSourceFile, catPaths, catNames, results);
			}
		}
	}
	
	
	/**
	 * Parse the Category and Keyword information from the files, create the Category, and add it to its proper place in the tree
	 * 
	 * @param columnsFile		File holding Keyword information
	 * @param dataSourceFile	File holding Category information
	 * @param results			Map from Build name to Lists to add results to / update 
	 * @return	True if loaded successfully, false if there was a problem
	 */
	private static boolean parseCategories (File columnsFile, File dataSourceFile, Map<String, String> catPaths, 
											Map<String, String> catNames, Map<String, List<Category>> results)
	{
		try
		{
			Properties	catProperties = new Properties ();
			InputStream	input = new FileInputStream (dataSourceFile);
			
			if (!loadProperties (input, catProperties))
				return false;
			
			String			catSource = getSourceAndDesc (catProperties);
			String			uniqueName = getUniqueName (catProperties);
			String			catPath = getCategory (catProperties, catPaths, uniqueName);
			String			humanName = getHumanName (catProperties, catNames, uniqueName);
			String			build = getBuild (catProperties);
			List<Category>	curList = results.get (build);
			
			if (curList == null)
			{
				curList = new ArrayList<Category> ();
				results.put (build, curList);
			}
			
//			System.err.print ("Build: ");
//			System.err.print (build);
//			System.err.print (", uniqueName: ");
//			System.err.print (uniqueName);
			if (humanName == null)
			{
//				System.err.println (", No human name");
				return false;	// No name, don't want
			}
			
//			System.err.print (", human name: ");
//			System.err.println (humanName);
			if (catPath == null)
				catPath = "Uncategorized";
			
			Category	parent = addPath (catPath, curList);
			Category	element = new Category (humanName, uniqueName, catSource, parent);
			
			addKeywords (element, columnsFile);
			curList.add (element);
			return true;
		}
		catch (IOException oops)
		{
			oops.printStackTrace ();
			return false;
		}
	}
	
	
	/**
	 * Get the Source description from the properties file
	 * 
	 * @param catProperties	{@linkplain Properties} to get data from
	 * @return	String, never null, only empty if Properties has no data for us: Source - Description
	 */
	private static String getSourceAndDesc (Properties catProperties)
	{
		String	source = getSource (catProperties);
		String	description = getDescription (catProperties);
		
		if (isEmpty (source))
		{
			if (isEmpty (description))
				return "";
			
			return description;
		}
		
		if (isEmpty (description))
			return source;
		
		return source + " - " + description;
	}
	
	
	/**
	 * Get the base build, stripping off any .p... or " from liftover..."
	 * 
	 * @param catProperties	{@linkplain Properties} to get data from
	 * @return	String, never null, only empty if Properties has no data for us: Build
	 */
	static String getBuild (String build)
	{
		int	len = build.length ();
		int	pos = len;
		
		for (String separator : kBuildSeparators)
		{
			int	test = build.indexOf (separator);
			
			if ((test > 0) && (test < pos))
				pos = test;
		}
		
		if (pos < len)
			return build.substring (0, pos);
		
		return build;
	}
	
	
	/**
	 * Add the path to the List if it doesn't exist, With "root" Categories 
	 * coming before all others and in alphabetical order, 
	 * and any other categories just getting added to the end
	 * 
	 * @param catPath	Path the calling Category is on, elements separated by ':'
	 * @param curList	Current list of Categories, to add things to / get things from
	 * @return	The Category for the last element of the path 
	 */
	private static Category addPath (String catPath, List<Category> curList)
	{
		if (isEmpty (catPath))
			return null;
		
		String[]	elements = SplitFile.mySplit (catPath, ":", kReturnAll);
		Category	last = null;
		String		root = elements[0];	// All results will have at least one element
		int			numCategories = curList.size ();
		int			which = 0;
		
		while (which < numCategories)
		{
			Category	test = curList.get (which);
			String		name = test.getName ();
			
			if (name.equals (root))
			{
				last = test;
				break;	// Found match, done
			}
			
			if (test.getParent () != null)
				break;	// Seen all root categories, done
			
			if (name.compareTo (root) > 0)
				break;	// Found a bigger root than new one, insert here
			
			++which;
		}
		
		if (last == null)	// Didn't find existing, create the path
		{
			last = new Category (root, root, "", null);
			curList.add (which, last);	// Will add to end when which == numCategories
			if (elements.length > 1)	// More bits to add
				last = addPath (last, elements, 1, curList);
		}
		else
		{
			int	pathPos = 1;
			int	numElements = elements.length;
			
			while (pathPos < numElements)	// Now add the rest
			{
				String	name = elements[pathPos];
				while (which < numCategories)	// See if next element of path is also already created
				{
					Category	test = curList.get (which);
					String		testName = test.getName ();
					
					if (testName.equals (name))
					{
						last = test;
						break;	// Found match, done for this name
					}
					
					++which;
				}
				
				if (which == numCategories)	// Didn't find match, add rest of path, we're done
				{
					last = addPath (last, elements, pathPos, curList);
					break;
				}
				
				++pathPos;
			}
		}
		
		return last;
	}
	
	
	/**
	 * Add the items in elements to curList, with parent being the parent of the first item added
	 * 
	 * @param parent	Item at the base of the added path
	 * @param elements	Path elements to add, starting at starting
	 * @param starting	First index of starting to use
	 * @param curList	Current list of Categories, to add things to
	 * @return	The final {@linkplain Category} created
	 */
	private static Category addPath (Category parent, String[] elements, int starting, List<Category> curList)
	{
		int			numElements = elements.length;
		Category	last = parent;
		
		for (int i = starting; i < numElements; ++i)
		{
			String	name = elements[i];
			
			last = new Category (name, name, "", last);
			curList.add (last);
		}
		
		return last;
	}
	
	private static final int	kUnique = 0;
	private static final int	kType = kUnique + 1;
	private static final int	kCount = kType + 1;
	private static final int	kDescription = kCount + 1;
	private static final int	kName = kDescription + 1;
	private static final int	kMaxNumCols = kName + 1;
	private static final String[]	kSpaceStrs = {".", "_"};
	private static final String[]	kBuildSeparators = {".", " "};
	protected static final String[]	kBlacklistHeader = {"### These fields will NOT be showing in BioRWeb.  All others will.", 
	                             	                    "### If this file is NOT present, then assume that all fields, except for " + 
	                             	                    "the golden attributes (those beginning with _) should be shown", 
	                             	                    "### If this file is present, but contains no columns, then assume that all fields " + 
	                             	                    "should be shown"};
	
	/**
	 * Parse the columnsFile, and add the results as {@linkplain Keyword}s to the {@linkplain Category}
	 * 
	 * @param element		Category to add to
	 * @param columnsFile	File holding the Keyword data
	 * @throws IOException	If there's a problem with the file
	 */
	private static void addKeywords (Category element, File columnsFile) throws IOException
	{
		Set<String>		blacklist = getBlackList (columnsFile);
		Compressor		comp = new Compressor (columnsFile, null);
		BufferedReader	dataReader = comp.getReader ();
		String			catUniqueName = element.getUniqueName ();
		StringBuilder	keyUniqueName = new StringBuilder (catUniqueName);
//		List<String>	golden = new ArrayList<String> ();
		String			line;
		
		keyUniqueName.append (':');
		int	baseLen = keyUniqueName.length ();
		while ((line = dataReader.readLine ()) != null)
		{
			if (line.startsWith ("#"))
				continue;	// Skip header lines
			
			String[]	elements = SplitFile.mySplit (line, "\t", kReturnAll, kIncludeBlank);
			String		uniqueName = elements[kUnique];
			String		name;
			int			numElements = elements.length;
			
//			if (uniqueName.startsWith ("_"))
//				golden.add (uniqueName);
			if (numElements >= kMaxNumCols)
				name = elements[kName];
			else
				name = cleanName (uniqueName);
			
			if (blacklist.contains (uniqueName))
//			if (isEmpty (name))
				continue;	// Items with a blank human name aren't displayed
			
			keyUniqueName.setLength (baseLen);
			keyUniqueName.append (uniqueName);
			
			if (numElements < kName)
				System.out.println ("Bad line");
			else
			{
				String	description = elements[kDescription];
				Keyword	theKeyword = new Keyword (keyUniqueName.toString (), name, description);
				
				element.addKeyword (theKeyword);
			}
		}
		
//		validateDataFile (columnsFile, golden, catUniqueName);
	}
	
	
	/**
	 * Parse the columnsFile, and return Keyword as 2d array of {@linkplain String}
	 * 
	 * @param columnsFile		File holding the Keyword data
	 * @param onlyUniqueHuman	Only return those keywords that have human names different from the 
	 * machine names
	 * @param ignoreBlacklist	If true, include all keywords, if false, use blacklist to eliminate 
	 * keywords
	 * @return	2d Array of String holding two String[].  1st one will have human names, 2nd one 
	 * will have the machine readable names.
	 */
	private static String[][] getKeywords (File columnsFile, boolean onlyUniqueHuman, boolean ignoreBlacklist)
	{
		try
		{
			Set<String>		blacklist;
			Compressor		comp = new Compressor (columnsFile, null);
			BufferedReader	dataReader = comp.getReader ();
			List<String>	humanNames = new ArrayList<String> ();
			List<String>	machineNames = new ArrayList<String> ();
			String			line;
			
			if (ignoreBlacklist)
				blacklist = new HashSet<String> ();
			else
				blacklist = getBlackList (columnsFile);
			
			while ((line = dataReader.readLine ()) != null)
			{
				if (line.startsWith ("#"))
					continue;	// Skip header lines
				
				String[]	elements = SplitFile.mySplit (line, "\t", kReturnAll, kIncludeBlank);
				String		uniqueName = elements[kUnique];	// Will always have at least 1 element
				String		name;
				String		cleanName = cleanName (uniqueName);
				int			numElements = elements.length;
				
				if (blacklist.contains (uniqueName))
					continue;
				
				if (numElements >= kMaxNumCols)
				{
					name = elements[kName];
					if (isEmpty (name))
						name = cleanName;
					if (onlyUniqueHuman && (name.equals (cleanName)))
						continue;
				}
				else if (!onlyUniqueHuman)
					name = cleanName;
				else
					continue;
				
				humanNames.add (name);
				machineNames.add (uniqueName);
			}
			
			int			numNames = Math.min (humanNames.size (), machineNames.size ());
			String[]	hNames = humanNames.toArray (new String[numNames]);
			String[]	mNames = machineNames.toArray (new String[numNames]);
			String[][]	results = new String[kNumNameCols][];
			
			results[kHumanNameCol] = hNames;
			results[kMachineNameCol] = mNames;
			return results;
		}
		catch (IOException oops)
		{
			oops.printStackTrace ();
		}
		
		return null;
	}
	
	
	/**
	 * Parse the columnsFile, and add the results as {@linkplain Keyword}s to the {@linkplain Category}
	 * 
	 * @param element		Category to add to
	 * @param columnsFile	File holding the Keyword data
	 * @throws IOException	If there's a problem with the file
	 */
	protected static void addKeywordsMakeBlacklist (Category element, File columnsFile) throws IOException
	{
		File			blacklistFile = new File (columnsFile.getParentFile (), columnsFile.getName () + ".blacklist");
		Compressor		comp = new Compressor (columnsFile, blacklistFile);
		BufferedReader	dataReader = comp.getReader ();
		BufferedWriter	blWriter = comp.getWriter ();
		String			catUniqueName = element.getUniqueName ();
		StringBuilder	keyUniqueName = new StringBuilder (catUniqueName);
		String			line;
		
		for (String hLine : kBlacklistHeader)
		{
			blWriter.write (hLine);
			blWriter.newLine ();
		}
		
		keyUniqueName.append (':');
		int	baseLen = keyUniqueName.length ();
		while ((line = dataReader.readLine ()) != null)
		{
			if (line.startsWith ("#"))
				continue;	// Skip header lines
			
			String[]	elements = SplitFile.mySplit (line, "\t", kReturnAll, kIncludeBlank);
			String		uniqueName = elements[kUnique];
			String		name;
			int			numElements = elements.length;
			
			if (numElements >= kMaxNumCols)
				name = elements[kName];
			else
				name = cleanName (uniqueName);
			
			if (isEmpty (name))
			{
				blWriter.write (uniqueName);
				blWriter.newLine ();
				continue;	// Items with a blank human name aren't displayed
			}
			
			keyUniqueName.setLength (baseLen);
			keyUniqueName.append (uniqueName);
			
			if (numElements < kName)
				System.out.println ("Bad line");
			else
			{
				String	description = elements[kDescription];
				Keyword	theKeyword = new Keyword (keyUniqueName.toString (), name, description);
				
				element.addKeyword (theKeyword);
			}
		}
		
		blWriter.close ();
	}
	
	
	/**
	 * Get the Blacklist for this columns file
	 * 
	 * @param columnsFile	File of interest
	 * @return
	 */
	private static Set<String> getBlackList (File columnsFile)
	{
		Set<String>	results = new HashSet<String> ();
		File		blacklist = new File (columnsFile.getParentFile (), columnsFile.getName () + ".blacklist");
		boolean		makeList = false;
		
		if (!blacklist.exists ())
		{
			blacklist = columnsFile;
			makeList = true;
		}
		
		try
		{
			Compressor		comp = new Compressor (blacklist, null);
			BufferedReader	dataReader = comp.getReader ();
			String			line;
			
			while ((line = dataReader.readLine ()) != null)
			{
				if (makeList)
				{
					if (line.startsWith ("#"))
						continue;	// Skip header lines
					
					String[]	elements = SplitFile.mySplit (line, "\t", kReturnAll, kIncludeBlank);
					String		uniqueName = elements[kUnique];
					
					if (uniqueName.startsWith ("_"))
						results.add (uniqueName);
				}
				else
					results.add (line);
			}
		}
		catch (IOException oops)
		{
			oops.printStackTrace ();
		}
		
		return results;
	}
	
	
	/**
	 * Method so can create and initialize a Set with a single command
	 * 
	 * @param theSet	Set to initialize
	 * @param theData	Data to add to it
	 * @return	Null if {@code theSet} is null.  Unchanged {@code theSet} if theData is null or 
	 * empty.  Else {@code theSet} with all of {@code theData} added to it
	 */
	private static <T> Set<T> populateSet (Set<T> theSet, T[] theData)
	{
		if (theSet == null)
			return null;
		
		if (theData == null)
			return theSet;
		
		for (T item :theData)
			theSet.add (item);
		
		return theSet;
	}
	
	
	/**
	 * Get a String describing the BioR function to be used on this catalog, or an error telling 
	 * why can't use BioR on this catalog
	 * 
	 * @param golden	Valid set holding the golden attributes from the catalog of interest
	 * @return	kSameVariant, kOverlap, or an error message
	 */
	private static final String getGoldenStatus (Set<String> golden)
	{
		String	goldenStatus;
		
		if (golden.isEmpty ())
			goldenStatus = "No Golden attributes";	// nothing more to test, no golden attributes
		else if (!(golden.contains ("_landmark") && golden.contains ("_minBP") && golden.contains ("_maxBP")))
			goldenStatus = getAsString ("Only has: ", golden, ", ");	// nothing more to test, not enough golden attributes
		else if (golden.contains ("_altAlleles") && golden.contains ("_refAllele"))
			goldenStatus = kSameVariant;
		else
			goldenStatus = kOverlap;
		
		return goldenStatus;
	}
		
	
	/**
	 * Validate that the non-id golden attributes columns always have data in them
	 * 
	 * @param columnsFile	File with the column info
	 */
	protected static final void validateDataFile (File columnsFile, List<String> lGolden, String uniqueName)
	{
		Set<String>			golden = new HashSet<String> (lGolden);
		Iterator<String>	iter = golden.iterator ();
		BufferedWriter		dataWriter = null;
		
		while (iter.hasNext ())
		{
			if (!kTestGolden.contains (iter.next ()))
				iter.remove ();
		}
		
		try
		{
			Compressor	comp = new Compressor (null, catalogFile, true);
			String		goldenStatus = getGoldenStatus (golden);
			String		name = columnsFile.getName ();
			
			dataWriter = comp.getWriter ();
			name = name.substring (0, name.length () - kColLen) + kData;
			
			File	dataFile = new File (columnsFile.getParentFile (), name);
			
			if (!dataFile.exists ())
				goldenStatus = "Missing catalog file";	// change message: no file
			
			dataWriter.write (uniqueName);
			dataWriter.write ("\t");
			dataWriter.write (goldenStatus);
			dataWriter.write ("\t");
			dataWriter.write (dataFile.getAbsolutePath ());
			dataWriter.newLine ();
		}
		catch (IOException oops)
		{
			oops.printStackTrace ();
		}
		finally
		{
			closeWriter (dataWriter);
		}
	}
		
	
	/**
	 * Get all the keywords and their human readable names for a specified catalog, if its 
	 * ".columns.tsv" file exists
	 * 
	 * @param catalogPath		Path to the catalog file.  If null or empty will return null
	 * @param onlyUniqueHuman	Only return those keywords that have human names different from the 
	 * machine names
	 * @return	Array of String arrays, where the first array holds the  human readable names and 
	 * the second array holds the matching machine readable keywords, or null if there was a problem
	 */
	private static final File getColumnsFile (String catalogPath)
	{
		if (isEmpty (catalogPath))
			return null;
		
		if (!catalogPath.endsWith (kCatFileEnd))
			return null;
		
		String	basePath = catalogPath.substring (0, catalogPath.length () - kCatFileLen);
		File	columnsFile = new File (basePath + kColumns);
		
		if (!columnsFile.exists ())
			return null;
		
		return columnsFile;
	}
	
	
	/**
	 * Get a String describing the BioR function to be used on this catalog, or an error telling 
	 * why can't use BioR on this catalog
	 * 
	 * @param columnsFile	File with the column info
	 * @return	kSameVariant, kOverlap, or an error message
	 */
	static final String getBioRType (String catalogPath)
	{
		File	columnsFile = getColumnsFile (catalogPath);
		
		if (columnsFile == null)
			return "Could not find columns file";
		
		return getBioRType (columnsFile);
	}
	
	
	/**
	 * Get a String describing the BioR function to be used on this catalog, or an error telling 
	 * why can't use BioR on this catalog
	 * 
	 * @param columnsFile	File with the column info
	 * @return	kSameVariant, kOverlap, or an error message
	 */
	private static final String getBioRType (File columnsFile)
	{
		try
		{
			Compressor		comp = new Compressor (columnsFile, null);
			BufferedReader	dataReader = comp.getReader ();
			Set<String>		golden = new HashSet<String> ();
			String			line;
			
			while ((line = dataReader.readLine ()) != null)
			{
				if (line.startsWith ("#"))
					continue;	// Skip header lines
				
				String[]	elements = SplitFile.mySplit (line, "\t", kReturnAll);
				String		uniqueName = elements[kUnique];
				
				if (uniqueName.startsWith ("_"))
					golden.add (uniqueName);
			}
			
			Iterator<String>	iter = golden.iterator ();
			
			while (iter.hasNext ())
			{
				if (!kTestGolden.contains (iter.next ()))
					iter.remove ();
			}
			
			return getGoldenStatus (golden);
		}
		catch (IOException oops)
		{
			oops.printStackTrace ();
		}
		
		return "IO Exception reading file";
	}
	
	
	/**
	 * Print the contents of a {@linkplain Map} to {@code out}
	 * 
	 * @param <S>		Map Key type
	 * @param <T>		Map value type
	 * @param title		Optional text to put in front of the Map info
	 * @param theMap	Map whose contents to dump, must not be null
	 * @param out		Where to print results.  If null will use {@linkplain System}.err
	 */
	static <S, T> void dump (String title, Map<S, T> theMap, PrintStream out)
	{
		dump (title, null, theMap, out);
	}
	
	
	/**
	 * Print the contents of a {@linkplain Map} to {@code out}
	 * 
	 * @param <S>		Map Key type
	 * @param <T>		Map value type
	 * @param title		Optional text to put in front of the Map info
	 * @param theMap	Map whose contents to dump, must not be null
	 * @param out		Where to print results.  If null will use {@linkplain System}.err
	 */
	private static <S, T> void dump (String title, String lineStart, Map<S, T> theMap, PrintStream out)
	{
		if (out == null)
			out = System.err;
		
		if (!isEmpty (title))
			out.println (title);
		
		boolean	first = true;
		
		for (Entry<S, T> pair : theMap.entrySet ())
		{
			S	key = pair.getKey ();
			T	value = pair.getValue ();
			
			if (first)
				first = false;
			else if (lineStart != null)
				System.out.print (lineStart);
			
			if (key instanceof Collection<?>)
				System.out.print (getAsString (null, (Collection<?>) key, ", "));
			else
				System.out.print (key);
			
			System.out.print ("\t");
			
			if (value instanceof Map<?, ?>)
				dump (null, (lineStart == null) ? "\t" : lineStart + "\t", (Map<?, ?>) value, out);
			else if (value instanceof Collection<?>)
				System.out.println (getAsString (null, (Collection<?>) value, ", "));
			else
				System.out.println (value);
		}
	}
	
	
	/**
	 * Turn a {@linkplain Collection} of Strings into a single String, with an optional prepended 
	 * title and the strings separated by spaces if no connector is offered
	 * 
	 * @param <T>		Type of the Collection
	 * @param title		Optional text to put in front of the strings
	 * @param items		Collection of Strings to merge together, in the Collection's ordering
	 * @param connector	String to insert between individual items, if null will use " "
	 * @return	String result of the concatenation, possibly empty, never null
	 */
	private static <T> String getAsString (String title, Collection<T> items, String connector)
	{
		StringBuilder	result = new StringBuilder (100);
		
		if (connector == null)
			connector = " ";
		
		if (title != null)
			result.append (title);
		
		boolean	first = true;
		
		for (T item :items)
		{
			if (first)
				first = false;
			else
				result.append (connector);
			
			result.append (item);
		}
		
		return result.toString ();
	}


	/**
	 * Validate that the non-id golden attributes columns always have data in them
	 * 
	 * @param columnsFile	File with the column info
	 */
	protected static void validateDataFile (File columnsFile, List<String> golden)
	{
		Iterator<String>	iter = golden.iterator ();
		BufferedReader		dataReader = null;
		
		while (iter.hasNext ())
		{
			if (!kTestGolden.contains (iter.next ()))
				iter.remove ();
		}
		if (golden.isEmpty ())
			return;	// nothing more to test, no golden attributes
		
		String	name = columnsFile.getName ();
		
		name = name.substring (0, name.length () - kColLen) + kData;
		File	dataFile = new File (columnsFile.getParentFile (), name);
		
		if (!dataFile.exists ())
			return;
		
		System.err.println (dataFile.getAbsolutePath ());
		try
		{
			Compressor	comp = new Compressor (dataFile, null);
			String		line;
			
			dataReader = comp.getReader ();
			while ((line = dataReader.readLine ()) != null)
			{
				String[]	elements = SplitFile.mySplit (line, "\t", kReturnAll, kIncludeBlank);
				
				if (elements.length < kNumDataCols)
					continue;
				
				String	json = elements[kJSONCol];
				boolean	printedJSON = false;
				
				iter = golden.iterator ();
				while (iter.hasNext ())
				{
					String	info = iter.next ();
					if (json.indexOf (info) < 0)
					{
						if (!printedJSON)
						{
							System.err.println (json);
							printedJSON = true;
						}
						System.err.println (info);
						iter.remove ();
						if (golden.isEmpty ())
							return;	// nothing more to test, no golden attributes
					}
				}
			}
		}
		catch (IOException oops)
		{
			oops.printStackTrace ();
		}
		finally
		{
			closeReader (dataReader);
		}
	}
	
	
	/**
	 * Clean up a "unique name" so that it's more human readable
	 * 
	 * @param name	String to clean, must not be null
	 * @return	Cleaned up string
	 */
	private static String cleanName (String name)
	{
		StringBuilder	replacer = new StringBuilder (name);
		
		for (String replace : kSpaceStrs)
		{
			int	pos;
			
			while ((pos = replacer.indexOf (replace)) >= 0)
			{
				replacer.replace (pos, pos + replace.length (), " ");
			}
		}
		
		return replacer.toString ();
	}
	
	
	/**
	 * Get a {@linkplain FilenameFilter} that returns directories, and catalog associated files
	 * 
	 * @return	The FilenameFilter
	 */
	private static FilenameFilter getCatalogFilter ()
	{
		return new FilenameFilter () {
			
			public boolean accept (File dir, String name)
			{
				if (name.endsWith (kDatasource) || name.endsWith (kColumns))
					return true;
				
				return new File (dir, name).isDirectory ();
			}
		};
	}
	
	
	/**
	 * Get a {@link Comparator<File>} that returns files in alphabetical order, followed 
	 * by directories in alphabetical order
	 * 
	 * @return	The Comparator<File>
	 */
	@SuppressWarnings ("javadoc")
	private static Comparator<File> getCatalogComparator ()
	{
		return new Comparator<File> () {

			public int compare (File first, File second)
			{
				boolean	firstDir = first.isDirectory ();
				boolean	secondDir = second.isDirectory ();
				if (firstDir)
				{
					if (!secondDir)
						return 1;
				}
				else if (secondDir)	// Only get here if firstDir is false
					return -1;
				
				return first.getName ().compareTo (second.getName ());
			}
		};
	}
	
	
	/**
	 * Take a file with at least two tab separated columns, and return a {@linkplain Map} 
	 * from the first column to the second
	 * 
	 * @param mapFile	Path to file holding the data, can be null or empty
	 * @return	Map from first column to the second, possibly empty, esp. if {@code mapFile} is 
	 * null or empty, never null 
	 * @throws IOException	File not found, file can't be read
	 */
	static final Map<String, String> getStringToStringMap (String mapFile) throws IOException
	{
		Map<String, String> results = new HashMap<String, String> ();
		
		if (isEmpty (mapFile))
			return results;
			
		BufferedReader	dataReader = Compressor.readFile (new File (mapFile));
		String			line;
		
		while ((line = dataReader.readLine ()) != null)
		{
			String[]	cols = SplitFile.mySplit (line, "\t", kReturnAll);
			
			if (cols.length < 2)
				continue;
			
			String	name = cols[0];
			String	path = cols[1];
			
			results.put (name, path);
		}
		
		return results;
	}
	
	
	/**
	 * Get properties from a stream and save them to the passed in Properties
	 * 
	 * @param input			Stream to get properties from. Must be open and valid, will close when done
	 * @param theProperties	{@linkplain Properties} in which to load the properties. Must not be null
	 * @return	True if loaded successfully, false if there was a problem
	 */
	static boolean loadProperties (InputStream input, Properties theProperties)
	{
		try
		{
			theProperties.load (input);
			input.close ();
		}
		catch (Exception ex)
		{
			System.out.println (ex.getMessage ());
			try
			{
				if (input != null)
					input.close ();
			}
			catch (IOException ioexp)
			{
				System.out.println (ioexp.getMessage ());
			}
			
			return false;
		}
		
		return true;
	}
	
	
	/**
	 * Utility routine for the common String test
	 * 
	 * @param tested	String to test
	 * @return	True if string is null or empty
	 */
	public static final boolean isEmpty (String tested)
	{
		return (tested == null) || tested.isEmpty ();
	}
	
	
	/**
	 * Close a Reader, reporting any problems that might have happened
	 * 
	 * @param theReader	Reader to close
	 * @return	True if closed it successfully, else false
	 */
	public static final boolean closeReader (Reader theReader)
	{
		if (theReader == null)
		{
			System.err.println ("closeReader called w/ null pointer");
			return false;
		}
		
		try
		{
			theReader.close ();
			return true;
		}
		catch (IOException oops)
		{
			oops.printStackTrace ();
			return false;
		}
	}

	
	/**
	 * Close a Writer, reporting any problems that might have happened
	 * 
	 * @param theWriter	Writer to close
	 * @return	True if closed it successfully, else false
	 */
	public static final boolean closeWriter (Writer theWriter)
	{
		if (theWriter == null)
		{
			System.err.println ("closeWriter called w/ null pointer");
			return false;
		}
		
		try
		{
			theWriter.close ();
			return true;
		}
		catch (IOException oops)
		{
			oops.printStackTrace ();
			return false;
		}
	}

}
