package edu.mayo.bior.pipeline.createcatalog;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import edu.mayo.exec.Command;

public class TabixCmd {

	private static Logger sLogger = LoggerFactory.getLogger(TabixCmd.class);
	
	public static void main(String[] args) {
		try {
			if( args.length > 0  ||  (args.length == 1  &&  args[0].equals("-h")) ) {
				System.out.println("Find the preferred location of the tabix command from (in this order):\n"
							+      "  $PATH\n"
						    +      "  $BIOR_LITE_HOME\n"
							+ 	   "  (relative project path - for tests)\n"
						    +      "  (resource path - for command line use)"
						);
				System.exit(0);
			}
			
			File tabixInPath      = TabixCmd.getTabixFromPath();
			File tabixInBiorHome  = TabixCmd.getTabixFromBiorHome();
			File tabixFromRelPath = TabixCmd.getTabixFromRelativePath();
			File tabixFromRscPath = TabixCmd.getTabixFromResourcePath();
			
			System.out.println("Tabix in $PATH:                "	+ getAbsPath(tabixInPath)     + ".  exists?: " + isExist(tabixInPath));
			System.out.println("Tabix in $BIOR_LITE_HOME/bin:  " 	+ getAbsPath(tabixInBiorHome) + ".  exists?: " + isExist(tabixInBiorHome));
			System.out.println("Tabix in relative path:        " 	+ getAbsPath(tabixFromRelPath)+ ".  exists?: " + isExist(tabixFromRelPath));
			System.out.println("Tabix in resource path:        "    + getAbsPath(tabixFromRscPath)+ ".  exists?: " + isExist(tabixFromRscPath));

			String tabixPrefPath    = TabixCmd.getTabixPreferredPath();
			System.out.println("Tabix preferred path:          "    + tabixPrefPath);
		} catch(Exception e) {
			e.printStackTrace();
		}
	}
	
	public static File createTabixIndex(String bgzipCatalogPath) throws IOException, InterruptedException {
		String tabixCmdPath = getTabixPreferredPath();

		String[] cmdArgs = new String[] {
				tabixCmdPath,
				"-s", "1",
				"-b", "2",
				"-e", "3",
				bgzipCatalogPath
		};
		Map<String,String> sysEnvMap = System.getenv();
		Command cmd = new Command(cmdArgs, sysEnvMap, /**isUseParentEnvironment=*/true);
		cmd.execute();
		String stderr = cmd.getStderr();
		String stdout = cmd.getStdout();
		sLogger.info("Info messages from building tabix index via command:");
		sLogger.info("StdOut: " + stdout);
		sLogger.warn("StdErr: " + stderr);
		File tabixIndexFile = new File(bgzipCatalogPath + ".tbi");
		
		if( ! tabixIndexFile.exists() ) {
			throw new FileNotFoundException("Tabix index could not be created: " + tabixIndexFile.getCanonicalPath());
		}
		
		return tabixIndexFile;
	}
	
	
	/** Choose from the tabix command path in this order:
	 *  1) tabix on the system $PATH
	 *  2) $BIOR_LITE_HOME/bin/tabixXXX
	 *  3) ${PROJECT_DIR}/src/main/resources/tabixXXX  (for testing only!) 
	 * @return  The preferred tabix path
	 * @throws FileNotFoundException 
	 */
	public static String getTabixPreferredPath() throws FileNotFoundException {
		// Try tabix within $PATH directories first
		File tabixOnPath = getTabixFromPath();
		if( isExist(tabixOnPath) )
			return tabixOnPath.getAbsolutePath();
		
		// Tabix wasn't found on $PATH, so try within the $BIOR_LITE_HOME/bin/tabixMac or tabixLinux
		File tabixInBiorBin = getTabixFromBiorHome();
		if( isExist(tabixInBiorBin) )
			return tabixInBiorBin.getAbsolutePath();
		
		// Tabix wasn't found in $BIOR_LITE_HOME/bin so try with relative path
		// Useful in tests only.  When used from the command line, the resource-path should be used instead.
		File tabixInRelativePath = getTabixFromRelativePath();
		if( isExist(tabixInRelativePath) )
			return tabixInRelativePath.getAbsolutePath();
		
		// Tabix wasn't found in relative path, so try with resource path
		File tabixFromResourcePath = getTabixFromResourcePath();
		if( isExist(tabixFromResourcePath) )
			return tabixFromResourcePath.getAbsolutePath();

		// Not found anywhere so throw exception
		throw new FileNotFoundException("Error: could not find tabix command");
	}

	protected static boolean isExist(File file) {
		return file != null  &&  file.exists();
	}
	
	protected static String getAbsPath(File file) {
		if( file == null )
			return "null";
		return file.getAbsolutePath();
	}
	
	public static File getTabixFromBiorHome() {
		return new File(getBiorHomeDirSysVar() + "/bin", getSpecialTabixCmdName());
	}
	
	public static File getTabixFromPath() {
		return getFromPath("tabix");
	}
	
	public static File getFromPath(String cmd) {
		String pathVar = getPathSysVar();
		// Split the path by the separator char
		String[] paths = pathVar.split(File.pathSeparator);
		for(String path : paths) {
			File tabixCmd = new File(path, cmd);
			if( tabixCmd.exists() )
				return tabixCmd;
		}
		// Not found
		return null;
	}
	
	public static File getTabixFromRelativePath() {
		String OS = System.getProperty("os.name").toLowerCase();
		boolean isMac = OS.contains("mac");
		File tabixCmdPath = new File("src/main/scripts/" + (isMac ? "tabixMac" : "tabixLinux"));
		return tabixCmdPath;
	}
	
	/** Get the path relative to this class */
	public static File getTabixFromResourcePath() {
		// NOTE: In production, this points to $BIOR_LITE_HOME/conf
		//       In testing it points to ${PROJECT_HOME}/target/classes
		String pathToClass = TabixCmd.class.getResource("/").getPath();
		System.out.println("pathToClass: " + pathToClass);
		return new File(pathToClass + "../bin",  getSpecialTabixCmdName());
	}
	

	/** Get the $BIOR_LITE_HOME directory from the system environment variables */
	public static String getBiorHomeDirSysVar() {
		return  System.getenv("BIOR_LITE_HOME");
	}
	
	/** Get the $PATH from the system environment variables */
	public static String getPathSysVar() {
		return  System.getenv("PATH");
	}
	
	private static String getSpecialTabixCmdName() {
		String OS = System.getProperty("os.name").toLowerCase();
		boolean isMac = OS.contains("mac");
		return isMac ? "tabixMac" : "tabixLinux";
	}
}
