package edu.mayo.bior.pipeline.createCatalogProps;

import java.util.ArrayList;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;

import edu.mayo.pipes.history.ColumnMetaData;
import edu.mayo.pipes.history.ColumnMetaData.Type;

/** Allow us to grab info from either a VCF-formatted datasource, or try to guess the types from a catalog crawler.
 *  These are converted to ColumnMetaData objects before writing the objects to a columns properties file
 * @author Michael Meiners (m054457)
 * Date created: Aug 16, 2013
 */
public class ColumnTypeCounter {
	/** Approved delimiters that may be used to separate values */
	public static final String[] DELIMITERS = { ",", ";" }; //, ":" };

	
	private ColumnMetaData mColMeta = new ColumnMetaData();

	// Since JsonArrays will be resolved to whatever their primitives are, we want to know if 
	// the JsonArrays were encountered so we can resolve the count to "."
	private int mCountJsonArray = 0;
	
	// Use this for testing only as it will take up a lot of memory in production
	private ArrayList<String> mExamples = new ArrayList<String>();
		
	private JsonParser jsonParser = new JsonParser();

	//=============================================================================
	
	public ColumnTypeCounter(String colName, String description) {
		mColMeta.setColumnName(colName);
		mColMeta.setDescription(description);
	}
	
	public ColumnMetaData getColumnMetaData() {
		return mColMeta;
	}
	
	
	/** This should only be called after ALL values have been added and we want to get the final results */
	public void setTypeAndCountForColumnMetaData() {
		// If the type is still null, then it must have been a JsonArray that was always empty
		if(mColMeta.getType() == null)
			mColMeta.setType(Type.String);
		
		// Establish count
		if( mColMeta.getType().equals(Type.Boolean) ) 
			mColMeta.setCount("0");
		else if( mCountJsonArray > 0 ) 
			mColMeta.setCount(".");
		else
			mColMeta.setCount("1");
	}

	/** Return count.  Ex: "." (multiple), "0" (flag/boolean), "1" (single value) */
	protected String determineCount(JsonElement jsonElem) {
		String count = "1";
		if( jsonElem.isJsonObject() ) {
			count = "1";
		} else if( jsonElem.isJsonArray() ) {
			count = ".";
		}else if( jsonElem.isJsonPrimitive() ) {
			JsonPrimitive prim = (JsonPrimitive)jsonElem;
			if(prim.isJsonArray() ) {
				count = ".";
			} else if( prim.isBoolean() ) {
				count = "0";
			} else {
				count = "1";
			}
		}
		return count;		
	}
	
	/** Return count.  Ex: "." (multiple), "0" (flag/boolean), "1" (single value) */
	protected String determineCount(String value) {
		return determineCount(this.jsonParser.parse(value));
	}
	
	protected Type determineType(JsonElement jsonElem) {
		Type type = null;
		if( jsonElem.isJsonNull() ) {
			type = null;
		} else if( jsonElem.isJsonObject() ) {
			type = Type.JSON;
		} else if( jsonElem.isJsonArray() ) {
			type = getTypeFromJsonArray((JsonArray)jsonElem);
			mCountJsonArray++;
		} else if( jsonElem.isJsonPrimitive() ) {
			type = getTypeFromJsonPrimitive((JsonPrimitive)jsonElem);
		} else {
			type = Type.String;
		}
		return type;		
	}
	
	protected Type determineType(String value) {
		return determineType(this.jsonParser.parse(value));
	}
	
	private Type getTypeFromJsonArray(JsonArray jsonArray) {
		// If empty list, then return null since we can't determine type
		if( jsonArray == null || jsonArray.size() == 0 )
			return null;
		
		// Look thru the JsonArray elements since there might be nulls out front that won't tell us the type.  Ex:  {"A":[null,null,"str"]}
		Type type = null;
		int i=0;
		while(type == null &&  i < jsonArray.size()) {
			JsonElement elem = jsonArray.get(i++);
			if( elem.isJsonNull() ) {
				type = null;
			} else if( elem.isJsonObject() ) {
				type = Type.JSON;
			} else if( elem.isJsonPrimitive() ) {
				type = getTypeFromJsonPrimitive((JsonPrimitive)elem);
			} else {
				type = Type.String;
			}
		}
		return type;
	}
	
	private Type getTypeFromJsonPrimitive(JsonPrimitive prim) {
		Type type = null;
		if( prim.isBoolean() )
			type = Type.Boolean;
		else if( prim.isNumber() ) {
			if( isInt(prim.toString()) )
				type = Type.Integer;
			else if( isFloat(prim.toString()) )
				type = Type.Float;
		} else
			type = Type.String;
		return type;
	}

	private boolean isFloat(String val) {
		try {
			Float.parseFloat(val);
			return true;
		} catch(Exception e) {
			return false;
		}
	}

	private boolean isInt(String val) {
		try {
			Integer.parseInt(val);
			return true;
		} catch(Exception e) {
			return false;
		}
	}
}
