package edu.mayo.bior.cli.cmd;

import org.apache.commons.cli.CommandLine;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Collection;

/**
 * Reusable Mayo Commons CLI code.
 *
 * TODO: push this into mayo-commons-cli project
 */
public class CommandUtil {

    enum FileAttributes {
        EXISTS,
        READABLE,
        WRITEABLE,
        EXECUTEABLE,
    }

    enum DirectoryAttributes {
        EXISTS,
        READABLE,
        WRITEABLE,
        EXECUTEABLE,
    }

    /**
     * Handles a required file option
     * @param cl The Apache CLI framework
     * @param option The file option
     * @param attrs Zero or more {@link FileAttributes} to customize validation checks
     * @return A {@link File} pointing to the file
     * @throws Exception Thrown if file does not exist or cannot be read.
     */
    public static File handleFile(CommandLine cl, String option, Collection<FileAttributes> attrs) throws Exception {
        if (cl.hasOption(option)) {
            File file = new File (cl.getOptionValue(option));

            if (attrs.contains(FileAttributes.EXISTS) && !file.exists())
                throw new FileNotFoundException(String.format("%s does not exist", file.getAbsolutePath()));

            if (!file.isFile())
                throw new IOException(String.format("%s is not a file", file.getAbsolutePath()));

            if (attrs.contains(FileAttributes.READABLE) && !file.canRead())
                throw new Exception(String.format("%s permissions issue (not readable)", file.getAbsolutePath()));

            if (attrs.contains(FileAttributes.WRITEABLE) && !file.canWrite())
                throw new Exception(String.format("%s permissions issue (not writeable)", file.getAbsolutePath()));

            if (attrs.contains(FileAttributes.EXECUTEABLE) && !file.canExecute())
                throw new Exception(String.format("%s permissions issue (not executeable)", file.getAbsolutePath()));

            return file;
        }
        throw new RuntimeException(String.format("Missing option: %s", option));
    }

    /**
     * Handles a required directory option
     * @param cl The Apache CLI framework
     * @param option The directory option
     * @param attrs Zero or more {@link DirectoryAttributes} to customize validation checks
     * @return A {@link File} pointing to the directory
     */
    public static File handleDirectory(CommandLine cl, String option, Collection<DirectoryAttributes> attrs) throws Exception {
        if (cl.hasOption(option)) {
            File dir = new File (cl.getOptionValue(option));

            if (attrs.contains(DirectoryAttributes.EXISTS) && !dir.exists())
                throw new FileNotFoundException(String.format("%s does not exist", dir.getAbsolutePath()));

            if (!dir.isDirectory())
                throw new IOException(String.format("%s is not a directory", dir.getAbsolutePath()));

            if (attrs.contains(DirectoryAttributes.READABLE) && !dir.canRead())
                throw new Exception(String.format("%s permissions issue (not readable)", dir.getAbsolutePath()));

            if (attrs.contains(DirectoryAttributes.WRITEABLE) && !dir.canWrite())
                throw new Exception(String.format("%s permissions issue (not writeable)", dir.getAbsolutePath()));

            if (attrs.contains(DirectoryAttributes.EXECUTEABLE) && !dir.canExecute())
                throw new Exception(String.format("%s permissions issue (not executeable)", dir.getAbsolutePath()));

            return dir;
        }
        throw new RuntimeException(String.format("Missing option: %s", option));
    }

    /**
     * Handles a required directory option
     * @param cl The Apache CLI framework
     * @param option The integer option
     * @param min OPTIONAL minimum value.  Pass NULL if not needed.
     * @param max OPTIONAL maximum value.  Pass NULL if not needed.
     * @return The integer
     */
    public static int handleInteger(CommandLine cl, String option, Integer min, Integer max) throws Exception {
        if (cl.hasOption(option)) {
            String value = cl.getOptionValue(option);
            int integer;
            try {
                integer = Integer.parseInt(value);
            } catch (NumberFormatException nfe) {
                throw new Exception(String.format("Option '%s' must be an integer. %s was supplied.", option, value));
            }

            if ((min != null) && (integer < min)) {
                throw new Exception(String.format("Option '%s' must be %d or higher. %d was supplied.", option, min, integer));
            }

            if ((max != null) && (integer > max)) {
                throw new Exception(String.format("Option '%s' must be %d or lower. %d was supplied.", option, max, integer));
            }

            return integer;
        }
        throw new RuntimeException(String.format("Missing option '%s'", option));
    }

    /**
     * Handles a required directory option
     * @param cl The Apache CLI framework
     * @param option The integer option
     * @param min OPTIONAL minimum value.  Pass NULL if not needed.
     * @param max OPTIONAL maximum value.  Pass NULL if not needed.
     * @return The integer
     */
    public static long handleLong(CommandLine cl, String option, Long min, Long max) throws Exception {
        if (cl.hasOption(option)) {
            String value = cl.getOptionValue(option);
            long longValue;
            try {
                longValue = Long.parseLong(value);
            } catch (NumberFormatException nfe) {
                throw new Exception(String.format("Option '%s' must be an integer. %s was supplied.", option, value));
            }

            if ((min != null) && (longValue < min)) {
                throw new Exception(String.format("Option '%s' must be %d or higher. %d was supplied.", option, min, longValue));
            }

            if ((max != null) && (longValue > max)) {
                throw new Exception(String.format("Option '%s' must be %d or lower. %d was supplied.", option, max, longValue));
            }

            return longValue;
        }
        throw new RuntimeException(String.format("Missing option '%s'", option));
    }
}
